/* $Id: wmhtest.c,v 1.14 2005/09/05 10:45:48 pekberg Exp $
******************************************************************************

   Test program for the "wmh" i.e. WindowManagerHints extension.

   Copyright (C) 1999 Andreas Beck - becka@ggi-project.org
  
   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>

#include "config.h"
#include <ggi/gg.h>
#include <ggi/ggi.h>
#include <ggi/ggi_ext.h>

#include <ggi/wmh.h>

static int small_window = 0;

static int ggi_resize(ggi_visual_t vis, ggi_mode *mode)
{
	ggi_color col;
	int x,y;

	fprintf(stderr,"Want resize to %dx%d\n",mode->visible.x,mode->visible.y);
	mode->virt.x=mode->visible.x;
	mode->virt.y=mode->visible.y;
	ggiCheckMode(vis,mode);
	ggiSetMode(vis,mode);
	col.r = 0xffff;
	col.g = 0xffff;
	col.b = 0xffff;
	ggiSetGCForeground(vis, ggiMapColor(vis, &col));
	if(small_window)
		ggiWmhAllowResize(vis,300,300,600,600,10,10);
	else
		ggiWmhAllowResize(vis,1,1,2000,2000,1,1);

	for(y=0; y < mode->visible.y; y += 10) {
		ggiDrawLine(vis,0,y,mode->visible.x-1,mode->visible.y-1-y);
	}	/* for */
	for(x=0; x < mode->visible.x; x += 10) {
		ggiDrawLine(vis,x,mode->visible.y-1,mode->visible.x-1-x,0);
	}	/* for */

	ggiFlush(vis);
	fprintf(stderr,"Want resize to %dx%d\n",mode->visible.x,mode->visible.y);

	return 0;
}	/* ggi_resize */


static int ggi_waitevent(ggi_visual_t vis)
{

	struct timeval tv={0,0};
	ggi_event ev, reqev;
	int reqswitch = 0;
	int n;

again:
	reqswitch = 0;
	ggiEventPoll(vis, emAll, &tv);
	n = ggiEventsQueued(vis, emAll);
	while( n-- ) {
		ggiEventRead(vis, &ev, emAll);
		fprintf(stderr,"Got in poll: %d\n",ev.any.type);
		switch(ev.any.type) {
		case evCommand:
			if (ev.cmd.code==GGICMD_REQUEST_SWITCH) {
				reqev = ev;
				reqswitch = 1;
			}
			break;
		case evKeyPress:
			return 1;
		}	/* switch */
	}	/* while */

	if (reqswitch) {
		ggi_cmddata_switchrequest *req;
		req = (ggi_cmddata_switchrequest *)&reqev.cmd.data;
		ggi_resize(vis,&(req->mode));
		goto again;
	}

	return 0;
}	/* ggi_waitevent */

int main(void)
{
	ggi_visual_t vis;
	ggi_color col;
	ggi_mode mode;
	int temp, inc, width, height;

	/* Initialize LibGGI */
	if (ggiInit() != 0) {
		fprintf(stderr, "Unable to initialize LibGGI\n");
		exit(1);
	}	/* if */

	/* Initialize WMH extension */
	if (ggiWmhInit() != 0) {
		ggiPanic("Unable to initialize LibGGIWMH extension\n");
	}	/* if */

	/* Open the default visual */
	if ((vis = ggiOpen(NULL)) == NULL) {
		ggiPanic("Unable to open default visual\n");
	}	/* if */

	/* Turn on asynchronous mode (which should always be used) */
	ggiSetFlags(vis, GGIFLAG_ASYNC);
	
	/* Set the default mode */
	if (ggiSetSimpleMode(vis, GGI_AUTO, GGI_AUTO, GGI_AUTO, GT_AUTO) < 0) {
		ggiPanic("Unable to set default mode\n");
	}	/* if */

	/* Attach the WMH extension to the visual */
	if (ggiWmhAttach(vis) < 0) {
		ggiPanic("Unable to attach WMH extension to visual\n");
	}	/* if */

	/* Set foreground color to white */
	col.r = 0xffff;
	col.g = 0xffff;
	col.b = 0xffff;
	ggiSetGCForeground(vis, ggiMapColor(vis, &col));

	/* Check what mode we actually got */
	ggiGetMode(vis, &mode);
	width = mode.visible.x;
	height = mode.visible.y;
	
	/* Draw a bunch of lines... */
	inc = height/66;
	for (temp = 0; temp <= (height-inc); temp += inc) {
		ggiDrawLine(vis, 0, temp, width-1, height-temp);
	}	/* for */

	/* ...and flush the screen */
	ggiFlush(vis);

	/* Now test the window stuff */
	ggiWmhSetTitle(vis,"This is LibGGIWMH resizing !");
	ggiWmhSetIconTitle(vis,"WMH Icon Title");

	ggiWmhAllowResize(vis,1,1,2000,2000,1,1);

	for (temp = 10; temp < height; temp+=(height-10)/30) { 
		ggiWmhResize(vis, temp, temp);
		ggiFlush(vis);
		if (ggi_waitevent(vis)) goto cleanup;
		ggUSleep(100000);
	}	/* for */

	ggiWmhSetTitle(vis,"This is LibGGIWMH moving !");
	for (temp = 0; temp < height/3; temp+=(height/3)/30) { 
		ggiWmhMove(vis, temp, temp);
		ggiFlush(vis);
		if (ggi_waitevent(vis)) goto cleanup;
		ggUSleep(100000);
	}	/* for */
	ggiWmhMove(vis, 40,40);

	ggiWmhSetTitle(vis,"This is LibGGIWMH at the bottom !");
	ggiWmhZOrder(vis,GGI_WMH_ZO_BOTTOM);
	ggiFlush(vis);
	if (ggi_waitevent(vis)) goto cleanup;
	ggUSleep(2000000);

	ggiWmhSetTitle(vis,"This is LibGGIWMH at the top !");
	ggiWmhZOrder(vis,GGI_WMH_ZO_TOP);
	ggiFlush(vis);
	if (ggi_waitevent(vis)) goto cleanup;
	ggUSleep(2000000);

	ggiWmhSetTitle(vis,"This is LibGGIWMH iconified !");
	ggiWmhIconify(vis,GGI_WMH_MAPSTATE_ICON);
	ggiFlush(vis);
	if (ggi_waitevent(vis)) goto cleanup;
	ggUSleep(2000000);

	ggiWmhSetTitle(vis,"This is LibGGIWMH big again !");
	ggiWmhIconify(vis,GGI_WMH_MAPSTATE_NORMAL);
	ggiFlush(vis);
	if (ggi_waitevent(vis)) goto cleanup;
	ggUSleep(2000000);

	ggiWmhSetTitle(vis,"This is LibGGIWMH withdrawn !");
	ggiWmhIconify(vis,GGI_WMH_MAPSTATE_WITHDRAWN);
	ggiFlush(vis);
	if (ggi_waitevent(vis)) goto cleanup;
	ggUSleep(2000000);

	ggiWmhSetTitle(vis,"This is LibGGIWMH big again !");
	ggiWmhIconify(vis,GGI_WMH_MAPSTATE_NORMAL);
	ggiFlush(vis);
	if (ggi_waitevent(vis)) goto cleanup;
	ggUSleep(2000000);

	ggiWmhSetTitle(vis,"This is LibGGIWMH 100x10 % !");
	ggiWmhMaximize(vis,100,10);
	ggiFlush(vis);
	if (ggi_waitevent(vis)) goto cleanup;
	ggUSleep(1000000);

	ggiWmhSetTitle(vis,"This is LibGGIWMH 10x100 % !");
	ggiWmhMaximize(vis,10,100);
	ggiFlush(vis);
	if (ggi_waitevent(vis)) goto cleanup;
	ggUSleep(1000000);

	ggiWmhSetTitle(vis,"This is LibGGIWMH 100x100 % !");
	ggiWmhMaximize(vis,100,100);
	ggiFlush(vis);
	if (ggi_waitevent(vis)) goto cleanup;
	ggUSleep(1000000);


	ggiWmhSetTitle(vis,"This is LibGGIWMH at the top !");
	ggiWmhZOrder(vis,GGI_WMH_ZO_TOP);
	ggiFlush(vis);
	if (ggi_waitevent(vis)) goto cleanup;
	ggUSleep(1000000);

	ggiWmhSetTitle(vis,"You may now manually resize between 300 and 600 pixels !");
	small_window = 1;
	ggiWmhAllowResize(vis,300,300,600,600,10,10);
	ggiWmhResize(vis,300,300);
	while(!ggi_waitevent(vis)) ggUSleep(50000);

  cleanup:
	/* Detach extension from visual */
	ggiWmhDetach(vis);

	/* Close visual */
	ggiClose(vis);

	/* Deinitialize WMH extension */
	ggiWmhExit();

	/* Deinitialize LibGGI */
	ggiExit();

	return 0;
}	/* main */
