#include "exim.h"
/*************************************************
*     Exim - an Internet mail transport agent    *
*************************************************/

/* Cygwin-specific code. December 2002. Updated Jan 2015.
   This is prefixed to the src/os.c file.

   This code was supplied by Pierre A. Humblet <Pierre.Humblet@ieee.org>
*/

/* We need a special mkdir that
   allows names starting with // */
#undef mkdir
int cygwin_mkdir( const char *path, mode_t mode )
{
  const char * p = path;
  if (*p == '/') while(*(p+1) == '/') p++;
  return mkdir(p, mode);
}

#ifndef COMPILE_UTILITY /* Utilities don't need special code */

#ifdef INCLUDE_PAM
#include "../pam/pam.c"
#endif
#include <alloca.h>

unsigned int cygwin_WinVersion;

/* Conflict between Windows definitions and others */
#ifdef NOERROR
#undef NOERROR
#endif
#ifdef DELETE
#undef DELETE
#endif
#define _NO_BOOL_TYPEDEF
#include <windows.h>
#include <ntstatus.h>
#include <lmcons.h>
#define EqualLuid(Luid1, Luid2) \
  ((Luid1.LowPart == Luid2.LowPart) && (Luid1.HighPart == Luid2.HighPart))
#include <sys/cygwin.h>

/* Special static variables */
static BOOL cygwin_debug = FALSE;
static int fakesetugid = 1; /* when not privileged, setugid = noop */

#undef setuid
int cygwin_setuid(uid_t uid )
{
  int res = 0;
  if (fakesetugid == 0) { 
    res = setuid(uid);
    if (cygwin_debug)
      fprintf(stderr, "setuid %u %u %d pid: %d\n",
              uid, getuid(),res, getpid());
  }
  return res;
}

#undef setgid
int cygwin_setgid(gid_t gid )
{
  int res = 0;
  if (fakesetugid == 0) { 
    res = setgid(gid);
    if (cygwin_debug)
      fprintf(stderr, "setgid %u %u %d pid: %d\n",
              gid, getgid(), res, getpid());
  }
  return res;
}

/* Background processes run at lower priority */
static void cygwin_setpriority()
{
  if (!SetPriorityClass(GetCurrentProcess(), BELOW_NORMAL_PRIORITY_CLASS))
    SetPriorityClass(GetCurrentProcess(), IDLE_PRIORITY_CLASS);
  return;
}


/* GetVersion()
   MSB: 1 for 95/98/ME; Next 7: build number, except for 95/98/ME
   Next byte: 0
   Next byte: minor version of OS
   Low  byte: major version of OS (3 or 4 for for NT, 5 for 2000 and XP) */
//#define VERSION_IS_58M(x) (x & 0x80000000) /* 95, 98, Me   */
//#define VERSION_IS_NT(x)  ((x & 0XFF) < 5) /* NT 4 or 3.51 */

/*
  Routine to find if process or thread is privileged
*/

enum {
  CREATE_BIT = 1,
};

static DWORD get_privileges ()
{
  char buffer[1024];
  DWORD i, length;
  HANDLE hToken = NULL;
  PTOKEN_PRIVILEGES privs;
  LUID cluid, rluid;
  DWORD ret = 0;

  privs = (PTOKEN_PRIVILEGES) buffer;

  if (OpenProcessToken (GetCurrentProcess(), TOKEN_QUERY, &hToken)
      && LookupPrivilegeValue (NULL, SE_CREATE_TOKEN_NAME, &cluid)
      && LookupPrivilegeValue(NULL, SE_RESTORE_NAME, &rluid)
      && (GetTokenInformation( hToken, TokenPrivileges,
                               privs, sizeof (buffer), &length)
          || (GetLastError () == ERROR_INSUFFICIENT_BUFFER
              && (privs = (PTOKEN_PRIVILEGES) alloca (length))
              && GetTokenInformation(hToken, TokenPrivileges,
                                     privs, length, &length)))) {
    for (i = 0; i < privs->PrivilegeCount; i++) {
      if (EqualLuid(privs->Privileges[i].Luid, cluid))
        ret |= CREATE_BIT;
      if (ret == (CREATE_BIT))
        break;
    }
  }
  else
    fprintf(stderr, "has_create_token_privilege %u\n", GetLastError());

  if (hToken)
    CloseHandle(hToken);

  return ret;
}

/* 
  We use cygwin_premain to fake a few things 
	and to provide some debug info 
*/
void cygwin_premain2(int argc, char ** argv, struct per_process * ptr)
{
  int i, res, is_daemon = 0, is_spoolwritable, is_privileged, is_eximuser;
  uid_t myuid, systemuid;
  gid_t mygid, adminsgid;
  struct passwd * pwp = NULL;
  struct stat buf;
  char *cygenv;
  SID(1, SystemSid, SECURITY_LOCAL_SYSTEM_RID);
  SID(2, AdminsSid, SECURITY_BUILTIN_DOMAIN_RID, DOMAIN_ALIAS_RID_ADMINS);
  DWORD priv_flags;

  myuid = getuid();
  mygid = getgid();
  cygwin_WinVersion = GetVersion();
  if ((cygenv = getenv("CYGWIN")) == NULL) cygenv = "";
  /* Produce some debugging on stderr,
     cannot yet use exim's debug functions.
     Exim does not use -c and ignores -n.
     Set lower priority for daemons */
  for (i = 1; i < argc; i++) {
    if (argv[i][0] == '-') {
      if (argv[i][1] == 'c') {
        ssize_t size;
        wchar_t *win32_path;
        argv[i][1] = 'n';  /* Replace -c by -n */
        cygwin_debug = TRUE;
        fprintf(stderr, "CYGWIN = \"%s\".\n", cygenv);
        if (((size = cygwin_conv_path(CCP_POSIX_TO_WIN_W,"/", win32_path, 0)) > 0)
 	 && ((win32_path = malloc(size)) != NULL)
         && (cygwin_conv_path(CCP_POSIX_TO_WIN_W,"/", win32_path, size) == 0)) {
	        fprintf(stderr, " Root / mapped to %ls.\n", win32_path);
		free(win32_path);
	}
      }
      else if (argv[i][1] == 'b' && argv[i][2] == 'd') {
        is_daemon = 1;
        cygwin_setpriority();
    }
  }
  }

  /* Nt/2000/XP
     We initially set the exim uid & gid to those of the "exim user",
       or to the root uid (SYSTEM) and exim gid (ADMINS),
     If privileged, we setuid to those.
     We always set the configure uid to the system uid.
     We always set the root uid to the real uid
       to allow exim imposed restrictions (bypassable by recompiling)
       and to avoid exec that cause loss of privilege
     If not privileged and unable to chown,
       we set the exim uid to our uid.
     If unprivileged and /var/spool/exim is writable and not running as listening daemon, 
       we fake all subsequent setuid. */

  /* Get the system and admins uid from their sids */
  if ((systemuid = cygwin_internal(CW_GET_UID_FROM_SID, & SystemSid)) == -1) {
	fprintf(stderr, "Cannot map System sid. Aborting\n");
	exit(1);
  }
  if ((adminsgid = cygwin_internal(CW_GET_GID_FROM_SID, & AdminsSid)) == -1) {
	fprintf(stderr, "Cannot map Admins sid. Aborting\n");
	exit(1);
  }

  priv_flags = get_privileges ();
  is_privileged = !!(priv_flags & CREATE_BIT);

  /* Call getpwnam for account exim after getting the local exim name */
  char exim_username[DNLEN + UNLEN + 2];
  if (cygwin_internal(CW_CYGNAME_FROM_WINNAME, "exim", exim_username, sizeof exim_username) != 0)
     pwp = getpwnam (exim_username);

  /* If cannot setuid to exim or and is not the daemon (which is assumed to be
     able to chown or to be the exim user) set the exim ugid to our ugid to avoid
     chown failures after creating files and to be able to setuid to exim in 
     exim.c ( "privilege not needed" ). */
  if ((is_privileged == 0) && (!is_daemon)) {
    exim_uid = myuid;
    exim_gid = mygid;
  }
  else if (pwp != NULL) {
    exim_uid = pwp->pw_uid;  /* Set it according to passwd */
    exim_gid = pwp->pw_gid;
    is_eximuser = 1;
  }
  else {
    exim_uid = systemuid;
    exim_gid = adminsgid;
    is_eximuser = 0;
  }

  res = stat("/var/spool/exim", &buf);
  /* Check if writable (and can be stat) */
  is_spoolwritable = ((res == 0) && ((buf.st_mode & S_IWOTH) != 0));

  fakesetugid = (is_privileged == 0) && (is_daemon == 0) && (is_spoolwritable == 1);

  if (is_privileged) {             /* Can setuid */
     if (cygwin_setgid(exim_gid) /* Setuid to exim */
         || cygwin_setuid(exim_uid)) {
	   fprintf(stderr, "Unable to setuid/gid to exim. priv_flags: %x\n", priv_flags);
      	   exit(0);          /* Problem... Perhaps not in 544 */
     }
  }

  /* Set the configuration file uid and gid to the system uid and admins gid. */
  config_uid = systemuid;
  config_gid = adminsgid;

  /* Pretend we are root to avoid useless exec
     and avoid exim set limitations.
     We are limited by file access rights */
  root_uid = getuid ();

  if (cygwin_debug) {
    fprintf(stderr, "Starting uid %u, gid %u, priv_flags %x, is_privileged %d, is_daemon %d, is_spoolwritable %d.\n",
            myuid, mygid, priv_flags, is_privileged, is_daemon, is_spoolwritable);
    fprintf(stderr, "root_uid %u, exim_uid %u, exim_gid %u, config_uid %u, config_gid %u, is_eximuser %d.\n",
            root_uid, exim_uid, exim_gid, config_uid, config_gid, is_eximuser);
  }
  return;
}

#endif /* COMPILE_UTILITY */
#include "../src/os.c"
