/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.commons.function;

import static org.apache.juneau.commons.utils.Utils.*;

import java.util.Optional;

import org.apache.juneau.commons.lang.*;

/**
 * Represents a simple tuple of 4 objects.
 *
 * <p>
 * This class is useful when you need a four-value composite key for HashMap lookups that properly implements
 * {@link #equals(Object)} and {@link #hashCode()} based on content rather than identity.
 *
 * <h5 class='section'>Array Support:</h5>
 * <p>
 * Unlike using arrays directly as HashMap keys, this class properly handles arrays by using
 * content-based equality and hashing via {@link HashCode#of(Object...)} which internally uses
 * {@link java.util.Arrays#hashCode(Object[])} for arrays.
 *
 * <h5 class='section'>See Also:</h5><ul>
 * 	<li class='jc'>{@link Tuple1}
 * 	<li class='jc'>{@link Tuple2}
 * 	<li class='jc'>{@link Tuple3}
 * 	<li class='jc'>{@link Tuple5}
 * </ul>
 *
 * @param <A> Object 1 type.
 * @param <B> Object 2 type.
 * @param <C> Object 3 type.
 * @param <D> Object 4 type.
 */
public class Tuple4<A,B,C,D> {

	/**
	 * Static creator.
	 *
	 * @param <A> Object 1 type.
	 * @param <B> Object 2 type.
	 * @param <C> Object 3 type.
	 * @param <D> Object 4 type.
	 * @param a Object 1.
	 * @param b Object 2.
	 * @param c Object 3.
	 * @param d Object 4.
	 * @return A new tuple object.
	 */
	public static <A,B,C,D> Tuple4<A,B,C,D> of(A a, B b, C c, D d) {
		return new Tuple4<>(a, b, c, d);
	}

	private final A a;
	private final B b;
	private final C c;
	private final D d;
	private final int hashCode;

	/**
	 * Constructor.
	 *
	 * @param a Object 1.
	 * @param b Object 2.
	 * @param c Object 3.
	 * @param d Object 4.
	 */
	public Tuple4(A a, B b, C c, D d) {
		this.a = a;
		this.b = b;
		this.c = c;
		this.d = d;
		this.hashCode = h(a, b, c, d);
	}

	@Override /* Overridden from Object */
	public boolean equals(Object o) {
		return o instanceof Tuple4<?,?,?,?> o2 && eq(this, o2, (x, y) -> eq(x.a, y.a) && eq(x.b, y.b) && eq(x.c, y.c) && eq(x.d, y.d));
	}

	/**
	 * Returns the first object in this tuple.
	 *
	 * @return The first object in this tuple.
	 */
	public A getA() { return a; }

	/**
	 * Returns the second object in this tuple.
	 *
	 * @return The second object in this tuple.
	 */
	public B getB() { return b; }

	/**
	 * Returns the third object in this tuple.
	 *
	 * @return The third object in this tuple.
	 */
	public C getC() { return c; }

	/**
	 * Returns the fourth object in this tuple.
	 *
	 * @return The fourth object in this tuple.
	 */
	public D getD() { return d; }

	/**
	 * Returns the first object in this tuple wrapped in an {@link Optional}.
	 *
	 * @return The first object wrapped in an Optional, or Optional.empty() if the value is null.
	 */
	public Optional<A> optA() {
		return Optional.ofNullable(a);
	}

	/**
	 * Returns the second object in this tuple wrapped in an {@link Optional}.
	 *
	 * @return The second object wrapped in an Optional, or Optional.empty() if the value is null.
	 */
	public Optional<B> optB() {
		return Optional.ofNullable(b);
	}

	/**
	 * Returns the third object in this tuple wrapped in an {@link Optional}.
	 *
	 * @return The third object wrapped in an Optional, or Optional.empty() if the value is null.
	 */
	public Optional<C> optC() {
		return Optional.ofNullable(c);
	}

	/**
	 * Returns the fourth object in this tuple wrapped in an {@link Optional}.
	 *
	 * @return The fourth object wrapped in an Optional, or Optional.empty() if the value is null.
	 */
	public Optional<D> optD() {
		return Optional.ofNullable(d);
	}

	@Override /* Overridden from Object */
	public int hashCode() {
		return hashCode;
	}
}