/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2016 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "cdw_string.h"
#include "gettext.h"
#include "cdw_debug.h"
#include "cdw_utils.h"
#include "cdw_udf.h"
#include "cdw_mkudffs_options.h"





extern const char *cdw_mkudffs_option_blocksize[];
extern const char *cdw_mkudffs_option_udfrev[];
extern const char *cdw_mkudffs_option_strategy[];
extern const char *cdw_mkudffs_option_spartable[];
extern const char *cdw_mkudffs_option_media_type[];
extern const char *cdw_mkudffs_option_space[];
extern const char *cdw_mkudffs_option_ad[];
extern const char *cdw_mkudffs_option_encoding[];




void cdw_udf_debug_print_options(cdw_udf_t *udf)
{
	cdw_vdm ("INFO: UDF options:\n");

	cdw_vdm ("INFO: UDF option: volume size        = %"PRIu64" bytes\n", udf->volume_size_bytes);

	cdw_vdm ("INFO: UDF option: mkudffs blocksize  = %s\n", cdw_mkudffs_option_blocksize[udf->mkudffs_blocksize]);
	cdw_vdm ("INFO: UDF option: mkudffs udfrev     = %s\n", cdw_mkudffs_option_udfrev[udf->mkudffs_udfrev]);
	cdw_vdm ("INFO: UDF option: mkudffs strategy   = %s\n", cdw_mkudffs_option_strategy[udf->mkudffs_strategy]);
	cdw_vdm ("INFO: UDF option: mkudffs spartable  = %s\n", cdw_mkudffs_option_spartable[udf->mkudffs_spartable]);
	cdw_vdm ("INFO: UDF option: mkudffs media type = %s\n", cdw_mkudffs_option_media_type[udf->mkudffs_media_type]);
	cdw_vdm ("INFO: UDF option: mkudffs space      = %s\n", cdw_mkudffs_option_space[udf->mkudffs_space]);
	cdw_vdm ("INFO: UDF option: mkudffs ad         = %s\n", cdw_mkudffs_option_ad[udf->mkudffs_ad]);
	cdw_vdm ("INFO: UDF option: mkudffs noefe      = %s\n", udf->mkudffs_noefe ? "true" : "false");
	cdw_vdm ("INFO: UDF option: mkudffs encoding   = %s\n", cdw_mkudffs_option_encoding[udf->mkudffs_encoding]);

	cdw_vdm ("INFO: UDF option: mkudffs other options = %s\n", udf->mkudffs_other_options);

	cdw_vdm ("INFO: UDF option: mkudffs lvid  = %s\n", udf->mkudffs_lvid);
	cdw_vdm ("INFO: UDF option: mkudffs vid   = %s\n", udf->mkudffs_vid);
	cdw_vdm ("INFO: UDF option: mkudffs vsid  = %s\n", udf->mkudffs_vsid);
	cdw_vdm ("INFO: UDF option: mkudffs fsid  = %s\n", udf->mkudffs_fsid);

	cdw_vdm ("INFO: UDF option: rsync options = %s\n", udf->rsync_options);
	cdw_vdm ("INFO: UDF option: mount point   = %s\n", udf->mount_point);

	return;
}





cdw_rv_t cdw_udf_copy(cdw_udf_t *target, cdw_udf_t *source)
{
	target->volume_size_bytes  = source->volume_size_bytes;

	target->mkudffs_blocksize  = source->mkudffs_blocksize;
	target->mkudffs_udfrev     = source->mkudffs_udfrev;
	target->mkudffs_strategy   = source->mkudffs_strategy;
	target->mkudffs_spartable  = source->mkudffs_spartable;
	target->mkudffs_media_type = source->mkudffs_media_type;
	target->mkudffs_space      = source->mkudffs_space;
	target->mkudffs_ad         = source->mkudffs_ad;
	target->mkudffs_noefe      = source->mkudffs_noefe;
	target->mkudffs_encoding   = source->mkudffs_encoding;

	strncpy(target->mkudffs_other_options, source->mkudffs_other_options, CDW_MKUDFFS_OTHER_OPTIONS_LEN_MAX);
	        target->mkudffs_other_options[CDW_MKUDFFS_OTHER_OPTIONS_LEN_MAX] = '\0';

	strncpy(target->mkudffs_lvid, source->mkudffs_lvid, CDW_UDF_LVID_LEN_MAX);
	        target->mkudffs_lvid[CDW_UDF_LVID_LEN_MAX] = '\0';
	strncpy(target->mkudffs_vid,  source->mkudffs_vid,  CDW_UDF_VID_LEN_MAX);
	        target->mkudffs_vid[CDW_UDF_VID_LEN_MAX] = '\0';
	strncpy(target->mkudffs_vsid, source->mkudffs_vsid, CDW_UDF_VSID_LEN_MAX);
	        target->mkudffs_vsid[CDW_UDF_VSID_LEN_MAX] = '\0';
	strncpy(target->mkudffs_fsid, source->mkudffs_fsid, CDW_UDF_FSID_LEN_MAX);
	        target->mkudffs_fsid[CDW_UDF_FSID_LEN_MAX] = '\0';

	strncpy(target->rsync_options, source->rsync_options, CDW_RSYNC_OPTIONS_LEN_MAX);
	        target->rsync_options[CDW_RSYNC_OPTIONS_LEN_MAX] = '\0';

	cdw_string_set(&(target->mount_point), source->mount_point);

	return CDW_OK;
}





/*
  Initialize pointers from \p udf with NULL, so that \p udf can be
  later safely passed to cdw_udf_set_defaults().
*/
cdw_rv_t cdw_udf_init(cdw_udf_t *udf)
{
	udf->mount_point = (char *) NULL;

	return CDW_OK;
}





/*
  Set default values for all fields in \p udf.
  You have to initialize \p udf with cw_udf_init() before calling this function.
*/
cdw_rv_t cdw_udf_set_defaults(cdw_udf_t *udf)
{
	udf->volume_size_bytes  = 0;

	udf->mkudffs_blocksize  = CDW_MKUDFFS_BLOCKSIZE_UNSPECIFIED;
	udf->mkudffs_udfrev     = CDW_MKUDFFS_UDFREV_UNSPECIFIED;
	udf->mkudffs_strategy   = CDW_MKUDFFS_STRATEGY_UNSPECIFIED;
	udf->mkudffs_spartable  = CDW_MKUDFFS_SPARTABLE_UNSPECIFIED;
	udf->mkudffs_media_type = CDW_MKUDFFS_MEDIA_TYPE_UNSPECIFIED;
	udf->mkudffs_space      = CDW_MKUDFFS_SPACE_UNSPECIFIED;
	udf->mkudffs_ad         = CDW_MKUDFFS_AD_UNSPECIFIED;
	udf->mkudffs_noefe      = false;  /* The default situation is that the command line flag is not passed to mkudffs. */
	udf->mkudffs_encoding   = CDW_MKUDFFS_ENCODING_UNSPECIFIED;

	udf->mkudffs_other_options[0] = '\0';

	udf->mkudffs_lvid[0] = '\0';
	udf->mkudffs_vid[0]  = '\0';
	udf->mkudffs_vsid[0] = '\0';
	udf->mkudffs_fsid[0] = '\0';

	udf->rsync_options[0] = '\0';

	/* This is a path that sooner or later will be passed to file
	   picker. The file picker won't accept NULL pointer as
	   initial argument. If the field won't be set to non-NULL
	   value by any other code, the picker would flop. To prevent
	   this, call strdup() here.

	   The string is empty because I can't make any assumptions
	   about where user would like to mount the UDF image. Once
	   the user sets value of mount point, it will be stored in
	   config file. After this first-time configuration by user,
	   the value will be re-used. But initial value must be empty
	   string. */
	udf->mount_point = strdup("");

	return CDW_OK;
}





/*
  Deallocate all pointers from \p udf, but not the \p udf itself.
*/
cdw_rv_t cdw_udf_clean(cdw_udf_t *udf)
{
	if (udf->mount_point) {
		free(udf->mount_point);
		udf->mount_point = (char *) NULL;
	}

	return CDW_OK;
}
