use Test::More;
use strict;
use warnings;
our ( $sub, $globref );

BEGIN {
	use_ok('Hades::Realm::OO');
	$sub     = sub { };
	$globref = \*globref;
}
subtest 'new' => sub {
	plan tests => 20;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	ok( $obj = Hades::Realm::OO->new(), q{$obj = Hades::Realm::OO->new()} );
	isa_ok( $obj, 'Hades::Realm::OO' );
	ok( $obj = Hades::Realm::OO->new(
			{   is_role => 1,
				meta    => {
					'algea' => {
						types      => { 'test' => 'test' },
						attributes => { 'test' => 'test' }
					}
				},
				current_class => 'nosoi'
			}
		),
		q{$obj = Hades::Realm::OO->new({ is_role => 1, meta => { 'algea' => { types => { 'test' => 'test' }, attributes => { 'test' => 'test' } } }, current_class => 'nosoi' })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   is_role => [],
				meta    => {
					'algea' => {
						types      => { 'test' => 'test' },
						attributes => { 'test' => 'test' }
					}
				},
				current_class => 'nosoi'
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ is_role => [], meta => { 'algea' => { types => { 'test' => 'test' }, attributes => { 'test' => 'test' } } }, current_class => 'nosoi' })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   is_role => {},
				meta    => {
					'algea' => {
						types      => { 'test' => 'test' },
						attributes => { 'test' => 'test' }
					}
				},
				current_class => 'nosoi'
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ is_role => {}, meta => { 'algea' => { types => { 'test' => 'test' }, attributes => { 'test' => 'test' } } }, current_class => 'nosoi' })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   is_role => 1,
				meta    => {
					'algea' =>
					    { types => [], attributes => { 'test' => 'test' } }
				},
				current_class => 'nosoi'
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ is_role => 1, meta => { 'algea' => { types => [], attributes => { 'test' => 'test' } } }, current_class => 'nosoi' })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   is_role => 1,
				meta    => {
					'algea' => {
						types      => 'algea',
						attributes => { 'test' => 'test' }
					}
				},
				current_class => 'nosoi'
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ is_role => 1, meta => { 'algea' => { types => 'algea', attributes => { 'test' => 'test' } } }, current_class => 'nosoi' })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   is_role => 1,
				meta    => {
					'algea' => {
						types      => undef,
						attributes => { 'test' => 'test' }
					}
				},
				current_class => 'nosoi'
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ is_role => 1, meta => { 'algea' => { types => undef, attributes => { 'test' => 'test' } } }, current_class => 'nosoi' })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   is_role => 1,
				meta    => {
					'algea' =>
					    { types => { 'test' => 'test' }, attributes => [] }
				},
				current_class => 'nosoi'
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ is_role => 1, meta => { 'algea' => { types => { 'test' => 'test' }, attributes => [] } }, current_class => 'nosoi' })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   is_role => 1,
				meta    => {
					'algea' => {
						types      => { 'test' => 'test' },
						attributes => 'algea'
					}
				},
				current_class => 'nosoi'
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ is_role => 1, meta => { 'algea' => { types => { 'test' => 'test' }, attributes => 'algea' } }, current_class => 'nosoi' })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   is_role => 1,
				meta    => {
					'algea' => {
						types      => { 'test' => 'test' },
						attributes => undef
					}
				},
				current_class => 'nosoi'
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ is_role => 1, meta => { 'algea' => { types => { 'test' => 'test' }, attributes => undef } }, current_class => 'nosoi' })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   is_role       => 1,
				meta          => { 'algea' => {} },
				current_class => 'nosoi'
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ is_role => 1, meta => { 'algea' => {} }, current_class => 'nosoi' })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   is_role       => 1,
				meta          => { 'algea' => [] },
				current_class => 'nosoi'
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ is_role => 1, meta => { 'algea' => [] }, current_class => 'nosoi' })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   is_role       => 1,
				meta          => { 'algea' => 'penthos' },
				current_class => 'nosoi'
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ is_role => 1, meta => { 'algea' => 'penthos' }, current_class => 'nosoi' })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   is_role       => 1,
				meta          => { 'algea' => undef },
				current_class => 'nosoi'
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ is_role => 1, meta => { 'algea' => undef }, current_class => 'nosoi' })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{ is_role => 1, meta => [], current_class => 'nosoi' } );
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ is_role => 1, meta => [], current_class => 'nosoi' })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{ is_role => 1, meta => 'thanatos', current_class => 'nosoi' } );
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ is_role => 1, meta => 'thanatos', current_class => 'nosoi' })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   is_role => 1,
				meta    => {
					'algea' => {
						types      => { 'test' => 'test' },
						attributes => { 'test' => 'test' }
					}
				},
				current_class => []
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ is_role => 1, meta => { 'algea' => { types => { 'test' => 'test' }, attributes => { 'test' => 'test' } } }, current_class => [] })}
	);
	eval {
		$obj = Hades::Realm::OO->new(
			{   is_role => 1,
				meta    => {
					'algea' => {
						types      => { 'test' => 'test' },
						attributes => { 'test' => 'test' }
					}
				},
				current_class => \1
			}
		);
	};
	like( $@, qr/invalid|type|constraint|greater|atleast/,
		q{$obj = Hades::Realm::OO->new({ is_role => 1, meta => { 'algea' => { types => { 'test' => 'test' }, attributes => { 'test' => 'test' } } }, current_class => \1 })}
	);
};
subtest 'current_class' => sub {
	plan tests => 7;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'current_class' );
	is( $obj->current_class, undef, q{$obj->current_class} );
	is_deeply( $obj->current_class('limos'),
		'limos', q{$obj->current_class('limos')} );
	eval { $obj->current_class( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->current_class([])} );
	eval { $obj->current_class( \1 ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->current_class(\1)} );
	is_deeply( $obj->current_class, 'limos', q{$obj->current_class} );
};
subtest 'meta' => sub {
	plan tests => 17;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'meta' );
	is( $obj->meta, undef, q{$obj->meta} );
	is_deeply(
		$obj->meta(
			{   'limos' => {
					types      => { 'test' => 'test' },
					attributes => { 'test' => 'test' }
				}
			}
		),
		{   'limos' => {
				types      => { 'test' => 'test' },
				attributes => { 'test' => 'test' }
			}
		},
		q{$obj->meta({ 'limos' => { types => { 'test' => 'test' }, attributes => { 'test' => 'test' } } })}
	);
	eval {
		$obj->meta(
			{   'limos' => { types => [], attributes => { 'test' => 'test' } }
			}
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'limos' => { types => [], attributes => { 'test' => 'test' } } })}
	);
	eval {
		$obj->meta(
			{   'limos' =>
				    { types => 'hypnos', attributes => { 'test' => 'test' } }
			}
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'limos' => { types => 'hypnos', attributes => { 'test' => 'test' } } })}
	);
	eval {
		$obj->meta(
			{   'limos' =>
				    { types => undef, attributes => { 'test' => 'test' } }
			}
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'limos' => { types => undef, attributes => { 'test' => 'test' } } })}
	);
	eval {
		$obj->meta(
			{   'limos' => { types => { 'test' => 'test' }, attributes => [] }
			}
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'limos' => { types => { 'test' => 'test' }, attributes => [] } })}
	);
	eval {
		$obj->meta(
			{   'limos' =>
				    { types => { 'test' => 'test' }, attributes => 'nosoi' }
			}
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'limos' => { types => { 'test' => 'test' }, attributes => 'nosoi' } })}
	);
	eval {
		$obj->meta(
			{   'limos' =>
				    { types => { 'test' => 'test' }, attributes => undef }
			}
		);
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'limos' => { types => { 'test' => 'test' }, attributes => undef } })}
	);
	eval { $obj->meta( { 'limos' => {} } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'limos' => {} })}
	);
	eval { $obj->meta( { 'limos' => [] } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'limos' => [] })}
	);
	eval { $obj->meta( { 'limos' => 'limos' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'limos' => 'limos' })}
	);
	eval { $obj->meta( { 'limos' => undef } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta({ 'limos' => undef })}
	);
	eval { $obj->meta( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta([])} );
	eval { $obj->meta('curae') };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->meta('curae')} );
	is_deeply(
		$obj->meta,
		{   'limos' => {
				types      => { 'test' => 'test' },
				attributes => { 'test' => 'test' }
			}
		},
		q{$obj->meta}
	);
};
subtest 'is_role' => sub {
	plan tests => 7;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'is_role' );
	is( $obj->is_role, undef, q{$obj->is_role} );
	is_deeply( $obj->is_role(1), 1, q{$obj->is_role(1)} );
	eval { $obj->is_role( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->is_role([])} );
	eval { $obj->is_role( {} ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->is_role({})} );
	is_deeply( $obj->is_role, 1, q{$obj->is_role} );
};
subtest 'clear_is_role' => sub {
	plan tests => 5;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'clear_is_role' );
	is_deeply( $obj->is_role(1), 1, q{$obj->is_role(1)} );
	ok( $obj->clear_is_role, q{$obj->clear_is_role} );
	is( $obj->is_role, undef, q{$obj->is_role} );
};
subtest 'module_generate' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'module_generate' );
	eval { $obj->module_generate( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->module_generate([])} );
	eval { $obj->module_generate('hypnos') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->module_generate('hypnos')}
	);
};
subtest 'build_class_inheritance' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_class_inheritance' );
};
subtest 'build_new' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_new' );
	eval { $obj->build_new( [], { 'test' => 'test' }, { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new([], { 'test' => 'test' }, { 'test' => 'test' })}
	);
	eval {
		$obj->build_new( 'nosoi', { 'test' => 'test' }, { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new('nosoi', { 'test' => 'test' }, { 'test' => 'test' })}
	);
	eval { $obj->build_new( bless( {}, 'Test' ), [], { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new(bless({}, 'Test'), [], { 'test' => 'test' })}
	);
	eval {
		$obj->build_new( bless( {}, 'Test' ), 'geras', { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new(bless({}, 'Test'), 'geras', { 'test' => 'test' })}
	);
	eval { $obj->build_new( bless( {}, 'Test' ), { 'test' => 'test' }, [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new(bless({}, 'Test'), { 'test' => 'test' }, [])}
	);
	eval {
		$obj->build_new( bless( {}, 'Test' ), { 'test' => 'test' }, 'phobos' );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_new(bless({}, 'Test'), { 'test' => 'test' }, 'phobos')}
	);
};
subtest 'build_clearer' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_clearer' );
};
subtest 'build_predicate' => sub {
	plan tests => 2;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_predicate' );
};
subtest 'build_accessor_no_arguments' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_accessor_no_arguments' );
	eval {
		$obj->build_accessor_no_arguments( [], ['test'],
			{ 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_no_arguments([], ['test'], { 'test' => 'test' })}
	);
	eval {
		$obj->build_accessor_no_arguments( 'nosoi', ['test'],
			{ 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_no_arguments('nosoi', ['test'], { 'test' => 'test' })}
	);
	eval {
		$obj->build_accessor_no_arguments( bless( {}, 'Test' ),
			{}, { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_no_arguments(bless({}, 'Test'), {}, { 'test' => 'test' })}
	);
	eval {
		$obj->build_accessor_no_arguments( bless( {}, 'Test' ),
			'algea', { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_no_arguments(bless({}, 'Test'), 'algea', { 'test' => 'test' })}
	);
	eval {
		$obj->build_accessor_no_arguments( bless( {}, 'Test' ), ['test'], [] );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_no_arguments(bless({}, 'Test'), ['test'], [])}
	);
	eval {
		$obj->build_accessor_no_arguments( bless( {}, 'Test' ),
			['test'], 'gaudia' );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_no_arguments(bless({}, 'Test'), ['test'], 'gaudia')}
	);
};
subtest 'build_accessor' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_accessor' );
	eval { $obj->build_accessor( [], 'phobos', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor([], 'phobos', { 'test' => 'test' })}
	);
	eval { $obj->build_accessor( 'algea', 'phobos', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor('algea', 'phobos', { 'test' => 'test' })}
	);
	eval {
		$obj->build_accessor( bless( {}, 'Test' ), [], { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor(bless({}, 'Test'), [], { 'test' => 'test' })}
	);
	eval {
		$obj->build_accessor( bless( {}, 'Test' ), \1, { 'test' => 'test' } );
	};
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor(bless({}, 'Test'), \1, { 'test' => 'test' })}
	);
	eval { $obj->build_accessor( bless( {}, 'Test' ), 'phobos', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor(bless({}, 'Test'), 'phobos', [])}
	);
	eval { $obj->build_accessor( bless( {}, 'Test' ), 'phobos', 'curae' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor(bless({}, 'Test'), 'phobos', 'curae')}
	);
};
subtest 'build_modify' => sub {
	plan tests => 8;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_modify' );
	eval { $obj->build_modify( [], 'hypnos', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_modify([], 'hypnos', { 'test' => 'test' })}
	);
	eval { $obj->build_modify( 'penthos', 'hypnos', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_modify('penthos', 'hypnos', { 'test' => 'test' })}
	);
	eval {
		$obj->build_modify( bless( {}, 'Test' ), [], { 'test' => 'test' } );
	};
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_modify(bless({}, 'Test'), [], { 'test' => 'test' })}
	);
	eval {
		$obj->build_modify( bless( {}, 'Test' ), \1, { 'test' => 'test' } );
	};
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_modify(bless({}, 'Test'), \1, { 'test' => 'test' })}
	);
	eval { $obj->build_modify( bless( {}, 'Test' ), 'hypnos', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_modify(bless({}, 'Test'), 'hypnos', [])}
	);
	eval { $obj->build_modify( bless( {}, 'Test' ), 'hypnos', 'limos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_modify(bless({}, 'Test'), 'hypnos', 'limos')}
	);
};
subtest 'after_class' => sub {
	plan tests => 6;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'after_class' );
	eval { $obj->after_class( [], { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->after_class([], { 'test' => 'test' })}
	);
	eval { $obj->after_class( 'curae', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->after_class('curae', { 'test' => 'test' })}
	);
	eval { $obj->after_class( bless( {}, 'Test' ), [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->after_class(bless({}, 'Test'), [])}
	);
	eval { $obj->after_class( bless( {}, 'Test' ), 'limos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->after_class(bless({}, 'Test'), 'limos')}
	);
};
subtest 'unique_types' => sub {
	plan tests => 5;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'unique_types' );
	eval { $obj->unique_types( \1, { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->unique_types(\1, { 'test' => 'test' })}
	);
	eval { $obj->unique_types( 'penthos', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->unique_types('penthos', [])}
	);
	eval { $obj->unique_types( 'penthos', 'nosoi' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->unique_types('penthos', 'nosoi')}
	);
};
subtest 'build_as_class' => sub {
	plan tests => 6;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_as_class' );
	eval { $obj->build_as_class( [], { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_as_class([], { 'test' => 'test' })}
	);
	eval { $obj->build_as_class( 'thanatos', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_as_class('thanatos', { 'test' => 'test' })}
	);
	eval { $obj->build_as_class( bless( {}, 'Test' ), [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_as_class(bless({}, 'Test'), [])}
	);
	eval { $obj->build_as_class( bless( {}, 'Test' ), 'nosoi' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_as_class(bless({}, 'Test'), 'nosoi')}
	);
};
subtest 'build_as_role' => sub {
	plan tests => 6;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_as_role' );
	eval { $obj->build_as_role( [], { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_as_role([], { 'test' => 'test' })}
	);
	eval { $obj->build_as_role( 'thanatos', { 'test' => 'test' } ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_as_role('thanatos', { 'test' => 'test' })}
	);
	eval { $obj->build_as_role( bless( {}, 'Test' ), [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_as_role(bless({}, 'Test'), [])}
	);
	eval { $obj->build_as_role( bless( {}, 'Test' ), 'algea' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_as_role(bless({}, 'Test'), 'algea')}
	);
};
subtest 'build_has_keywords' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_has_keywords' );
	eval { $obj->build_has_keywords( {} ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_has_keywords({})}
	);
	eval { $obj->build_has_keywords('geras') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_has_keywords('geras')}
	);
};
subtest 'build_has' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_has' );
	eval { $obj->build_has( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_has([])} );
	eval { $obj->build_has('limos') };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_has('limos')} );
};
subtest 'build_extends_keywords' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_extends_keywords' );
	eval { $obj->build_extends_keywords( {} ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_extends_keywords({})}
	);
	eval { $obj->build_extends_keywords('gaudia') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_extends_keywords('gaudia')}
	);
};
subtest 'build_extends' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_extends' );
	eval { $obj->build_extends( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_extends([])} );
	eval { $obj->build_extends('phobos') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_extends('phobos')}
	);
};
subtest 'build_with_keywords' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_with_keywords' );
	eval { $obj->build_with_keywords( {} ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_with_keywords({})}
	);
	eval { $obj->build_with_keywords('hypnos') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_with_keywords('hypnos')}
	);
};
subtest 'build_with' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_with' );
	eval { $obj->build_with( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_with([])} );
	eval { $obj->build_with('aporia') };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_with('aporia')} );
};
subtest 'build_requires_keywords' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_requires_keywords' );
	eval { $obj->build_requires_keywords( {} ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_requires_keywords({})}
	);
	eval { $obj->build_requires_keywords('thanatos') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_requires_keywords('thanatos')}
	);
};
subtest 'build_requires' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_requires' );
	eval { $obj->build_requires( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_requires([])} );
	eval { $obj->build_requires('aporia') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_requires('aporia')}
	);
};
subtest 'build_before_keywords' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_before_keywords' );
	eval { $obj->build_before_keywords( {} ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_before_keywords({})}
	);
	eval { $obj->build_before_keywords('limos') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_before_keywords('limos')}
	);
};
subtest 'build_before' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_before' );
	eval { $obj->build_before( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_before([])} );
	eval { $obj->build_before('thanatos') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_before('thanatos')}
	);
};
subtest 'build_around_keywords' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_around_keywords' );
	eval { $obj->build_around_keywords( {} ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_around_keywords({})}
	);
	eval { $obj->build_around_keywords('gaudia') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_around_keywords('gaudia')}
	);
};
subtest 'build_around' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_around' );
	eval { $obj->build_around( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_around([])} );
	eval { $obj->build_around('limos') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_around('limos')}
	);
};
subtest 'build_after_keywords' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_after_keywords' );
	eval { $obj->build_after_keywords( {} ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_after_keywords({})}
	);
	eval { $obj->build_after_keywords('geras') };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_after_keywords('geras')}
	);
};
subtest 'build_after' => sub {
	plan tests => 4;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_after' );
	eval { $obj->build_after( [] ) };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_after([])} );
	eval { $obj->build_after('nosoi') };
	like( $@, qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_after('nosoi')} );
};
subtest 'build_accessor_builder' => sub {
	plan tests => 6;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_accessor_builder' );
	eval { $obj->build_accessor_builder( [], 'gaudia' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_builder([], 'gaudia')}
	);
	eval { $obj->build_accessor_builder( \1, 'gaudia' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_builder(\1, 'gaudia')}
	);
	eval { $obj->build_accessor_builder( 'limos', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_builder('limos', [])}
	);
	eval { $obj->build_accessor_builder( 'limos', \1 ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_builder('limos', \1)}
	);
};
subtest 'build_accessor_coerce' => sub {
	plan tests => 6;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_accessor_coerce' );
	eval { $obj->build_accessor_coerce( [], 'geras' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_coerce([], 'geras')}
	);
	eval { $obj->build_accessor_coerce( \1, 'geras' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_coerce(\1, 'geras')}
	);
	eval { $obj->build_accessor_coerce( 'hypnos', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_coerce('hypnos', [])}
	);
	eval { $obj->build_accessor_coerce( 'hypnos', \1 ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_coerce('hypnos', \1)}
	);
};
subtest 'build_accessor_trigger' => sub {
	plan tests => 6;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_accessor_trigger' );
	eval { $obj->build_accessor_trigger( [], 'aporia' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_trigger([], 'aporia')}
	);
	eval { $obj->build_accessor_trigger( \1, 'aporia' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_trigger(\1, 'aporia')}
	);
	eval { $obj->build_accessor_trigger( 'limos', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_trigger('limos', [])}
	);
	eval { $obj->build_accessor_trigger( 'limos', \1 ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_trigger('limos', \1)}
	);
};
subtest 'build_accessor_default' => sub {
	plan tests => 6;
	ok( my $obj = Hades::Realm::OO->new( {} ),
		q{my $obj = Hades::Realm::OO->new({})}
	);
	can_ok( $obj, 'build_accessor_default' );
	eval { $obj->build_accessor_default( [], 'hypnos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_default([], 'hypnos')}
	);
	eval { $obj->build_accessor_default( \1, 'hypnos' ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_default(\1, 'hypnos')}
	);
	eval { $obj->build_accessor_default( 'thanatos', [] ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_default('thanatos', [])}
	);
	eval { $obj->build_accessor_default( 'thanatos', \1 ) };
	like(
		$@,
		qr/invalid|value|type|constraint|greater|atleast/,
		q{$obj->build_accessor_default('thanatos', \1)}
	);
};
done_testing();
