// -*- c++ -*-

/*
 *  Copyright (C) 2003, Ian Reinhart Geiser <geiseri@kde.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 */

#include <kdebug.h>
#include <kjsembed/jsvalueproxy.h>
#include <kjsembed/jsbinding.h>
#include <qvariant.h>
#include <qbrush.h>

#include "brush_imp.h"

namespace KJSEmbed {
namespace Bindings {

BrushImp::BrushImp( KJS::ExecState *exec, int id )
    : JSProxyImp(exec), mid(id)
{
}

BrushImp::~BrushImp()
{
}

void BrushImp::addBindings( KJS::ExecState *exec, KJS::Object &object ) {

    kdDebug() << "BrushImp::addBindings()" << endl;
    JSValueProxy *op = JSProxy::toValueProxy( object.imp() );
    if ( !op ) {
        kdWarning() << "BrushImp::addBindings() failed, not a JSValueProxy" << endl;
        return;
    }

    if ( op->typeName() != "QBrush" ) {
	kdWarning() << "BrushImp::addBindings() failed, type is " << op->typeName() << endl;
	return;
    }

    JSProxy::MethodTable methods[] = {
	{ Methodstyle,  "style"},
	{ MethodsetStyle, "setStyle"},
	{ MethodColor, "color"},
	{ MethodsetColor, "setColor"},
	{ 0, 0 }
    };

    int idx = 0;
    do {
        BrushImp *meth = new BrushImp( exec, methods[idx].id );
        object.put( exec , methods[idx].name, KJS::Object(meth) );
        ++idx;
    } while( methods[idx].id );

    //
    // Define the enum constants
    //
    struct EnumValue {
	const char *id;
	int val;
    };

    EnumValue enums[] = {
	// BrushType
	{ "NoBrush", 0 },
	{ "SolidBrush", 1 },
	{ "Dense1Pattern", 2 },
	{ "Dense2Pattern", 3 },
	{ "Dense3Pattern", 4 },
	{ "Dense4Pattern", 5 },
	{ "Dense5Pattern", 6 },
	{ "Dense6Pattern", 7 },
	{ "Dense7Pattern", 8 },
	{ "HorPattern", 9 },
	{ "VerPattern", 10 },
	{ "CrossPattern", 11},
	{ "BDiagPattern", 12 },
	{ "FDiagPattern", 13 },
	{ "DiagCrossPattern", 14},

	{ 0, 0 }
    };

    int enumidx = 0;
    do {
        object.put( exec, enums[enumidx].id, KJS::Number(enums[enumidx].val), KJS::ReadOnly );
        ++enumidx;
    } while( enums[enumidx].id );
}

KJS::Value BrushImp::call( KJS::ExecState *exec, KJS::Object &self, const KJS::List &args ) {

    kdDebug() << "BrushImp::call() " << mid << endl;
    JSValueProxy *op = JSProxy::toValueProxy( self.imp() );
    if ( !op ) {
        kdWarning() << "BrushImp::call() failed, not a JSValueProxy" << endl;
        return KJS::Value();
    }

    if ( op->typeName() != "QBrush" ) {
	kdWarning() << "BrushImp::call() failed, type is " << op->typeName() << endl;
	return KJS::Value();
    }

    QBrush brush = op->toVariant().toBrush();

    KJS::Value retValue = KJS::Value();
    switch ( mid ) {
    case Methodstyle:
    {
    	return KJS::Number((int) brush.style() );
	break;
    }
    case MethodsetStyle:
    {
    	int style = extractInt(exec, args, 0);
	brush.setStyle((Qt::BrushStyle)style);
	break;
    }
    case MethodColor:
    {
    	return convertToValue(exec, brush.color());
	break;
    }
    case MethodsetColor:
    {
    	QColor color = extractQColor(exec, args, 0);
	brush.setColor(color);
	break;
    }
    default:
        kdWarning() << "Brush has no method " << mid << endl;
        break;
    }

    op->setValue(brush);
    return retValue;
}

} // namespace KJSEmbed::Bindings
} // namespace KJSEmbed
