/* pal
 *
 * Copyright (C) 2004, Scott Kuhl
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdlib.h>    /* for exit() */
#include <stdio.h>
#include <readline/readline.h>
#include <time.h>

#include "main.h"
#include "output.h"
#include "rl.h"
#include "event.h"
#include "search.h"

char* pal_rl_no_match()
{ return NULL; }

/* prompt for required input */
/* caller is responsible for freeing the prompt and the read line */
gchar* pal_rl_get_line(char* prompt)
{
    char *line = NULL;
    char *locale_prompt = NULL;
    do
    {
	if(line != NULL)
	    g_free(line);

	locale_prompt = g_locale_from_utf8(prompt, -1, NULL, NULL, NULL);
	if(locale_prompt == NULL)
	    line = readline(prompt); /* this shouldn't happen */
	else
	{
	    line = readline(locale_prompt);
	    g_free(locale_prompt);
	}

	if(line == NULL) /* line is null with ^D */
	    exit(0);

	g_strstrip(line);

    }
    while(*line == '\0');

    /* try to convert to utf8 if it isn't ascii or utf8 already. */
    if(!g_utf8_validate(line, -1, NULL))
    {
	gchar* utf8_string = g_locale_to_utf8(line, -1, NULL, NULL, NULL);
	if(utf8_string == NULL)
	{
	    pal_output_error(_("WARNING: Failed to convert your input into UTF-8.\n"));
	    return line;
	}
	else
	{
	    if(settings->verbose)
		g_printerr("%s\n", _("Converted string to UTF-8."));
	    g_free(line);
	    return utf8_string;
	}
    }

    return line;
}

gboolean pal_rl_get_y_n(char* prompt)
{
    gchar *s = NULL;

    do
    {
	g_free(s);
	s = pal_rl_get_line(prompt);
    }
    while(g_ascii_strcasecmp(s, _("y")) != 0 &&
	  g_ascii_strcasecmp(s, _("n")) != 0);

    if(g_ascii_strcasecmp(s, _("y")) == 0)
	return TRUE;
    return FALSE;

}


void pal_rl_default_text_fn()
{
    gchar* locale_default_text;
    locale_default_text = g_locale_from_utf8(pal_rl_default_text, -1,
					     NULL, NULL, NULL);
    if(locale_default_text == NULL)
	rl_insert_text(pal_rl_default_text); /* this shouldn't happen */
    else
    {
	rl_insert_text(locale_default_text);
	g_free(locale_default_text);
    }

    rl_redisplay();
}

/* d gets filled in with GDate entered by the user to find the PalEvent. */
PalEvent* pal_rl_get_event(GDate** d, gboolean allow_global)
{
    gchar* s = NULL;
    PalEvent* event = NULL;
    *d = NULL;


    while(1)
    {
	pal_output_fg(BRIGHT, YELLOW, "> ");
	pal_output_wrap(_("Use \"today\" to access TODO events."),2,2);

	pal_output_fg(BRIGHT, GREEN, "> ");
	pal_output_wrap(_("Valid date formats include: yyyymmdd, Jan 1 2000, 1 Jan 2000, 4 days away"),2,2);

	s = pal_rl_get_line(_("Date for event or search string: "));
	*d = get_query_date(s, FALSE);

	if(*d != NULL)
	{
	    gint event_num = -1;

	    g_print("\n");
	    pal_output_date(*d, TRUE, TRUE);
	    g_print("\n");

	    {   /* Don't allow user select a day without events on it */
		GList* events = get_events(*d);
		gint num_events = g_list_length(events);
		if(num_events==0)
		    continue;
	    }

	    while(1)
	    {
		pal_output_fg(BRIGHT, YELLOW, "> ");
		pal_output_wrap(_("Use \"0\" to use a different date or search string."),2,2);

		s = pal_rl_get_line(_("Select event number: "));
		if(strcmp(s, "0") == 0)
		    return pal_rl_get_event(d, allow_global);

		if(sscanf(s, "%i", &event_num) != 1)
		    continue;

		event = pal_output_event_num(*d, event_num);
		if(event != NULL)
		{
		    if(!event->global || allow_global)
			return event;

		    pal_output_fg(BRIGHT, RED, "> ");
		    pal_output_wrap(_("This event is in a global calendar file.  You can change this event only by editing the global calendar file manually (root access might be required)."),2,2);
		}
	    }
	}

	else /* d == NULL */
	{
	    gchar* search_string = g_strdup(s);
	    gint event_num = -1;
	    GDate* date = g_date_new();
	    g_date_set_time(date,  time(NULL));

	    if(pal_search_view(search_string, date, 365, TRUE) == 0)
		continue;

	    while(1)
	    {
		pal_output_fg(BRIGHT, YELLOW, "> ");
		pal_output_wrap(_("Use \"0\" to use a different date or search string."),2,2);

		s = pal_rl_get_line(_("Select event number: "));
		if(strcmp(s, "0") == 0)
		    return pal_rl_get_event(d, allow_global);

		if(sscanf(s, "%i", &event_num) != 1)
		    continue;

		event = pal_search_event_num(event_num, d, search_string, date, 365);
		if(event != NULL)
		{
		    if(!event->global || allow_global)
			return event;

		    pal_output_fg(BRIGHT, RED, "> ");
		    pal_output_wrap(_("This event is in a global calendar file.  You can change this event only by editing the global calendar file manually (root access might be required)."),2,2);

		}
	    }

	    g_free(search_string);
	}



	if(*d != NULL) g_date_free(*d);
	if(s != NULL) g_free(s);

    } /* end while(1); */

    /* impossible */
    return NULL;



}

/* Returns the hashtable key for the day that the user inputs.  The
 * user can select a TODO event by entering TODO for the date.  */
gchar* pal_rl_get_date()
{
    gchar* s = NULL;
    GDate* d = NULL;

    do
    {
	pal_output_fg(BRIGHT, GREEN, "> ");
	pal_output_wrap(_("Valid date formats include: yyyymmdd, Jan 1 2000, 1 Jan 2000, 4 days away"),2,2);

	s = pal_rl_get_line(_("Date for event: "));
	d = get_query_date(s, FALSE);

	if(d != NULL)
	{
	    gchar buf[1024];

	    g_print("\n");

	    pal_output_fg(BRIGHT, GREEN, "> ");
	    g_print(_("Events on the date you selected:\n"));

	    g_print("\n");
	    pal_output_date(d, TRUE, FALSE);
	    g_print("\n");

	    pal_output_fg(BRIGHT, GREEN, "> ");
	    g_print(_("Is this the correct date?"));
	    g_print("\n");

	    g_date_strftime(buf, 1024, _("%a %e %b %Y - Accept? [y/n]: "), d);

	    if(pal_rl_get_y_n(buf))
	    {
		s = get_key(d);
		g_date_free(d);
		return s;
	    }
	}

	else
	{
	    if(g_ascii_strcasecmp(s, "todo") == 0)
		return g_strdup("TODO");
	}

	if(d != NULL) g_date_free(d);
	if(s != NULL) g_free(s);

    } while(1);



    /* impossible */
    return NULL;
}



