/* getvolinfo.c
 *   Originally published:
 *      http://cygwin.com/ml/cygwin/2007-08/msg00040.html
 *
 * Copyright (c) 2008, Corinna Vinschen
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
#include <stdio.h>
#include <string.h>
#include <sys/cygwin.h>
#define _WIN32_WINNT 0x0600
#include <windows.h>
#include <winternl.h>
#include <ntstatus.h>
#include <wchar.h>

#ifndef FILE_RETURNS_CLEANUP_RESULT_INFO
#define FILE_RETURNS_CLEANUP_RESULT_INFO	0x00000200
#endif
#ifndef FILE_SUPPORTS_POSIX_UNLINK_RENAME
#define FILE_SUPPORTS_POSIX_UNLINK_RENAME	0x00000400
#endif
#ifndef FILE_SUPPORTS_INTEGRITY_STREAMS
#define FILE_SUPPORTS_INTEGRITY_STREAMS		0x04000000
#endif
#ifndef FILE_SUPPORTS_BLOCK_REFCOUNTING
#define FILE_SUPPORTS_BLOCK_REFCOUNTING		0x08000000
#endif
#ifndef FILE_SUPPORTS_SPARSE_VDL
#define FILE_SUPPORTS_SPARSE_VDL		0x10000000
#endif
#ifndef FILE_DAX_VOLUME
#define FILE_DAX_VOLUME				0x20000000
#endif
#ifndef FILE_SUPPORTS_GHOSTING
#define FILE_SUPPORTS_GHOSTING			0x40000000
#endif

int __stdcall
sys_wcstombs (char *tgt, int tlen, const WCHAR *src, int slen)
{
  int ret;

  ret = WideCharToMultiByte (GetOEMCP (), 0, src, slen, tgt, tlen, NULL, NULL);
  if (ret)
    tgt[ret < tlen ? ret : tlen - 1] = '\0';
  return ret;
}

int
main (int argc, char **argv)
{
  char * winpath;
  DWORD flags = 0;
  HANDLE h;
  UNICODE_STRING wpath;
  UNICODE_STRING upath;
  OBJECT_ATTRIBUTES attr;
  IO_STATUS_BLOCK io;
  NTSTATUS stat;

  if (argc < 2)
    {
      fprintf (stderr, "usage: %s path\n", argv[0]);
      return 1;
    }
  winpath = (char *) cygwin_create_path (CCP_POSIX_TO_WIN_A | CCP_ABSOLUTE, argv[1]);
  if (winpath == NULL)
    {
      fprintf (stderr, "create_cygwin_path (posix to win32 absolute) failed\n");
      return 1;
    }
  if (!RtlCreateUnicodeStringFromAsciiz (&wpath, winpath))
    {
      fprintf (stderr, "RtlCreateUnicodeStringFromAsciiz failed\n");
      free (winpath);
      return 1;
    }
  if (!RtlDosPathNameToNtPathName_U (wpath.Buffer, &upath, NULL, NULL))
    {
      fprintf (stderr, "RtlDosPathNameToNtPathName_U failed\n");
      RtlFreeUnicodeString (&wpath);
      free (winpath);
      return 1;
    }
  InitializeObjectAttributes (&attr, &upath, OBJ_CASE_INSENSITIVE, NULL, NULL);
  stat = NtOpenFile (&h, READ_CONTROL, &attr, &io, FILE_SHARE_VALID_FLAGS,
		     FILE_OPEN_FOR_BACKUP_INTENT);
  if (stat == STATUS_INVALID_PARAMETER)
    stat = NtOpenFile (&h, FILE_READ_DATA, &attr, &io,
			 FILE_SHARE_VALID_FLAGS, FILE_OPEN_FOR_BACKUP_INTENT);
  if (stat == STATUS_NO_MEDIA_IN_DEVICE)
    {
      upath.Length = 6 * sizeof (WCHAR);
      stat = NtOpenFile (&h, READ_CONTROL, &attr, &io, FILE_SHARE_VALID_FLAGS, 0);
    }
  if (!NT_SUCCESS (stat))
    {
      char buf[1024];
      wcstombs (buf, upath.Buffer, upath.Length / sizeof (WCHAR));
      buf[upath.Length / sizeof (WCHAR)] = '\0';
      fprintf (stderr, "NtOpenFile(%s) failed, %08x\n", buf, stat);
      return 1;
    }

  char buf[1024];
  char name[256];

  stat = NtQueryVolumeInformationFile (h, &io, buf, 1024,
						FileFsDeviceInformation);
  if (NT_SUCCESS (stat))
    {
      PFILE_FS_DEVICE_INFORMATION pfi =
	    (PFILE_FS_DEVICE_INFORMATION) buf;
      printf ("Device Type        : %x\n", pfi->DeviceType);
      printf ("Characteristics    : %x\n", pfi->Characteristics);
    }
  else
    fprintf (stderr, "FileFsDeviceInformation failed, %08x\n", stat);
  stat = NtQueryVolumeInformationFile (h, &io, buf, 1024,
						FileFsVolumeInformation);
  if (NT_SUCCESS (stat))
    {
      PFILE_FS_VOLUME_INFORMATION pfi =
	    (PFILE_FS_VOLUME_INFORMATION) buf;
      if (pfi->VolumeLabelLength)
	{
	  sys_wcstombs (name, 256, pfi->VolumeLabel,
			pfi->VolumeLabelLength / sizeof (WCHAR));
	  printf ("Volume Name        : <%s>\n", name);
	}
      else
	printf ("Volume Name        : <>\n");
      printf ("Serial Number      : %u\n", pfi->VolumeSerialNumber);
    }
  else
    fprintf (stderr, "FileFsVolumeInformation failed, %08x\n", stat);
  stat = NtQueryVolumeInformationFile (h, &io, buf, 1024,
				       FileFsAttributeInformation);
  if (NT_SUCCESS (stat))
    {
      PFILE_FS_ATTRIBUTE_INFORMATION pfi =
	    (PFILE_FS_ATTRIBUTE_INFORMATION) buf;
      printf ("Max Filenamelength : %u\n",pfi->MaximumComponentNameLength);
      sys_wcstombs (name, 256, pfi->FileSystemName,
		    pfi->FileSystemNameLength / sizeof (WCHAR));
      printf ("Filesystemname     : <%s>\n", name);
      printf ("Flags              : %x\n", flags = pfi->FileSystemAttributes);

      printf ("  FILE_CASE_SENSITIVE_SEARCH        : %s\n",
	      (flags & FILE_CASE_SENSITIVE_SEARCH) ? "TRUE" : "FALSE");
      printf ("  FILE_CASE_PRESERVED_NAMES         : %s\n",
	      (flags & FILE_CASE_PRESERVED_NAMES) ? "TRUE" : "FALSE");
      printf ("  FILE_UNICODE_ON_DISK              : %s\n",
	      (flags & FILE_UNICODE_ON_DISK) ? "TRUE" : "FALSE");
      printf ("  FILE_PERSISTENT_ACLS              : %s\n",
	      (flags & FILE_PERSISTENT_ACLS) ? "TRUE" : "FALSE");
      printf ("  FILE_FILE_COMPRESSION             : %s\n",
	      (flags & FILE_FILE_COMPRESSION) ? "TRUE" : "FALSE");
      printf ("  FILE_VOLUME_QUOTAS                : %s\n",
	      (flags & FILE_VOLUME_QUOTAS) ? "TRUE" : "FALSE");
      printf ("  FILE_SUPPORTS_SPARSE_FILES        : %s\n",
	      (flags & FILE_SUPPORTS_SPARSE_FILES) ? "TRUE" : "FALSE");
      printf ("  FILE_SUPPORTS_REPARSE_POINTS      : %s\n",
	      (flags & FILE_SUPPORTS_REPARSE_POINTS) ? "TRUE" : "FALSE");
      printf ("  FILE_SUPPORTS_REMOTE_STORAGE      : %s\n",
	      (flags & FILE_SUPPORTS_REMOTE_STORAGE) ? "TRUE" : "FALSE");
      printf ("  FILE_RETURNS_CLEANUP_RESULT_INFO  : %s\n",
	      (flags & FILE_RETURNS_CLEANUP_RESULT_INFO) ? "TRUE" : "FALSE");
      printf ("  FILE_SUPPORTS_POSIX_UNLINK_RENAME : %s\n",
	      (flags & FILE_SUPPORTS_POSIX_UNLINK_RENAME) ? "TRUE" : "FALSE");
      printf ("  FILE_VOLUME_IS_COMPRESSED         : %s\n",
	      (flags & FILE_VOLUME_IS_COMPRESSED) ? "TRUE" : "FALSE");
      printf ("  FILE_SUPPORTS_OBJECT_IDS          : %s\n",
	      (flags & FILE_SUPPORTS_OBJECT_IDS) ? "TRUE" : "FALSE");
      printf ("  FILE_SUPPORTS_ENCRYPTION          : %s\n",
	      (flags & FILE_SUPPORTS_ENCRYPTION) ? "TRUE" : "FALSE");
      printf ("  FILE_NAMED_STREAMS                : %s\n",
	      (flags & FILE_NAMED_STREAMS) ? "TRUE" : "FALSE");
      printf ("  FILE_READ_ONLY_VOLUME             : %s\n",
	      (flags & FILE_READ_ONLY_VOLUME) ? "TRUE" : "FALSE");
      printf ("  FILE_SEQUENTIAL_WRITE_ONCE        : %s\n",
	      (flags & FILE_SEQUENTIAL_WRITE_ONCE) ? "TRUE" : "FALSE");
      printf ("  FILE_SUPPORTS_TRANSACTIONS        : %s\n",
	      (flags & FILE_SUPPORTS_TRANSACTIONS) ? "TRUE" : "FALSE");
      printf ("  FILE_SUPPORTS_HARD_LINKS          : %s\n",
	      (flags & FILE_SUPPORTS_HARD_LINKS) ? "TRUE" : "FALSE");
      printf ("  FILE_SUPPORTS_EXTENDED_ATTRIBUTES : %s\n",
	      (flags & FILE_SUPPORTS_EXTENDED_ATTRIBUTES) ? "TRUE" : "FALSE");
      printf ("  FILE_SUPPORTS_OPEN_BY_FILE_ID     : %s\n",
	      (flags & FILE_SUPPORTS_OPEN_BY_FILE_ID) ? "TRUE" : "FALSE");
      printf ("  FILE_SUPPORTS_USN_JOURNAL         : %s\n",
	      (flags & FILE_SUPPORTS_USN_JOURNAL) ? "TRUE" : "FALSE");
      printf ("  FILE_SUPPORTS_INTEGRITY_STREAMS   : %s\n",
	      (flags & FILE_SUPPORTS_INTEGRITY_STREAMS) ? "TRUE" : "FALSE");
      printf ("  FILE_SUPPORTS_BLOCK_REFCOUNTING   : %s\n",
	      (flags & FILE_SUPPORTS_BLOCK_REFCOUNTING) ? "TRUE" : "FALSE");
      printf ("  FILE_SUPPORTS_SPARSE_VDL          : %s\n",
	      (flags & FILE_SUPPORTS_SPARSE_VDL) ? "TRUE" : "FALSE");
      printf ("  FILE_DAX_VOLUME                   : %s\n",
	      (flags & FILE_DAX_VOLUME) ? "TRUE" : "FALSE");
      printf ("  FILE_SUPPORTS_GHOSTING            : %s\n",
	      (flags & FILE_SUPPORTS_GHOSTING) ? "TRUE" : "FALSE");
    }
  else
    fprintf (stderr, "FileFsAttributeInformation failed, %08x\n", stat);

  NtClose (h);
  free (winpath);
  RtlFreeUnicodeString (&upath);
  RtlFreeUnicodeString (&wpath);
  return 0;
}

