/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.uniffle.test;

import java.io.File;
import java.util.Map;

import scala.Tuple2;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.spark.SparkConf;
import org.apache.spark.api.java.JavaPairRDD;
import org.apache.spark.api.java.JavaSparkContext;
import org.apache.spark.shuffle.RssSparkConfig;
import org.apache.spark.sql.SparkSession;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

import org.apache.uniffle.common.config.RssBaseConf;
import org.apache.uniffle.common.rpc.ServerType;
import org.apache.uniffle.coordinator.CoordinatorConf;
import org.apache.uniffle.server.ShuffleServerConf;
import org.apache.uniffle.storage.util.StorageType;

import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertTrue;

public class ShuffleUnregisterWithLocalfileTest extends SparkIntegrationTestBase {

  @BeforeAll
  public static void setupServers(@TempDir File tmpDir) throws Exception {
    CoordinatorConf coordinatorConf = coordinatorConfWithoutPort();
    Map<String, String> dynamicConf = Maps.newHashMap();
    dynamicConf.put(CoordinatorConf.COORDINATOR_REMOTE_STORAGE_PATH.key(), HDFS_URI + "rss/test");
    dynamicConf.put(RssSparkConfig.RSS_STORAGE_TYPE.key(), StorageType.LOCALFILE.name());
    addDynamicConf(coordinatorConf, dynamicConf);
    storeCoordinatorConf(coordinatorConf);

    ShuffleServerConf grpcShuffleServerConf =
        shuffleServerConfWithoutPort(0, tmpDir, ServerType.GRPC);
    grpcShuffleServerConf.setString("rss.storage.type", StorageType.LOCALFILE.name());
    grpcShuffleServerConf.setString(
        RssBaseConf.RSS_STORAGE_BASE_PATH.key(),
        grpcShuffleServerConf.get(RssBaseConf.RSS_STORAGE_BASE_PATH).get(0));
    storeShuffleServerConf(grpcShuffleServerConf);

    ShuffleServerConf nettyShuffleServerConf =
        shuffleServerConfWithoutPort(1, tmpDir, ServerType.GRPC_NETTY);
    nettyShuffleServerConf.setString("rss.storage.type", StorageType.LOCALFILE.name());
    nettyShuffleServerConf.setString(
        RssBaseConf.RSS_STORAGE_BASE_PATH.key(),
        nettyShuffleServerConf.get(RssBaseConf.RSS_STORAGE_BASE_PATH).get(0));
    storeShuffleServerConf(nettyShuffleServerConf);

    startServersWithRandomPorts();
  }

  @Override
  public void updateSparkConfCustomer(SparkConf sparkConf) {}

  private int runCounter = 0;

  @Test
  public void unregisterShuffleTest() throws Exception {
    run();
  }

  @Override
  public Map runTest(SparkSession spark, String fileName) throws Exception {
    // take a rest to make sure shuffle server is registered
    Thread.sleep(3000);
    JavaSparkContext jsc = new JavaSparkContext(spark.sparkContext());
    JavaPairRDD<String, String> javaPairRDD1 =
        jsc.parallelizePairs(
            Lists.newArrayList(
                new Tuple2<>("a", "1"), new Tuple2<>("b", "2"),
                new Tuple2<>("c", "3"), new Tuple2<>("d", "4")),
            2);
    JavaPairRDD<String, Iterable<String>> javaPairRDD = javaPairRDD1.groupByKey().sortByKey();
    Map map = javaPairRDD.collectAsMap();

    // The second run will use the rss. and we should check the effectiveness of unregisterShuffle
    // method.
    if (runCounter == 1) {
      String path =
          grpcShuffleServers
              .get(0)
              .getShuffleServerConf()
              .get(RssBaseConf.RSS_STORAGE_BASE_PATH)
              .get(0);
      String appPath = new File(path).listFiles()[0].getAbsolutePath();
      String shufflePath = appPath + "/0";
      assertTrue(new File(shufflePath).exists());

      spark.sparkContext().env().blockManager().master().removeShuffle(0, true);

      // Wait some time to cleanup the shuffle resource for shuffle-server
      Thread.sleep(1000);
      assertFalse(new File(shufflePath).exists());
      assertTrue(new File(appPath).exists());

      // After unregistering partial shuffle, newly shuffle could work
      map = javaPairRDD.collectAsMap();
      shufflePath = appPath + "/1";
      assertTrue(new File(shufflePath).exists());
    }
    runCounter++;
    return map;
  }
}
