/****************************************************************************
 * arch/arm/src/samv7/sam_timerisr.c
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.  The
 * ASF licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 *
 ****************************************************************************/

/****************************************************************************
 * Included Files
 ****************************************************************************/

#include <nuttx/config.h>

#include <stdint.h>
#include <time.h>
#include <debug.h>
#include <nuttx/arch.h>
#include <arch/board/board.h>

#include "nvic.h"
#include "clock/clock.h"
#include "arm_internal.h"
#include "chip.h"

/****************************************************************************
 * Pre-processor Definitions
 ****************************************************************************/

/* Select MCU-specific settings
 *
 * The SysTick timer is driven by the output of the Mast Clock Controller
 * prescaler output (i.e., the MDIV output divider is not applied so that
 * the driving frequency is the same as the CPU frequency).
 *
 * The SysTick calibration value is fixed to 37500 which allows the
 * generation of a time base of 1 ms with SysTick clock to the maximum
 * frequency on MCK divided by 8. (?)
 */

#define SAM_SYSTICK_CLOCK  BOARD_CPU_FREQUENCY

/* The desired timer interrupt frequency is provided by the definition
 * CLK_TCK (see include/time.h).  CLK_TCK defines the desired number of
 * system clock ticks per second.  That value is a user configurable setting
 * that defaults to 100 (100 ticks per second = 10 MS interval).
 */

#define SYSTICK_RELOAD ((SAM_SYSTICK_CLOCK / CLK_TCK) - 1)

/* The size of the reload field is 24 bits.  Verify that the reload value
 * will fit in the reload register.
 */

#define SYSTICK_MAX 0x00ffffff
#if SYSTICK_RELOAD > SYSTICK_MAX
#  error SYSTICK_RELOAD exceeds the range of the RELOAD register
#endif

/****************************************************************************
 * Private Functions
 ****************************************************************************/

/****************************************************************************
 * Function:  sam_timerisr
 *
 * Description:
 *   The timer ISR will perform a variety of services for various portions
 *   of the systems.
 *
 ****************************************************************************/

static int sam_timerisr(int irq, uint32_t *regs, void *arg)
{
  /* Process timer interrupt */

  nxsched_process_timer();
  return 0;
}

/****************************************************************************
 * Public Functions
 ****************************************************************************/

/****************************************************************************
 * Function:  up_adjtime
 *
 * Description:
 *   Adjusts timer period. This call is used when adjusting timer period as
 *   defined in adjtime() function.
 *
 * Input Parameters:
 *   ppb - Adjustment in parts per billion (nanoseconds per second).
 *         Zero is default rate, positive value makes clock run faster
 *         and negative value slower.
 *
 * Assumptions:
 *   Called from within critical section or interrupt context.
 *
 ****************************************************************************/

#ifdef CONFIG_CLOCK_ADJTIME
void up_adjtime(long ppb)
{
  uint32_t period = SYSTICK_RELOAD;

  if (ppb != 0)
    {
      period -= (long long)ppb * SYSTICK_RELOAD / 1000000000;
    }

  /* Check whether period is at maximum value. */

  if (period > SYSTICK_MAX)
    {
      period = SYSTICK_MAX;
    }

  putreg32(period, NVIC_SYSTICK_RELOAD);
}
#endif

/****************************************************************************
 * Function:  up_timer_initialize
 *
 * Description:
 *   This function is called during start-up to initialize the timer
 *   interrupt.
 *
 ****************************************************************************/

void up_timer_initialize(void)
{
  uint32_t regval;

  /* Configure SysTick to interrupt at the requested rate */

  putreg32(SYSTICK_RELOAD, NVIC_SYSTICK_RELOAD);
  putreg32(0, NVIC_SYSTICK_CURRENT);

  /* Attach the timer interrupt vector */

  irq_attach(SAM_IRQ_SYSTICK, (xcpt_t)sam_timerisr, NULL);

  /* Enable SysTick interrupts (no divide-by-8) */

  regval = (NVIC_SYSTICK_CTRL_CLKSOURCE | NVIC_SYSTICK_CTRL_TICKINT |
            NVIC_SYSTICK_CTRL_ENABLE);
  putreg32(regval, NVIC_SYSTICK_CTRL);

  /* And enable the timer interrupt */

  up_enable_irq(SAM_IRQ_SYSTICK);
}
