/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.remote;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Map;
import java.util.zip.CRC32;
import java.util.zip.CheckedInputStream;
import java.util.zip.CheckedOutputStream;
import org.apache.nifi.events.EventReporter;
import org.apache.nifi.remote.ClientTransactionCompletion;
import org.apache.nifi.remote.Peer;
import org.apache.nifi.remote.Transaction;
import org.apache.nifi.remote.TransactionCompletion;
import org.apache.nifi.remote.TransferDirection;
import org.apache.nifi.remote.codec.FlowFileCodec;
import org.apache.nifi.remote.exception.ProtocolException;
import org.apache.nifi.remote.io.CompressionInputStream;
import org.apache.nifi.remote.io.CompressionOutputStream;
import org.apache.nifi.remote.protocol.DataPacket;
import org.apache.nifi.remote.protocol.Response;
import org.apache.nifi.remote.protocol.ResponseCode;
import org.apache.nifi.remote.util.StandardDataPacket;
import org.apache.nifi.reporting.Severity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractTransaction
implements Transaction {
    protected final Logger logger = LoggerFactory.getLogger(this.getClass());
    protected final Peer peer;
    protected final TransferDirection direction;
    private final CRC32 crc = new CRC32();
    private final boolean compress;
    protected final FlowFileCodec codec;
    protected final EventReporter eventReporter;
    protected final int protocolVersion;
    private final int penaltyMillis;
    protected final String destinationId;
    protected Transaction.TransactionState state;
    protected boolean dataAvailable = false;
    private final long creationNanoTime = System.nanoTime();
    private int transfers = 0;
    private long contentBytes = 0L;

    public AbstractTransaction(Peer peer, TransferDirection direction, boolean useCompression, FlowFileCodec codec, EventReporter eventReporter, int protocolVersion, int penaltyMillis, String destinationId) {
        this.peer = peer;
        this.state = Transaction.TransactionState.TRANSACTION_STARTED;
        this.direction = direction;
        this.compress = useCompression;
        this.codec = codec;
        this.eventReporter = eventReporter;
        this.protocolVersion = protocolVersion;
        this.penaltyMillis = penaltyMillis;
        this.destinationId = destinationId;
    }

    protected void close() throws IOException {
    }

    @Override
    public void send(byte[] content, Map<String, String> attributes) throws IOException {
        this.send(new StandardDataPacket(attributes, new ByteArrayInputStream(content), content.length));
    }

    @Override
    public void error() {
        block2: {
            this.state = Transaction.TransactionState.ERROR;
            try {
                this.close();
            }
            catch (IOException e) {
                this.logger.warn("Failed to close transaction due to {}", (Object)e.getMessage());
                if (!this.logger.isDebugEnabled()) break block2;
                this.logger.warn("", (Throwable)e);
            }
        }
    }

    @Override
    public Transaction.TransactionState getState() {
        return this.state;
    }

    @Override
    public Peer getCommunicant() {
        return this.peer;
    }

    @Override
    public final DataPacket receive() throws IOException {
        try {
            try {
                if (this.state != Transaction.TransactionState.DATA_EXCHANGED && this.state != Transaction.TransactionState.TRANSACTION_STARTED) {
                    throw new IllegalStateException("Cannot receive data from " + String.valueOf(this.peer) + " because Transaction State is " + String.valueOf((Object)this.state));
                }
                if (this.direction == TransferDirection.SEND) {
                    throw new IllegalStateException("Attempting to receive data from " + String.valueOf(this.peer) + " but started a SEND Transaction");
                }
                if (!this.dataAvailable) {
                    return null;
                }
                if (this.transfers > 0) {
                    Response dataAvailableCode = this.readTransactionResponse();
                    switch (dataAvailableCode.getCode()) {
                        case CONTINUE_TRANSACTION: {
                            this.logger.debug("{} {} Indicates Transaction should continue", (Object)this, (Object)this.peer);
                            this.dataAvailable = true;
                            break;
                        }
                        case FINISH_TRANSACTION: {
                            this.logger.debug("{} {} Indicates Transaction should finish", (Object)this, (Object)this.peer);
                            this.dataAvailable = false;
                            break;
                        }
                        default: {
                            throw new ProtocolException("Got unexpected response from " + String.valueOf(this.peer) + " when asking for data: " + String.valueOf(dataAvailableCode));
                        }
                    }
                }
                if (!this.dataAvailable) {
                    return null;
                }
                this.logger.debug("{} Receiving data from {}", (Object)this, (Object)this.peer);
                InputStream is = this.peer.getCommunicationsSession().getInput().getInputStream();
                InputStream dataIn = this.compress ? new CompressionInputStream(is) : is;
                DataPacket packet = this.codec.decode(new CheckedInputStream(dataIn, this.crc));
                if (packet == null) {
                    this.dataAvailable = false;
                } else {
                    ++this.transfers;
                    this.contentBytes += packet.getSize();
                }
                this.state = Transaction.TransactionState.DATA_EXCHANGED;
                return packet;
            }
            catch (IOException ioe) {
                throw new IOException("Failed to receive data from " + String.valueOf(this.peer) + " due to " + String.valueOf(ioe), ioe);
            }
        }
        catch (Exception e) {
            this.error();
            throw e;
        }
    }

    protected abstract Response readTransactionResponse() throws IOException;

    protected final void writeTransactionResponse(ResponseCode response) throws IOException {
        this.writeTransactionResponse(response, null);
    }

    protected void writeTransactionResponse(ResponseCode response, String explanation) throws IOException {
        this.writeTransactionResponse(response, explanation, true);
    }

    protected abstract void writeTransactionResponse(ResponseCode var1, String var2, boolean var3) throws IOException;

    @Override
    public final void confirm() throws IOException {
        block18: {
            try {
                try {
                    if (this.state == Transaction.TransactionState.TRANSACTION_STARTED && !this.dataAvailable && this.direction == TransferDirection.RECEIVE) {
                        this.state = Transaction.TransactionState.TRANSACTION_CONFIRMED;
                        return;
                    }
                    if (this.state != Transaction.TransactionState.DATA_EXCHANGED) {
                        throw new IllegalStateException("Cannot confirm Transaction because state is " + String.valueOf((Object)this.state) + "; Transaction can only be confirmed when state is " + String.valueOf((Object)Transaction.TransactionState.DATA_EXCHANGED));
                    }
                    if (this.direction == TransferDirection.RECEIVE) {
                        Response confirmTransactionResponse;
                        if (this.dataAvailable) {
                            throw new IllegalStateException("Cannot complete transaction because the sender has already sent more data than client has consumed.");
                        }
                        this.logger.trace("{} Sending CONFIRM_TRANSACTION Response Code to {}", (Object)this, (Object)this.peer);
                        String calculatedCRC = String.valueOf(this.crc.getValue());
                        this.writeTransactionResponse(ResponseCode.CONFIRM_TRANSACTION, calculatedCRC);
                        try {
                            confirmTransactionResponse = this.readTransactionResponse();
                        }
                        catch (IOException ioe) {
                            this.logger.error("Failed to receive response code from {} when expecting confirmation of transaction", (Object)this.peer);
                            if (this.eventReporter != null) {
                                this.eventReporter.reportEvent(Severity.ERROR, "Site-to-Site", "Failed to receive response code from " + String.valueOf(this.peer) + " when expecting confirmation of transaction");
                            }
                            throw ioe;
                        }
                        this.logger.trace("{} Received {} from {}", new Object[]{this, confirmTransactionResponse, this.peer});
                        switch (confirmTransactionResponse.getCode()) {
                            case CONFIRM_TRANSACTION: {
                                break;
                            }
                            case BAD_CHECKSUM: {
                                throw new IOException(String.valueOf(this) + " Received a BadChecksum response from peer " + String.valueOf(this.peer));
                            }
                            default: {
                                throw new ProtocolException(String.valueOf(this) + " Received unexpected Response from peer " + String.valueOf(this.peer) + " : " + String.valueOf(confirmTransactionResponse) + "; expected 'Confirm Transaction' Response Code");
                            }
                        }
                        this.state = Transaction.TransactionState.TRANSACTION_CONFIRMED;
                        break block18;
                    }
                    this.logger.debug("{} Sent FINISH_TRANSACTION indicator to {}", (Object)this, (Object)this.peer);
                    this.writeTransactionResponse(ResponseCode.FINISH_TRANSACTION);
                    String calculatedCRC = String.valueOf(this.crc.getValue());
                    Response transactionConfirmationResponse = this.readTransactionResponse();
                    if (transactionConfirmationResponse.getCode() == ResponseCode.CONFIRM_TRANSACTION) {
                        this.logger.trace("{} Received {} from {}", new Object[]{this, transactionConfirmationResponse, this.peer});
                        String receivedCRC = transactionConfirmationResponse.getMessage();
                        if (this.protocolVersion > 3 && !receivedCRC.equals(calculatedCRC)) {
                            this.writeTransactionResponse(ResponseCode.BAD_CHECKSUM);
                            throw new IOException(String.valueOf(this) + " Sent data to peer " + String.valueOf(this.peer) + " but calculated CRC32 Checksum as " + calculatedCRC + " while peer calculated CRC32 Checksum as " + receivedCRC + "; canceling transaction and rolling back session");
                        }
                    } else {
                        throw new ProtocolException("Expected to receive 'Confirm Transaction' response from peer " + String.valueOf(this.peer) + " but received " + String.valueOf(transactionConfirmationResponse));
                    }
                    this.writeTransactionResponse(ResponseCode.CONFIRM_TRANSACTION, "");
                    this.state = Transaction.TransactionState.TRANSACTION_CONFIRMED;
                }
                catch (IOException ioe) {
                    throw new IOException("Failed to confirm transaction with " + String.valueOf(this.peer) + " due to " + String.valueOf(ioe), ioe);
                }
            }
            catch (Exception e) {
                this.error();
                throw e;
            }
        }
    }

    @Override
    public final TransactionCompletion complete() throws IOException {
        try {
            if (this.state != Transaction.TransactionState.TRANSACTION_CONFIRMED) {
                throw new IllegalStateException("Cannot complete transaction with " + String.valueOf(this.peer) + " because state is " + String.valueOf((Object)this.state) + "; Transaction can only be completed when state is " + String.valueOf((Object)Transaction.TransactionState.TRANSACTION_CONFIRMED));
            }
            boolean backoff = false;
            if (this.direction == TransferDirection.RECEIVE) {
                if (this.transfers == 0) {
                    this.state = Transaction.TransactionState.TRANSACTION_COMPLETED;
                    ClientTransactionCompletion clientTransactionCompletion = new ClientTransactionCompletion(false, 0, 0L, System.nanoTime() - this.creationNanoTime);
                    return clientTransactionCompletion;
                }
                this.logger.debug("{} Sending TRANSACTION_FINISHED to {}", (Object)this, (Object)this.peer);
                this.writeTransactionResponse(ResponseCode.TRANSACTION_FINISHED);
                this.state = Transaction.TransactionState.TRANSACTION_COMPLETED;
            } else {
                Response transactionResponse;
                try {
                    transactionResponse = this.readTransactionResponse();
                }
                catch (IOException e) {
                    throw new IOException(String.valueOf(this) + " Failed to receive a response from " + String.valueOf(this.peer) + " when expecting a TransactionFinished Indicator. It is unknown whether or not the peer successfully received/processed the data. " + String.valueOf(e), e);
                }
                this.logger.debug("{} Received {} from {}", new Object[]{this, transactionResponse, this.peer});
                if (transactionResponse.getCode() == ResponseCode.TRANSACTION_FINISHED_BUT_DESTINATION_FULL) {
                    this.peer.penalize(this.destinationId, this.penaltyMillis);
                    backoff = true;
                } else if (transactionResponse.getCode() != ResponseCode.TRANSACTION_FINISHED) {
                    throw new ProtocolException("After sending data to " + String.valueOf(this.peer) + ", expected TRANSACTION_FINISHED response but got " + String.valueOf(transactionResponse));
                }
                this.state = Transaction.TransactionState.TRANSACTION_COMPLETED;
            }
            ClientTransactionCompletion clientTransactionCompletion = new ClientTransactionCompletion(backoff, this.transfers, this.contentBytes, System.nanoTime() - this.creationNanoTime);
            return clientTransactionCompletion;
        }
        catch (IOException ioe) {
            try {
                throw new IOException("Failed to complete transaction with " + String.valueOf(this.peer) + " due to " + String.valueOf(ioe), ioe);
            }
            catch (Exception e) {
                this.error();
                throw e;
            }
        }
        finally {
            this.close();
        }
    }

    @Override
    public final void cancel(String explanation) throws IOException {
        if (this.state == Transaction.TransactionState.TRANSACTION_CANCELED || this.state == Transaction.TransactionState.TRANSACTION_COMPLETED || this.state == Transaction.TransactionState.ERROR) {
            throw new IllegalStateException("Cannot cancel transaction because state is already " + String.valueOf((Object)this.state));
        }
        try {
            this.writeTransactionResponse(ResponseCode.CANCEL_TRANSACTION, explanation == null ? "<No explanation given>" : explanation);
            this.state = Transaction.TransactionState.TRANSACTION_CANCELED;
        }
        catch (IOException ioe) {
            this.error();
            throw new IOException("Failed to send 'cancel transaction' message to " + String.valueOf(this.peer) + " due to " + String.valueOf(ioe), ioe);
        }
        finally {
            this.close();
        }
    }

    public final String toString() {
        return this.getClass().getSimpleName() + "[Url=" + this.peer.getUrl() + ", TransferDirection=" + String.valueOf((Object)this.direction) + ", State=" + String.valueOf((Object)this.state) + "]";
    }

    @Override
    public final void send(DataPacket dataPacket) throws IOException {
        try {
            try {
                if (this.state != Transaction.TransactionState.DATA_EXCHANGED && this.state != Transaction.TransactionState.TRANSACTION_STARTED) {
                    throw new IllegalStateException("Cannot send data to " + String.valueOf(this.peer) + " because Transaction State is " + String.valueOf((Object)this.state));
                }
                if (this.direction == TransferDirection.RECEIVE) {
                    throw new IllegalStateException("Attempting to send data to " + String.valueOf(this.peer) + " but started a RECEIVE Transaction");
                }
                if (this.transfers > 0) {
                    this.writeTransactionResponse(ResponseCode.CONTINUE_TRANSACTION, null, false);
                }
                this.logger.debug("{} Sending data to {}", (Object)this, (Object)this.peer);
                OutputStream os = this.peer.getCommunicationsSession().getOutput().getOutputStream();
                OutputStream dataOut = this.compress ? new CompressionOutputStream(os) : os;
                CheckedOutputStream out = new CheckedOutputStream(dataOut, this.crc);
                this.codec.encode(dataPacket, out);
                if (this.compress) {
                    ((OutputStream)out).close();
                }
                ++this.transfers;
                this.contentBytes += dataPacket.getSize();
                this.state = Transaction.TransactionState.DATA_EXCHANGED;
            }
            catch (IOException ioe) {
                throw new IOException("Failed to send data to " + String.valueOf(this.peer) + " due to " + String.valueOf(ioe), ioe);
            }
        }
        catch (Exception e) {
            this.error();
            throw e;
        }
    }
}

