/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.RemovalCause;
import com.github.benmanes.caffeine.cache.RemovalListener;
import com.github.benmanes.caffeine.cache.Ticker;
import java.time.Duration;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.iceberg.BaseMetadataTable;
import org.apache.iceberg.CommitCallbackTransaction;
import org.apache.iceberg.HasTableOperations;
import org.apache.iceberg.MetadataTableType;
import org.apache.iceberg.MetadataTableUtils;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.Schema;
import org.apache.iceberg.SortOrder;
import org.apache.iceberg.Table;
import org.apache.iceberg.TableOperations;
import org.apache.iceberg.Transaction;
import org.apache.iceberg.catalog.Catalog;
import org.apache.iceberg.catalog.Namespace;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.exceptions.AlreadyExistsException;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CachingCatalog
implements Catalog {
    private static final Logger LOG = LoggerFactory.getLogger(CachingCatalog.class);
    private static final MetadataTableType[] METADATA_TABLE_TYPE_VALUES = MetadataTableType.values();
    private final Catalog catalog;
    private final boolean caseSensitive;
    protected final long expirationIntervalMillis;
    protected final Cache<TableIdentifier, Table> tableCache;

    public static Catalog wrap(Catalog catalog) {
        return CachingCatalog.wrap(catalog, -1L);
    }

    public static Catalog wrap(Catalog catalog, long expirationIntervalMillis) {
        return CachingCatalog.wrap(catalog, true, expirationIntervalMillis);
    }

    public static Catalog wrap(Catalog catalog, boolean caseSensitive, long expirationIntervalMillis) {
        return new CachingCatalog(catalog, caseSensitive, expirationIntervalMillis);
    }

    private CachingCatalog(Catalog catalog, boolean caseSensitive, long expirationIntervalMillis) {
        this(catalog, caseSensitive, expirationIntervalMillis, Ticker.systemTicker());
    }

    protected CachingCatalog(Catalog catalog, boolean caseSensitive, long expirationIntervalMillis, Ticker ticker) {
        Preconditions.checkArgument((expirationIntervalMillis != 0L ? 1 : 0) != 0, (String)"When %s is set to 0, the catalog cache should be disabled. This indicates a bug.", (Object)"cache.expiration-interval-ms");
        this.catalog = catalog;
        this.caseSensitive = caseSensitive;
        this.expirationIntervalMillis = expirationIntervalMillis;
        this.tableCache = this.createTableCache(ticker);
    }

    private Cache<TableIdentifier, Table> createTableCache(Ticker ticker) {
        Caffeine cacheBuilder = Caffeine.newBuilder().softValues();
        if (this.expirationIntervalMillis > 0L) {
            return cacheBuilder.removalListener((RemovalListener)new MetadataTableInvalidatingRemovalListener()).executor(Runnable::run).expireAfterAccess(Duration.ofMillis(this.expirationIntervalMillis)).ticker(ticker).build();
        }
        return cacheBuilder.build();
    }

    private TableIdentifier canonicalizeIdentifier(TableIdentifier tableIdentifier) {
        if (this.caseSensitive) {
            return tableIdentifier;
        }
        return tableIdentifier.toLowerCase();
    }

    public String name() {
        return this.catalog.name();
    }

    public List<TableIdentifier> listTables(Namespace namespace) {
        return this.catalog.listTables(namespace);
    }

    public Table loadTable(TableIdentifier ident) {
        TableIdentifier canonicalized = this.canonicalizeIdentifier(ident);
        Table cached = (Table)this.tableCache.getIfPresent((Object)canonicalized);
        if (cached != null) {
            return cached;
        }
        Table table = (Table)this.tableCache.get((Object)canonicalized, arg_0 -> ((Catalog)this.catalog).loadTable(arg_0));
        if (table instanceof BaseMetadataTable) {
            TableIdentifier originTableIdentifier = TableIdentifier.of((String[])canonicalized.namespace().levels());
            Table originTable = (Table)this.tableCache.get((Object)originTableIdentifier, arg_0 -> ((Catalog)this.catalog).loadTable(arg_0));
            if (originTable instanceof HasTableOperations) {
                TableOperations ops = ((HasTableOperations)originTable).operations();
                MetadataTableType type = MetadataTableType.from(canonicalized.name());
                Table metadataTable = MetadataTableUtils.createMetadataTableInstance(ops, this.catalog.name(), originTableIdentifier, canonicalized, type);
                this.tableCache.put((Object)canonicalized, (Object)metadataTable);
                return metadataTable;
            }
        }
        return table;
    }

    public boolean dropTable(TableIdentifier ident, boolean purge) {
        boolean dropped = this.catalog.dropTable(ident, purge);
        this.invalidateTable(ident);
        return dropped;
    }

    public void renameTable(TableIdentifier from, TableIdentifier to) {
        this.catalog.renameTable(from, to);
        this.invalidateTable(from);
    }

    public void invalidateTable(TableIdentifier ident) {
        this.catalog.invalidateTable(ident);
        TableIdentifier canonicalized = this.canonicalizeIdentifier(ident);
        this.tableCache.invalidate((Object)canonicalized);
        this.tableCache.invalidateAll(this.metadataTableIdentifiers(canonicalized));
    }

    public Table registerTable(TableIdentifier identifier, String metadataFileLocation) {
        Table table = this.catalog.registerTable(identifier, metadataFileLocation);
        this.invalidateTable(identifier);
        return table;
    }

    private Iterable<TableIdentifier> metadataTableIdentifiers(TableIdentifier ident) {
        ImmutableList.Builder builder = ImmutableList.builder();
        for (MetadataTableType type : METADATA_TABLE_TYPE_VALUES) {
            builder.add((Object)TableIdentifier.parse((String)(String.valueOf(ident) + "." + type.name())));
            builder.add((Object)TableIdentifier.parse((String)(String.valueOf(ident) + "." + type.name().toLowerCase(Locale.ROOT))));
        }
        return builder.build();
    }

    public Catalog.TableBuilder buildTable(TableIdentifier identifier, Schema schema) {
        return new CachingTableBuilder(identifier, schema);
    }

    class MetadataTableInvalidatingRemovalListener
    implements RemovalListener<TableIdentifier, Table> {
        MetadataTableInvalidatingRemovalListener() {
        }

        public void onRemoval(TableIdentifier tableIdentifier, Table table, RemovalCause cause) {
            LOG.debug("Evicted {} from the table cache ({})", (Object)tableIdentifier, (Object)cause);
            if (RemovalCause.EXPIRED.equals((Object)cause) && !MetadataTableUtils.hasMetadataTableName(tableIdentifier)) {
                CachingCatalog.this.tableCache.invalidateAll(CachingCatalog.this.metadataTableIdentifiers(tableIdentifier));
            }
        }
    }

    private class CachingTableBuilder
    implements Catalog.TableBuilder {
        private final TableIdentifier ident;
        private final Catalog.TableBuilder innerBuilder;

        private CachingTableBuilder(TableIdentifier identifier, Schema schema) {
            this.innerBuilder = CachingCatalog.this.catalog.buildTable(identifier, schema);
            this.ident = identifier;
        }

        public Catalog.TableBuilder withPartitionSpec(PartitionSpec spec) {
            this.innerBuilder.withPartitionSpec(spec);
            return this;
        }

        public Catalog.TableBuilder withSortOrder(SortOrder sortOrder) {
            this.innerBuilder.withSortOrder(sortOrder);
            return this;
        }

        public Catalog.TableBuilder withLocation(String location) {
            this.innerBuilder.withLocation(location);
            return this;
        }

        public Catalog.TableBuilder withProperties(Map<String, String> properties) {
            this.innerBuilder.withProperties(properties);
            return this;
        }

        public Catalog.TableBuilder withProperty(String key, String value) {
            this.innerBuilder.withProperty(key, value);
            return this;
        }

        public Table create() {
            AtomicBoolean created = new AtomicBoolean(false);
            Table table = (Table)CachingCatalog.this.tableCache.get((Object)CachingCatalog.this.canonicalizeIdentifier(this.ident), identifier -> {
                created.set(true);
                return this.innerBuilder.create();
            });
            if (!created.get()) {
                throw new AlreadyExistsException("Table already exists: %s", new Object[]{this.ident});
            }
            return table;
        }

        public Transaction createTransaction() {
            return this.innerBuilder.createTransaction();
        }

        public Transaction replaceTransaction() {
            return CommitCallbackTransaction.addCallback(this.innerBuilder.replaceTransaction(), () -> CachingCatalog.this.invalidateTable(this.ident));
        }

        public Transaction createOrReplaceTransaction() {
            return CommitCallbackTransaction.addCallback(this.innerBuilder.createOrReplaceTransaction(), () -> CachingCatalog.this.invalidateTable(this.ident));
        }
    }
}

