
/*
 *	af_netrom.c
 *
 *	NET/ROM sockets specific functions
 */

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>

#include "hmalloc.h"

#include "csock.h"
#include "af_netrom.h"
#include "log.h"
#include "config.h"

#ifdef HAVE_NETROM

#include <linux/ax25.h>
#include <linux/netrom.h>
#include <linux/rose.h>

#include "axutils.h"
#include "nrconfig.h"
#include "procutils.h"

/*
 *	Bind() a netrom interface and listen() on it
 */

void netrom_listen(struct listenq_t *lq)
{
	log(L_SINFO, "Binding NET/ROM interface %s:%s%s",
		lq->port, lq->call, lq->compress ? " (compressed)" : "");
	
	if ((lq->fd = socket(lq->af_type, SOCK_SEQPACKET, 0)) < 0) {
		log(L_CRIT, "netrom_listen(): socket(): %s", strerror(errno));
		exit(1);
	}
	
	if (lq->fd > maxfd)
		maxfd = lq->fd;
	
	if (bind(lq->fd, (struct sockaddr *)&lq->sockaddr, lq->addrlen) < 0) {
		log(L_CRIT, "Could not bind() NET/ROM interface %s:%s: %s", 
			lq->port, lq->call, strerror(errno));
		close(lq->fd);
		exit(1);
	}
	
	if (listen(lq->fd, SOMAXCONN) < 0) {
		log(L_CRIT, "Could not listen() on NET/ROM interface %s:%s: %s",
			lq->port, lq->call, strerror(errno));
		close(lq->fd);
		exit(1);
	}
	
	FD_SET(lq->fd, &readfds);
}

/*
 *	Start listening on netrom interfaces
 */

void netrom_listens(void)
{
	struct listenq_t *lq;
	
	for (lq = listenq; (lq); lq = lq->next)
		if (lq->af_type == AF_NETROM)
			netrom_listen(lq);
}

/*
 *	Accept a NETROM connection
 */

void netrom_accept(struct listenq_t *lq)
{
	int i;
	int addrlen, new_fd;
	struct csock_t *s;
	union {
		struct full_sockaddr_ax25 ax25;
		struct sockaddr_rose rose;
	} sockaddr;
	
	i = 1;
	ioctl(lq->fd, FIONBIO, &i);	/* Set up a non-blocking accept() */
	
	addrlen = sizeof(struct full_sockaddr_ax25);
	new_fd = accept(lq->fd, (struct sockaddr *)&sockaddr, &addrlen);
	
	i = 0;
	ioctl(lq->fd, FIONBIO, &i);
	
	if (new_fd < 0) {
		if ((errno == EWOULDBLOCK) || (errno == EINTR))
			return;
		
		log(L_ERR, "NET/ROM accept() error: %s", strerror(errno));
		log(L_ERR, "Closing NET/ROM interface %s:%s (fd %d)", 
			lq->port, lq->call, lq->fd);
		
		close(lq->fd);
		lq->fd = -1;
		return;
	}
	
	i = 1;
	ioctl(new_fd, FIONBIO, &i);	/* Make the socket non-blocking */
	
	s = sock_alloc(new_fd, AF_NETROM, DEF_BUFLEN_NR, DEF_BUFLEN_NR, lq->compress);
	s->eoltype = ax25_eol;
	s->eol = AX25_EOL;
	
	s->call = hstrdup(ax2asc(&sockaddr.ax25.fsa_ax25.sax25_call));
	s->node = hstrdup(ax2asc(&sockaddr.ax25.fsa_digipeater[0]));
	
	addrlen = sizeof(struct full_sockaddr_ax25);
	s->port = hstrdup(lq->port);
	
	sock_login(s);
}

/*
 *	Prepare an outgoing NET/ROM connection
 */
 
int netrom_prepare_outgoing(struct csock_t *s, int argc, char **argv)
{
	char path[20];
	char *portcall;
	struct proc_nr_nodes *np;
	
	if (argc < 2)
		log(L_ERR, "netrom_prepare_outgoing(): Not enough arguments, need at least port and destination");
	
	portcall = nr_config_get_addr(argv[0]);
	if (!portcall) {
		log(L_ERR, "netrom_prepare_outgoing(): Invalid port: %s", argv[0]);
		return -1;
	}
	
	/* Why on earth is this different from ax.25 ????? */
	sprintf(path, "%s %s", portcall, (char *)clucall);
	convert_call(path, &s->sockaddr.ax25);
	s->sockaddr.ax25.fsa_ax25.sax25_family = AF_NETROM;
	s->addrlen = sizeof(struct full_sockaddr_ax25);
	
	if (bind(s->fd, (struct sockaddr *)&s->sockaddr, s->addrlen) == -1) {
		log(L_ERR, "netrom_prepare_outgoing(): AF_NETROM: bind(): %s", strerror(errno));
		return -1;
	}
	
	if ((np = find_node(argv[1], NULL)) == NULL) {
		log(L_ERR, "netrom_prepare_outgoing(): No such node: %s", argv[1]);
		return -1;
	}
	
	if (convert_call(np->call, &s->sockaddr.ax25) == -1) {
		log(L_ERR, "convert_call failed: %s", strerror(errno));
		return -1;
	}
	s->sockaddr.ax25.fsa_ax25.sax25_family = AF_NETROM;
	
	s->call = strdup(argv[1]);
	s->port = strdup(argv[0]);
	s->eoltype = ax25_eol;
	s->eol = AX25_EOL;
	
	return 0;
}

#endif

