
/*
 *	cfg_iface.c
 *
 *	interface.conf directive parsing functions
 */
 
#include <stdlib.h>
#include <string.h>
#include <linux/ax25.h>
#include <linux/netrom.h>
#include <linux/rose.h>
#include <netdb.h>
              
#include "axutils.h"
#include "axconfig.h"
#include "nrconfig.h"

#include "cfg_iface.h"
#include "af_inet.h"
#include "af_ax25.h"
#include "af_netrom.h"
#include "csock.h"
#include "log.h"
#include "hmalloc.h"
#include "cstring.h"

int ax25_ports_loaded = 0;
int netrom_ports_loaded = 0;

/*
 *	Process port options
 */
 
int listen_options(struct listenq_t *p, int argc, char **argv)
{
	int i;
	
	for (i = 0; i < argc; i++) {
		if (!strcasecmp(argv[i], "compress"))
			p->compress = 1;
		else {
			log(L_CRIT, "Invalid listening option: \"%s\"", argv[i]);
			return -1;
		}
	}
	
	return 0;
}

/*
 *	TcpListen
 */

int do_tcplisten(struct listenq_t **dest, int argc, char **argv)
{
	struct listenq_t *p;
	char *host, *port;
	struct hostent *he;
	
	if (argc < 3)
		return -1;
	
	p = hmalloc(sizeof(struct listenq_t));
	
	p->next = *dest;
	*dest = p;
	p->compress = 0;
	
	if ((port = strchr(argv[2], ':'))) {
		host = argv[2];
		*port = '\0';
		port++;
		if (!(he = gethostbyname(host))) {
			log(L_CRIT, "Could not resolve \"%s\": %s", host,
				h_strerror(h_errno));
			return -1;
		}
		memcpy(&p->sockaddr.inet.sin_addr.s_addr, he->h_addr_list[0], he->h_length);
	} else {
		p->sockaddr.inet.sin_addr.s_addr = INADDR_ANY;
		port = argv[2];
	}
	
	p->af_type = AF_INET;
	p->sockaddr.inet.sin_port = atoi(port);
	p->accept = &tcp_accept;
	
	p->port = hstrlwr(hstrdup(argv[1]));
	p->node = hstrlwr(aptoa(p->sockaddr.inet.sin_addr, p->sockaddr.inet.sin_port));
	
	return listen_options(p, argc - 3, &argv[3]);
}

/*
 *	AX25Listen
 */

int do_ax25listen(struct listenq_t **dest, int argc, char **argv)
{
	struct listenq_t *p;
	char *port, *call;
	char *addr;
	
	if (!ax25_ports_loaded) {
		if (ax25_config_load_ports() == 0) {
			log(L_CRIT, "No AX.25 port data configured (axports), cannot listen on AX.25!");
		 	return -1;
		}
		ax25_ports_loaded = 1;
	}
	
	if (argc < 2)
		return -1;
	
	p = hmalloc(sizeof(struct listenq_t));
	
	p->next = *dest;
	*dest = p;
	p->compress = 0;
	
	if ((call = strchr(argv[1], ':'))) {
		*call = '\0';
		call++;
		port = argv[1];
		p->port = hstrdup(port);
		if (strcmp(port, "*") != 0) {
			if ((addr = ax25_config_get_addr(port)) == NULL) {
				log(L_CRIT, "Invalid AX.25 port: %s", port);
				return -1;
			}
		}
	} else {
		call = argv[1];
		port = NULL;
		p->port = hstrdup("*");
	}
	
	strupr(call);
	p->call = hstrdup(call);
	
	p->af_type = AF_AX25;
	p->accept = &ax25_accept;
	
	p->sockaddr.ax25.fsa_ax25.sax25_family = AF_AX25;
	p->sockaddr.ax25.fsa_ax25.sax25_ndigis = 1;
	convert_call_entry(call, p->sockaddr.ax25.fsa_ax25.sax25_call.ax25_call);
	if ((port) && (strcmp(port, "*") != 0))
		convert_call_entry(addr, p->sockaddr.ax25.fsa_digipeater[0].ax25_call);
	else
		p->sockaddr.ax25.fsa_digipeater[0] = null_ax25_address;
	
	p->addrlen = sizeof(struct full_sockaddr_ax25);
	
	return listen_options(p, argc - 2, &argv[2]);
}

/*
 *	NetromListen
 */

int do_netromlisten(struct listenq_t **dest, int argc, char **argv)
{
	struct listenq_t *p;
	char *port = argv[1];
	char *addr;
	
	if (!netrom_ports_loaded) {
		if (nr_config_load_ports() == 0) {
			log(L_CRIT, "No NET/ROM port data configured (nrports), cannot listen on NET/ROM!");
		 	return -1;
		}
		netrom_ports_loaded = 1;
	}
	
	if (argc < 2)
		return -1;
	
	p = hmalloc(sizeof(struct listenq_t));
	
	p->next = *dest;
	*dest = p;
	p->compress = 0;
	
	if ((addr = nr_config_get_addr(port)) == NULL) {
		log(L_CRIT, "Invalid NET/ROM port: %s", port);
		return -1;
	}
	
	p->call = NULL;
	p->port = hstrdup(port);
	p->af_type = AF_NETROM;
	p->accept = &netrom_accept;
	
	p->sockaddr.ax25.fsa_ax25.sax25_family = AF_NETROM;
	p->sockaddr.ax25.fsa_ax25.sax25_ndigis = 0;
	convert_call_entry(addr, p->sockaddr.ax25.fsa_ax25.sax25_call.ax25_call);
	
	p->addrlen = sizeof(struct full_sockaddr_ax25);
	
	return listen_options(p, argc - 2, &argv[2]);
}

