/**************************************************************************
     dkop      disk to DVD backup and restore
     Free software licensed under GNU General Public License v.2
     source URL: http://kornelix.squarespace.com/dkop
***************************************************************************/

#include "zlib.h"
#include <fcntl.h>
#include <dirent.h>
#include <sys/ioctl.h>
#include <sys/mount.h>
#include <linux/cdrom.h>

#define dkop_title "dkop v.34: copy disk files to DVD"                     //  dkop v.34
#define dkop_license "GNU General Public License v.2"
#define debug 0

//  parameters and limits

#define normfont "monospace"
#define boldfont "monospace bold"                                          //  v.27
#define showfile "gedit"                                                   //  program to view text file    v.29
#define vrcc  512*1024                                                     //  verify read I/O size
#define maxnx 1000                                                         //  max no. include/exclude recs.
#define maxfs 200000                                                       //  max no. disk or DVD files    v.26
#define maxhist 200                                                        //  max no. history files        v.29
#define giga  1000000000.0                                                 //  gigabyte (not 1024**3)
#define dvdcapmin 1.0                                                      //  DVD medium capacity sanity check
#define dvdcapmax 8.0                                                      //    (gigabytes)
#define modtimetolr 0.001                                                  //  tolerance for file "equal" mod times
#define nano 0.000000001                                                   //  nanosecond
#define gforce "-use-the-force-luke=tty -use-the-force-luke=notray"        //  growisofs hidden options

//  special control files on DVD

#define V_DKOPDIRK  "/dkop-data/"                                          //  dkop special files on DVD
#define V_FILEPOOP  V_DKOPDIRK "filepoop"                                  //  directory data file
#define V_JOBFILE   V_DKOPDIRK "jobfile"                                   //  backup job data file
#define V_DATETIME  V_DKOPDIRK "datetime"                                  //  date-time file

//  GTK GUI widgets

GtkWidget      *mWin, *mVbox, *mScroll, *mLog;                             //  main window
GtkTextBuffer  *logBuff;
GtkWidget      *fc_dialogbox, *fc_widget;                                  //  file-chooser dialog
GtkWidget      *editwidget;                                                //  edit box in file selection dialogs

//  file scope variables

int      killFlag;                                                         //  tell function to quit
int      pauseFlag;                                                        //  tell function to pause/resume
int      menuLock;                                                         //  menu lock flag
int      commFail;                                                         //  command failure flag   v.13
int      threadcount;                                                      //  count of running threads
int      Fdialog;                                                          //  dialog in progress
int      clrun;                                                            //  flag, command line 'run' command
char     subprocName[20];                                                  //  name of created subprocess
char     scriptParam[200];                                                 //  parameter from script file
char     mbmode[20], mvmode[20];                                           //  actual backup, verify modes
double   pctdone;                                                          //  % done from growisofs
char     scrFile[maxfcc];                                                  //  command line script file
char     backupDT[16];                                                     //  nominal backup date: yyyymmdd-hhmm

char     TFdiskfiles[100], TFdvdfiles[100];                                //  /home/user/.dkop/xxxx  scratch files
char     TFjobfile[100], TFfilepoop[100], TFdatetime[100];
char     TFrestorefiles[100], TFrestoredirks[100];

//  available DVD devices and mount points

int      ndvds, maxdvds = 8;
int      nmps, maxmps = 8;
char     dvddevs[8][20];                                                   //  DVD devices, /dev/sr0 etc.
char     dvddesc[8][40];                                                   //  DVD device descriptions
char     dvddevdesc[8][60];                                                //  combined device and description
char     dvdmps[8][40];                                                    //  DVD mount points, /media/xxxx

//  backup job data

char     BJfile[maxfcc];                                                   //  backup job file
char     BJdvd[20];                                                        //  DVD device: /dev/hdb
char     BJmp[40];                                                         //  mount point: /media/dvd
int      BJmpcc;                                                           //  mount point cc
double   BJcap;                                                            //  DVD medium capacity (GB)  v.14
char     BJbmode[20];                                                      //  backup: full/incremental/accumulate
char     BJvmode[20];                                                      //  verify: full/incremental/thorough
int      BJndvd;                                                           //  no. DVD media required
int      BJval;                                                            //  backup job data validated
int      BJmod;                                                            //  backup job data modified
char    *BJinex[maxnx];                                                    //  backup include/exclude records
int      BJfiles[maxnx];                                                   //  corresp. file count per rec
double   BJbytes[maxnx];                                                   //  corresp. byte count per rec
int      BJdvdno[maxnx];                                                   //  corresp. DVD sequence no. (1,2...)
int      BJnx;                                                             //  actual record count < maxnx

//  DVD medium data

int      dvdmtd;                                                           //  DVD mounted
char     mediumDT[16];                                                     //  DVD medium last use date-time
int      mediumNB;                                                         //  DVD medium dkop usage count
time_t   dvdtime;                                                          //  DVD device mod time
int      dvdnum;                                                           //  DVD medium sequence no.
char     oldlabel[32];                                                     //  old DVD label                v.29
char     newlabel[32];                                                     //  new DVD label (full backup)  v.30

//  disk file data (backup file set)

struct dfrec {                                                             //  disk file record    v.14
   char     *file;                                                         //    directory/filename
   double   size;                                                          //    byte count
   double   mtime;                                                         //    mod time       v.23
   int      stat;                                                          //    fstat() status
   int      dvd;                                                           //    assigned DVD number
   int      inclx;                                                         //    include rec for this file
   char     disp;                                                          //    status: new modified unchanged
   char     ivf;                                                           //    flag for incr. verify
};

dfrec    Drec[maxfs];                                                      //  disk file data records
int      Dnf;                                                              //  actual file count < maxfs
double   Dbytes;                                                           //  disk files, total bytes
double   Dbytes2;                                                          //  bytes for current DVD medium

//  DVD file data 

struct vfrec {                                                             //  DVD file record     v.23
   char     *file;                                                         //    directory/file (- /media/xxx)
   double   size;                                                          //    byte count
   double   mtime;                                                         //    mod time
   int      stat;                                                          //    fstat() status
   char     disp;                                                          //    status: deleted modified unchanged
};

vfrec    Vrec[maxfs];                                                      //  DVD file data records        v.23
int      Vnf;                                                              //  actual file count < maxfs
double   Vbytes;                                                           //  DVD files, total bytes

//  disk:DVD comparison data

int      nnew, ndel, nmod, nunc;                                           //  counts: new del mod unch
int      Mfiles;                                                           //  new + mod + del file count
double   Mbytes;                                                           //  new + mod files, total bytes

//  restore job data

char    *RJinex[maxnx];                                                    //  file restore include/exclude recs.
int      RJnx;                                                             //  actual RJinex count < maxnx
int      RJval;                                                            //  restore job data validated
char     RJfrom[maxfcc];                                                   //  restore copy-from: /home/.../
char     RJto[maxfcc];                                                     //  restore copy-to: /home/.../

struct   rfrec {                                                           //  restore file record
   char     *file;                                                         //  restore filespec: /home/.../file.ext 
   double   size;                                                          //  byte count
};

rfrec    Rrec[maxfs];                                                      //  restore file data records  v.24
int      Rnf;                                                              //  actual file count < maxfs
double   Rbytes;                                                           //  total bytes


//  dkop local functions    

int initfunc(void * data);                                                 //  GTK init function
void buttonfunc(GtkWidget *item, const char *menu);                        //  process toolbar button event
void menufunc(GtkWidget *item, const char *menu);                          //  process menu select event
void * menu_thread_func(void *);                                           //  run menu function in a thread
void * script_thread_func(void *);                                         //  execute script file in a thread

int quit_dkop(char *);                                                     //  exit application
int clearScreen(char *);                                                   //  clear logging window
int signalFunc(char *);                                                    //  kill/pause/resume curr. function
int checkKillPause();                                                      //  test flags: killFlag and pauseFlag

int fileOpen(char *);                                                      //  file open dialog
int fileSave(char *);                                                      //  file save dialog
int BJload(char * fspec);                                                  //  backup job data <<< file
int BJstore(char * fspec, int ndvd = 0);                                   //  backup job data >>> file
int BJvload(char *);                                                       //  load job file from DVD

int BJedit(char *);                                                        //  backup job edit dialog
int BJedit_event(zdialog *zd, const char * event);                         //  dialog event function
int BJedit_compl(zdialog *zd, int zstat);                                  //  dialog completion function
int BJedit_stuff(zdialog * zd);                                            //  stuff dialog widgets with job data
int BJedit_fetch(zdialog * zd);                                            //  set job data from dialog widgets

int Backup(char *);                                                        //  backup menu function
int FullBackup(char * vmode);                                              //  full backup + verify
int IncrBackup(char * bmode, char * vmode);                                //  incremental / accumulate + verify
int Verify(char *);                                                        //  verify functions

int Report(char *);                                                        //  report functions
int get_backup_files(char *);                                              //  file stats. per include/exclude
int report_summary_diffs(char *);                                          //  disk:DVD differences summary
int report_directory_diffs(char *);                                        //  disk:DVD differences by directory
int report_file_diffs(char *);                                             //  disk:DVD differences by file
int list_backup_files(char *);                                             //  list all disk files in backup set
int list_DVD_files(char *);                                                //  list all files on DVD
int find_files(char *);                                                    //  find files on disk, DVD, backup hist
int view_backup_hist(char *);                                              //  view backup history files

int RJedit(char *);                                                        //  restore job edit dialog
int RJresponse(GtkDialog *, int, void *);                                  //  RJedit response
int RJlist(char *);                                                        //  list DVD files to be restored
int Restore(char *);                                                       //  file restore function

int getDVDs();                                                             //  get avail. DVD devices, mount points
int setDVDdevice(char *);                                                  //  set DVD device and mount point  
int setDVDlabel(char *);                                                   //  set new DVD label    v.30
int mountDVD(char *);                                                      //  mount DVD + echo outputs (menu)
int mountDVDn(int ntry);                                                   //  mount DVD + echo outputs (internal)
int ejectDVD(char *);                                                      //  eject DVD + echo outputs
int resetDVD(char *);                                                      //  hardware reset   v.12
int eraseDVD(char *);                                                      //  fill DVD with zeros (long time)
int formatDVD(char *);                                                     //  quick format DVD  v.12

int saveScreen(char *);                                                    //  save logging window to file
int helpFunc(char *);                                                      //  help function

int fc_dialog(char * dirk);                                                //  file chooser dialog
int fc_response(GtkDialog *, int, void *);                                 //  fc_dialog response

int writeDT();                                                             //  write date-time to temp file
int save_filepoop();                                                       //  save file owner & permissions data
int restore_filepoop();                                                    //  restore file owner & perm. data
int createBackupHist();                                                    //  create backup history file

int inexParse(char * rec, char *& rtype, char *& fspec);                   //  parse include/exclude record
int BJvalidate(char *);                                                    //  validate backup job data
int RJvalidate();                                                          //  validate restore job data
int nxValidate(char **recs, int nr);                                       //  validate include/exclude recs

int dGetFiles();                                                           //  generate backup file list from job
int vGetFiles();                                                           //  find all DVD files
int rGetFiles();                                                           //  generate restore job file list
int setFileDisps();                                                        //  set file disps: new del mod unch
int SortFileList(char * recs, int RL, int NR, char sort);                  //  sort file list in memory
int filecomp(const char *file1, const char *file2);                        //  compare directories before files

int BJreset();                                                             //  reset backup job file data
int RJreset();                                                             //  reset restore job data
int dFilesReset();                                                         //  reset disk file data and free memory
int vFilesReset();                                                         //  reset DVD file data and free memory
int rFilesReset();                                                         //  reset restore file data, free memory

char * checkFile(char * dfile, int compf, double &tcc);                    //  validate file on backup medium
char * copyFile(char *vfile, char *dfile);                                 //  copy file from DVD to disk

int track_filespec(char * filespec);                                       //  track filespec on screen, no scroll
int track_filespec_err(char * filespec, char * errmess);                   //  error logger for track_filespec()
char  * kleenex(const char *name);                                         //  clean exotic file names for output 

int do_shell(char * pname, char * command);                                //  shell command + output to window
int track_growisofs_files(char *buff);                                     //  convert %done to filespec, output


//  dkop menu table                                                        //  v.09

struct menuent {
   char     menu1[20], menu2[40];                                          //  top-menu, sub-menu
   int      lock;                                                          //  lock funcs: no run parallel
   int      thread;                                                        //  run in thread
   int      (*mfunc)(char *);                                              //  processing function
};

#define nmenu  43
struct menuent menus[nmenu] = {
//  top-menu    sub-menu                lock  thread  menu-function
{  "button",   "edit job",                1,    0,    BJedit         },
{  "button",   "clear",                   0,    0,    clearScreen    },
{  "button",   "run job",                 1,    1,    Backup         },
{  "button",   "run DVD",                 1,    1,    Backup         },
{  "button",   "pause",                   0,    0,    signalFunc     },
{  "button",   "resume",                  0,    0,    signalFunc     },
{  "button",   "kill job",                0,    0,    signalFunc     },
{  "button",   "quit",                    0,    0,    quit_dkop      },
{  "File",     "open job",                1,    0,    fileOpen       },
{  "File",     "open DVD",                1,    0,    BJvload        },
{  "File",     "edit job",                1,    0,    BJedit         },
{  "File",     "show job",                0,    0,    BJvalidate     },
{  "File",     "save job",                0,    0,    fileSave       },
{  "File",     "run job",                 1,    1,    Backup         },
{  "File",     "run DVD",                 1,    1,    Backup         },
{  "File",     "quit",                    0,    0,    quit_dkop      },
{  "Backup",   "full",                    1,    1,    Backup         },
{  "Backup",   "incremental",             1,    1,    Backup         },
{  "Backup",   "accumulate",              1,    1,    Backup         },
{  "Verify",   "full",                    1,    1,    Verify         },
{  "Verify",   "incremental",             1,    1,    Verify         },
{  "Verify",   "thorough",                1,    1,    Verify         },
{  "Report",   "get backup files",        1,    1,    Report         },
{  "Report",   "diffs summary",           1,    1,    Report         },
{  "Report",   "diffs by directory",      1,    1,    Report         },
{  "Report",   "diffs by file",           1,    1,    Report         },
{  "Report",   "list backup files",       1,    1,    Report         },
{  "Report",   "list DVD files",          1,    1,    Report         },
{  "Report",   "find files",              1,    1,    Report         },
{  "Report",   "view backup hist",        1,    1,    Report         },
{  "Report",   "save screen",             0,    0,    saveScreen     },
{  "Restore",  "setup DVD restore",       1,    0,    RJedit         },
{  "Restore",  "list restore files",      1,    1,    RJlist         },
{  "Restore",  "restore files",           1,    1,    Restore        },
{  "DVD",      "set DVD device",          1,    0,    setDVDdevice   },
{  "DVD",      "set DVD label",           1,    0,    setDVDlabel    },
{  "DVD",      "erase DVD",               1,    1,    eraseDVD       },
{  "DVD",      "format DVD",              1,    1,    formatDVD      },
{  "DVD",      "mount DVD",               1,    1,    mountDVD       },
{  "DVD",      "eject DVD",               1,    1,    ejectDVD       },
{  "DVD",      "reset DVD",               0,    1,    resetDVD       },
{  "Help",     "about",                   0,    0,    helpFunc       },
{  "Help",     "contents",                0,    1,    helpFunc       }  };


//  dkop main program

int main(int argc, char *argv[])
{
   PangoFontDescription    *deffont;                                       //  main window default font
   int      ii, mbar, tbar;                                                //  menubar and toolbar handles
   int      mFile, mBackup, mVerify, mReport, mRestore, mDVD, mHelp;       //  submenu handles

   gtk_init(&argc, &argv);                                                 //  GTK command line options
   if (! g_thread_supported())                                             //  suddenly required for new gtk   v.33
         g_thread_init(0);                                                 //  initz. GTK for threads
   gdk_threads_init();
   zlockInit();

   initz_appfiles(0);                                                      //  set up application directory  v.27

   clrun = 0;                                                              //  no command line run command
   *scrFile = 0;                                                           //  no script file
   *BJfile = 0;                                                            //  no backup job file

   for (ii = 1; ii < argc; ii++)                                           //  get command line options
   {
      if (strEqu(argv[ii],"-job") && argc > ii+1)                          //  -job jobfile  (load job)
            strcpy(BJfile,argv[++ii]);
      else if (strEqu(argv[ii],"-run") && argc > ii+1)                     //  -run jobfile  (load and run job)
          { strcpy(BJfile,argv[++ii]); clrun++; }
      else if (strEqu(argv[ii],"-script") && argc > ii+1)                  //  -script scriptfile  (execute script)
            strcpy(scrFile,argv[++ii]);
      else  strcpy(BJfile,argv[ii]);                                       //  assume a job file and load it   v.28
   }
   
   mWin = gtk_window_new(GTK_WINDOW_TOPLEVEL);                             //  create main window
   gtk_window_set_title(GTK_WINDOW(mWin),dkop_title);
   gtk_window_set_position(GTK_WINDOW(mWin),GTK_WIN_POS_CENTER);
   gtk_window_set_default_size(GTK_WINDOW(mWin),800,500);
   
   mVbox = gtk_vbox_new(0,0);                                              //  vertical packing box
   gtk_container_add(GTK_CONTAINER(mWin),mVbox);                           //  add to main window

   mScroll = gtk_scrolled_window_new(0,0);                                 //  scrolled window
   gtk_box_pack_end(GTK_BOX(mVbox),mScroll,1,1,0);                         //  add to main window mVbox
   
   mLog = gtk_text_view_new();                                             //  text edit window
   gtk_container_add(GTK_CONTAINER(mScroll),mLog);                         //  add to scrolled window

   deffont = pango_font_description_from_string(normfont);                 //  default font   v.27
   gtk_widget_modify_font(mLog,deffont);

   logBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(mLog));                //  get related text buffer
   gtk_text_buffer_set_text(logBuff,"", -1);

   mbar = create_menubar(mVbox);                                           //  create menu bar and menus   v.25
      mFile = add_menubar_item(mbar,"File",menufunc);                      
         add_submenu_item(mFile,"open job",menufunc);
         add_submenu_item(mFile,"open DVD",menufunc);
         add_submenu_item(mFile,"edit job",menufunc);
         add_submenu_item(mFile,"show job",menufunc);
         add_submenu_item(mFile,"save job",menufunc);
         add_submenu_item(mFile,"run job",menufunc);
         add_submenu_item(mFile,"run DVD",menufunc);
         add_submenu_item(mFile,"quit",menufunc);
      mBackup = add_menubar_item(mbar,"Backup",menufunc);
         add_submenu_item(mBackup,"full",menufunc);
         add_submenu_item(mBackup,"incremental",menufunc);
         add_submenu_item(mBackup,"accumulate",menufunc);
      mVerify = add_menubar_item(mbar,"Verify",menufunc);
         add_submenu_item(mVerify,"full",menufunc);
         add_submenu_item(mVerify,"incremental",menufunc);
         add_submenu_item(mVerify,"thorough",menufunc);
      mReport = add_menubar_item(mbar,"Report",menufunc);
         add_submenu_item(mReport,"get backup files",menufunc);
         add_submenu_item(mReport,"diffs summary",menufunc);
         add_submenu_item(mReport,"diffs by directory",menufunc);
         add_submenu_item(mReport,"diffs by file",menufunc);
         add_submenu_item(mReport,"list backup files",menufunc);
         add_submenu_item(mReport,"list DVD files",menufunc);
         add_submenu_item(mReport,"find files",menufunc);
         add_submenu_item(mReport,"view backup hist",menufunc);
         add_submenu_item(mReport,"save screen",menufunc);
      mRestore = add_menubar_item(mbar,"Restore",menufunc);
         add_submenu_item(mRestore,"setup DVD restore",menufunc);
         add_submenu_item(mRestore,"list restore files",menufunc);
         add_submenu_item(mRestore,"restore files",menufunc);
      mDVD = add_menubar_item(mbar,"DVD",menufunc);
         add_submenu_item(mDVD,"set DVD device",menufunc);
         add_submenu_item(mDVD,"set DVD label",menufunc);
         add_submenu_item(mDVD,"mount DVD",menufunc);
         add_submenu_item(mDVD,"eject DVD",menufunc);
         add_submenu_item(mDVD,"reset DVD",menufunc);
         add_submenu_item(mDVD,"erase DVD",menufunc);
         add_submenu_item(mDVD,"format DVD",menufunc);
      mHelp = add_menubar_item(mbar,"Help",menufunc);
         add_submenu_item(mHelp,"about",menufunc);
         add_submenu_item(mHelp,"contents",menufunc);

   tbar = create_toolbar(mVbox);                                           //  create toolbar and buttons   v.25

   add_toolbar_button(tbar,"edit job","edit backup job","icons/editjob.png",buttonfunc);
   add_toolbar_button(tbar,"run job","run backup job","icons/burn.png",buttonfunc);
   add_toolbar_button(tbar,"run DVD","run job on DVD","icons/burn.png",buttonfunc);
   add_toolbar_button(tbar,"pause","pause running job","gtk-media-pause",buttonfunc);
   add_toolbar_button(tbar,"resume","resume running job","gtk-media-play",buttonfunc);
   add_toolbar_button(tbar,"kill job","kill running job","gtk-stop",buttonfunc);
   add_toolbar_button(tbar,"clear","clear screen","gtk-clear",buttonfunc);
   add_toolbar_button(tbar,"quit","quit dkop","gtk-quit",buttonfunc);

   gtk_widget_show_all(mWin);                                              //  show all widgets

   G_SIGNAL(mWin,"destroy",quit_dkop,0)                                    //  connect window destroy event
   
   gtk_init_add((GtkFunction) initfunc,0);                                 //  setup initial call from gtk_main()

   gdk_threads_enter();
   gtk_main();                                                             //  process window events
   gdk_threads_leave();

   return 0;
}


//  initial function called from gtk_main() at startup

int initfunc(void * data)
{
   int         ii;
   char        *home, *appdirk;
   time_t      datetime;
   pthread_t   tid;
   
   menufunc(null,"Help");                                                  //  show version and license
   menufunc(null,"about");
   
   appdirk = getz_appdirk();                                               //  temp file names   v.29
   sprintf(TFdiskfiles,"%s/diskfiles",appdirk);
   sprintf(TFdvdfiles,"%s/dvdfiles",appdirk);
   sprintf(TFfilepoop,"%s/filepoop",appdirk);
   sprintf(TFjobfile,"%s/jobfile",appdirk);
   sprintf(TFdatetime,"%s/datetime",appdirk);
   sprintf(TFrestorefiles,"%s/restorefiles.sh",appdirk);
   sprintf(TFrestoredirks,"%s/restoredirks.sh",appdirk);

   datetime = time(0);
   printf("dkop errlog %s \n",ctime(&datetime));

   menuLock = threadcount = Fdialog = 0;                                   //  initialize controls
   killFlag = pauseFlag = commFail = 0;
   strcpy(subprocName,"");
   strcpy(scriptParam,"");

   getDVDs();                                                              //  get avail. DVDs and mount points 

   strcpy(BJdvd,dvddevs[0]);                                               //  DVD device
   strcpy(BJmp,dvdmps[0]);                                                 //  mount point
   BJmpcc = strlen(BJmp);                                                  //  mount point cc
   BJcap = 4.0;                                                            //  capacity (default GB)
   *oldlabel = 0;                                                          //  old DVD label             v.29
   *newlabel = 0;                                                          //  new DVD label             v.30
   strcpy(BJbmode,"full");                                                 //  backup mode
   strcpy(BJvmode,"full");                                                 //  verify mode
   BJval = 0;                                                              //  not validated
   BJmod = 0;                                                              //  not modified

   BJnx = 4;                                                               //  backup include/exclude recs
   for (ii = 0; ii < BJnx; ii++) BJinex[ii] = zmalloc(50);
   
   home = getenv("HOME");                                                  //  get "/home/username"
   if (! home) home = "/home/xxx";
   strcpy(BJinex[0],"# dkop default backup job");                          //  initz. default backup specs
   sprintf(BJinex[1],"include %s/*",home);                                 //  include /home/username/*     v.14
   sprintf(BJinex[2],"exclude %s/.Trash/*",home);                          //  exclude /home/username/.Trash/*
   sprintf(BJinex[3],"exclude %s/.thumbnails/*",home);                     //  exclude /home/username/.thumbnails/*

   Dnf = Vnf = Rnf = Mfiles = 0;                                           //  file counts = 0
   Dbytes = Dbytes2 = Vbytes = Mbytes = 0.0;                               //  byte counts = 0

   strcpy(RJfrom,"/home/");                                                //  file restore copy-from location
   strcpy(RJto,"/home/");                                                  //  file restore copy-to location
   RJnx = 0;                                                               //  no. restore include/exclude recs
   RJval = 0;                                                              //  restore job not validated

   strcpy(mediumDT,"unknown");                                             //  DVD medium last backup date-time
   mediumNB = 0;                                                           //    and dkop usage count
   dvdtime = -1;                                                           //  DVD device mod time
   dvdmtd = 0;                                                             //  DVD not mounted

   if (*BJfile) {                                                          //  command line job file
      BJload(BJfile);
      if (commFail) return 0;
   }

   if (clrun) {                                                            //  command line run command
      menufunc(null,"File");
      menufunc(null,"run job");
   }

   if (*scrFile) pthread_create(&tid,0,script_thread_func,0);              //  command line script file  v.09

   return 0;
}


//  process toolbar button events (simulate menu selection)

void buttonfunc(GtkWidget *item, const char *button)
{
   char     button2[20], *pp;
   
   strncpy0(button2,button,19);
   pp = strchr(button2,'\n');                                              //  replace \n with blank
   if (pp) *pp = ' ';

   menufunc(item,"button");
   menufunc(item,button2);
   return;
}


//  process menu selection event

void menufunc(GtkWidget *, const char *menu)                               //  v.09 table-driven
{
   int            ii;
   static char    menu1[20] = "", menu2[40] = "";
   char           command[100];
   pthread_t      tid;
   
   for (ii = 0; ii < nmenu; ii++) 
         if (strEqu(menu,menus[ii].menu1)) break;                          //  mark top-menu selection
   if (ii < nmenu) { strcpy(menu1,menu); return;  }
   
   for (ii = 0; ii < nmenu; ii++) 
         if (strEqu(menu1,menus[ii].menu1) && 
             strEqu(menu,menus[ii].menu2)) break;                          //  mark sub-menu selection
   if (ii < nmenu) strcpy(menu2,menu);

   else {                                                                  //  no match to menus
      wprintf(mLog," *** bad command: %s \n",menu);
      commFail++;
      return;
   }

   if (menuLock && menus[ii].lock) {                                       //  no lock funcs can run parallel
      zmessageACK("wait for current function to complete");
      return;
   }

   if (! menuLock) {   
      killFlag = pauseFlag = commFail = 0;                                 //  reset controls
      *subprocName = 0;
   }
   
   if (! *scrFile) {
      snprintf(command,99,"\n""command: %s > %s \n",menu1,menu2);
      wprintx(mLog,0,command,boldfont);                                    //  v.27
   }
   
   if (menus[ii].thread) {
      ++threadcount;
      if (menus[ii].lock) ++menuLock;
      pthread_create(&tid,0,menu_thread_func,(void *) ii);                 //  start thread to run menu function
   }
   
   else  {
      if (menus[ii].lock) ++menuLock;
      menus[ii].mfunc(menu2);                                              //  or call menu function directly
      if (menus[ii].lock) --menuLock;
   }

   return;
}


//  thread shell function - run menu function in a thread

void * menu_thread_func(void * mii)                                        //  v.09
{
   int ii = (int) mii;

   menus[ii].mfunc(menus[ii].menu2);                                       //  call menu function
   if (menus[ii].lock) --menuLock;
   --threadcount;
   return 0;
}


//  thread function to execute menu commands from a script file            //  v.09

void * script_thread_func(void * nothing)
{
   FILE     *fid;
   int      cc, Nth;
   char     buff[200], *pp;
   char     menu1[20], menu2[40];
   
   fid = fopen(scrFile,"r");                                               //  open file
   if (! fid) {
      wprintf(mLog," *** can't open script file: %s \n",scrFile);
      commFail++;
      *scrFile = 0;
      return 0;
   }

   while (true)
   {
      if (checkKillPause()) break;                                         //  exit script
      if (commFail) break;
      
      pp = fgets_trim(buff,199,fid,1);                                     //  read next record
      if (! pp) break;                                                     //  EOF

      wprintf(mLog,"\n""Script: %s \n",buff);                              //  write to log
      
      pp = strchr(buff,'#');                                               //  get rid of comments
      if (pp) *pp = 0;
      cc = strTrim(buff);                                                  //  and trailing blanks
      if (cc < 2) continue;

      *menu1 = *menu2 = 0;
      *scriptParam = 0;

      Nth = 1;                                                             //  parse menu1 > menu2 > parameter
      pp = strField(buff,'>',Nth++);
      if (pp) strncpy0(menu1,pp,20);
      pp = strField(buff,'>',Nth++);
      if (pp) strncpy0(menu2,pp,40);
      pp = strField(buff,'>',Nth++);
      if (pp) strncpy0(scriptParam,pp,200);
      
      strTrim(menu1);                                                      //  get rid of trailing blanks
      strTrim(menu2);

      if (strEqu(menu1,"exit")) break;
      
      menufunc(null,menu1);                                                //  simulate menu entries
      menufunc(null,menu2);

      while (threadcount) sleep(1);                                        //  if thread, wait for compl.
      while (Fdialog) sleep(1);                                            //  if dialog, wait for compl.
   }
   
   wprintf(mLog,"script exiting \n");
   fclose(fid);
   *scrFile = 0;
   return 0;
}


//  quit dkop, with last chance to save edits to backup job data

int quit_dkop(char * menu)
{
   int      yn;

   if (BJmod) {                                                            //  job data was modified
      yn = zmessageYN("SAVE changes to dkop job?");                        //  give user a chance to save mods
      if (yn) fileSave(null);
   }

   if (dvdmtd) ejectDVD(null);                                             //  eject DVD

   gtk_main_quit();                                                        //  tell gtk_main() to quit
   return 0;
}


//  clear logging window

int clearScreen(char * menu)
{
   wclear(mLog);
   return 0;
}


//  kill/pause/resume current function - called from menu function

int signalFunc(char * menu)
{
   if (strEqu(menu,"kill job"))
   {
      if (! menuLock) {
         wprintf(mLog,"\n""ready \n");                                     //  already dead
         return 0;
      }
      
      if (killFlag) {                                                      //  redundant kill
         if (*subprocName) {
            wprintf(mLog," *** kill again: %s \n",subprocName);
            signalProc(subprocName,"kill");                                //  kill subprocess
         }
         else wprintf(mLog," *** waiting for function to quit \n");        //  or wait for function to die
         return 0;
      }

      wprintf(mLog," *** KILL current function \n");                       //  initial kill
      pauseFlag = 0;
      killFlag = 1;

      if (*subprocName) {
         signalProc(subprocName,"resume");
         signalProc(subprocName,"kill");
      }

      return 0;
   }

   if (strEqu(menu,"pause")) {
      pauseFlag = 1;
      if (*subprocName) signalProc(subprocName,"pause");
      return 0;
   }

   if (strEqu(menu,"resume")) {
      pauseFlag = 0;
      if (*subprocName) signalProc(subprocName,"resume");
      return 0;
   }
   
   else zappcrash("signalFunc: %s",menu);
   return 0;
}


//  check kill and pause flags
//  called periodically from long-running functions

int checkKillPause()
{
   while (pauseFlag)                                                       //  idle loop while paused
   {
      zsleep(0.1);
      zmainloop();                                                         //  process menus
   }

   if (killFlag) return 1;                                                 //  return true = stop now
   return 0;                                                               //  return false = continue
}


//  file open dialog - get backup job data from a file

int fileOpen(char * menu)
{
   char        *file;
   int         err = 0;
   
   if (*scriptParam) {                                                     //  get file from script   v.09
      strcpy(BJfile,scriptParam);
      *scriptParam = 0;
      err = BJload(BJfile);
      return err;
   }

   ++Fdialog;

   file = zgetfile("open backup job",getz_appdirk(),"open","hidden");      //  get file from user     v.27b
   if (file) {
      if (strlen(file) > maxfcc-2) zappcrash("pathname too big");
      strcpy(BJfile,file);
      zfree(file);
      err = BJload(BJfile);                                                //  get job data from file
   }
   else err = 1;

   --Fdialog;
   return err;
}


//  file save dialog - save backup job data to a file

int fileSave(char * menu)
{
   char        *file;
   int         nstat, err = 0;
   
   if (*scriptParam) {                                                     //  get file from script   v.09
      strcpy(BJfile,scriptParam);
      *scriptParam = 0;
      BJstore(BJfile);
      return 0;
   }

   if (! BJval) {
      nstat = zmessageYN("Job data not valid, save anyway?");
      if (! nstat) return 0;
   }

   ++Fdialog;

   if (! *BJfile) strcpy(BJfile,"dkop.job");                               //  if no job file, use default
   file = zgetfile("save backup job",BJfile,"save","hidden");
   if (file) {
      if (strlen(file) > maxfcc-2) zappcrash("pathname too big");
      strcpy(BJfile,file);
      zfree(file);
      err = BJstore(BJfile);
      if (! err) BJmod = 0;                                                //  job not modified  v.11
   }
   
   --Fdialog;
   return 0;
}


//  backup job data <<< file
//  errors not checked here are checked in BJvalidate()

int BJload(char * fspec)
{
   FILE           *fid;
   char           buff[1000];
   char          *fgs, *rtype, *rdata;
   int            cc, Nth, nerrs;

   BJreset();                                                              //  clear old job from memory
   nerrs = 0;

   snprintf(buff,999,"\n""loading job file: %s \n",fspec);
   wprintx(mLog,0,buff,boldfont);

   fid = fopen(fspec,"r");                                                 //  open file
   if (! fid) {
      wprintf(mLog," *** cannot open job file: %s \n",fspec);
      commFail++;
      return 1;
   }

   while (true)                                                            //  read file
   {
      fgs = fgets_trim(buff,998,fid,1);
      if (! fgs) break;                                                    //  EOF
      cc = strlen(buff);
      if (cc > 996) { 
         wprintf(mLog," *** input record too big \n");
         nerrs++;
         continue;
      }

      Nth = 1;
      rtype = strField(buff,' ',Nth++);                                    //  parse 1st field, record type
      if (! rtype) rtype = "#";                                            //  blank record is comment
      strToLower(rtype);

      if (strEqu(rtype,"device")) {
         rdata = strField(buff,' ',Nth++);                                 //  DVD device: /dev/dvd
         if (rdata) strncpy0(BJdvd,rdata,19);
         continue;
      }

      if (strEqu(rtype,"mount")) {
         rdata = strField(buff,' ',Nth++);                                 //  DVD mount point: /media/dvd
         if (rdata) {
            strncpy0(BJmp,rdata,39);
            BJmpcc = strlen(BJmp);
            if (BJmpcc && (BJmp[BJmpcc-1] == '/')) BJmp[BJmpcc--] = 0;     //  remove trailing /
         }
         continue;
      }

      if (strEqu(rtype,"dvdcap")) {
         rdata = strField(buff,' ',Nth++);                                 //  DVD capacity, GB
         if (rdata) convSD(rdata,BJcap);
         continue;
      }

      if (strEqu(rtype,"backup")) {
         rdata = strField(buff,' ',Nth++);                                 //  backup mode
         if (rdata) {
            strToLower(rdata);
            strncpy0(BJbmode,rdata,19);
         }
         continue;
      }

      if (strEqu(rtype,"verify")) {
         rdata = strField(buff,' ',Nth++);                                 //  verify mode
         if (rdata) {
            strToLower(rdata);
            strncpy0(BJvmode,rdata,19);
         }
         continue;
      }
      
      if (strcmpv(rtype,"include","exclude","#",0)) {
         BJinex[BJnx] = zmalloc(cc+1);                                     //  include/exclude or comment rec.  v.31
         strcpy(BJinex[BJnx],buff);
         if (++BJnx >= maxnx) {
            wprintf(mLog," *** exceed %d include/exclude recs \n",maxnx);
            nerrs++;
            break;
         }
         continue;
      }
      
      wprintf(mLog," *** unrecognized record: %s \n",buff);
      nerrs++;
      continue;
   }
   
   fclose(fid);                                                            //  close file
   BJmod = 0;                                                              //  new job, not modified

   BJvalidate(0);                                                          //  validation checks, set BJval
   if (! nerrs && BJval) return 0;
   BJval = 0;
   commFail++;
   return 1;
}


//  backup job data >>> file

int BJstore(char * fspec, int ndvd)
{
   FILE     *fid;
   int      ii;

   fid = fopen(fspec,"w");                                                 //  open file
   if (! fid) { 
      wprintf(mLog," *** cannot open file: %s \n",fspec); 
      commFail++;
      return 1; 
   }
   
   fprintf(fid,"device %s \n",BJdvd);                                      //  device /dev/dvd
   fprintf(fid,"mount %s \n",BJmp);                                        //  mount /media/dvd
   fprintf(fid,"dvdcap  %.1f \n",BJcap);                                   //  dvdcap  N.N
   fprintf(fid,"backup %s \n",BJbmode);                                    //  backup full/incremental/accumulate
   fprintf(fid,"verify %s \n",BJvmode);                                    //  verify full/incremental/thorough

   if (! ndvd)
      for (ii = 0; ii < BJnx; ii++)                                        //  output all include/exclude recs
         fprintf(fid,"%s \n",BJinex[ii]);

   else {                                                                  //  output only recs for one DVD medium 
      for (ii = 0; ii < BJnx && BJdvdno[ii] < ndvd; ii++);                 //    of multi-volume set   v.14
      for ( ; ii < BJnx; ii++)
         if (BJdvdno[ii] <= ndvd)                                          //  matching includes (dvdno = ndvd)
            fprintf(fid,"%s \n",BJinex[ii]);                               //   + all following excludes (dvdno = 0)
   }

   fclose(fid);
   return 0;
}   


//  backup job data <<< DVD job file
//  get job file from prior backup to this same medium

int BJvload(char * menu)
{
   char     vjfile[100];
   
   BJreset();                                                              //  reset job data   

   mountDVDn(2);                                                           //  (re) mount DVD
   if (! dvdmtd) { 
      commFail++; 
      return 1; 
   }

   strcpy(vjfile,BJmp);                                                    //  dvd mount point
   strcat(vjfile,V_JOBFILE);                                               //  + dvd job file
   BJload(vjfile);                                                         //  load job file (BJval set)
   if (BJval) return 0;
   commFail++;
   return 1;
}


//  edit dialog for backup job data
//  v.26  add convenience buttons to load from or save to job file

int BJedit(char * menu)
{
   zdialog        *zd;
   
   ++Fdialog;

   zd = zdialog_new("edit backup job","browse","done","cancel","clear",0);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=20");              //  v.28 "name=val"
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb3","hb1");
   zdialog_add_widget(zd,"vbox","vb4","hb1");

   zdialog_add_widget(zd,"label","labdev","vb1","DVD device ");            //  DVD device   [_________][v]
   zdialog_add_widget(zd,"label","labmp","vb1","mount point");             //  mount point  [_________][v]
   zdialog_add_widget(zd,"label","labcap","vb1","capacity GB");            //  capacity GB  [___]
   zdialog_add_widget(zd,"combo","entdvd","vb2",BJdvd);
   zdialog_add_widget(zd,"combo","entmp","vb2",BJmp);
   zdialog_add_widget(zd,"hbox","hb3","vb2");
   zdialog_add_widget(zd,"entry","entcap","hb3",0,"scc=5");

   zdialog_add_widget(zd,"label","labbmode","vb3","Backup Mode  ");
   zdialog_add_widget(zd,"label","labvmode","vb4","Verify Mode   ");       //  Backup Mode      Verify Mode
   zdialog_add_widget(zd,"radio","bmrb1","vb3","full");                    //  (o) full         (o) full
   zdialog_add_widget(zd,"radio","bmrb2","vb3","incremental");             //  (o) incremental  (o) incremental
   zdialog_add_widget(zd,"radio","bmrb3","vb3","accumulate");              //  (o) accumulate   (o) thorough
   zdialog_add_widget(zd,"radio","vmrb1","vb4","full");
   zdialog_add_widget(zd,"radio","vmrb2","vb4","incremental");
   zdialog_add_widget(zd,"radio","vmrb3","vb4","thorough");

   zdialog_add_widget(zd,"hsep","sep1","dialog");
   zdialog_add_widget(zd,"hbox","hb4","dialog",0,"space=10");              //  [open job] [DVD job] [save as]
   zdialog_add_widget(zd,"button","bopen","hb4","open job file");
   zdialog_add_widget(zd,"button","bdvd","hb4","open DVD job");
   zdialog_add_widget(zd,"button","bsave","hb4"," save as ");

   zdialog_add_widget(zd,"hsep","sep2","dialog");                          //  edit box for include/exclude recs
   zdialog_add_widget(zd,"label","labinex","dialog","Include / Exclude Files");
   zdialog_add_widget(zd,"frame","frminex","dialog",0,"expand");
   zdialog_add_widget(zd,"scrwin","scrwinex","frminex");
   zdialog_add_widget(zd,"edit","edinex","scrwinex");
   
   BJedit_stuff(zd);                                                       //  stuff dialog widgets with job data

   zdialog_resize(zd,400,500);
   zdialog_run(zd,BJedit_event,BJedit_compl);                              //  run dialog + event and response funcs
   return 0;
}


//  job edit dialog widgets  <<<  backup job data in memory

int BJedit_stuff(zdialog * zd)
{
   int      ii;

   for (ii = 0; ii < ndvds; ii++)
         zdialog_cb_app(zd,"entdvd",dvddevdesc[ii]);
   for (ii = 0; ii < nmps; ii++) 
         zdialog_cb_app(zd,"entmp",dvdmps[ii]);

   zdialog_stuff(zd,"entcap",BJcap);

   if (strEqu(BJbmode,"full")) zdialog_stuff(zd,"bmrb1",1);
   if (strEqu(BJbmode,"incremental")) zdialog_stuff(zd,"bmrb2",1);
   if (strEqu(BJbmode,"accumulate")) zdialog_stuff(zd,"bmrb3",1);
   if (strEqu(BJvmode,"full")) zdialog_stuff(zd,"vmrb1",1);
   if (strEqu(BJvmode,"incremental")) zdialog_stuff(zd,"vmrb2",1);
   if (strEqu(BJvmode,"thorough")) zdialog_stuff(zd,"vmrb3",1);

   editwidget = zdialog_widget(zd,"edinex");
   wclear(editwidget);
   for (int ii = 0; ii < BJnx; ii++) 
         wprintf(editwidget,"%s""\n",BJinex[ii]);
   
   return 0;
}


//  job edit dialog widgets  >>>  backup job data in memory

int BJedit_fetch(zdialog * zd)
{
   int            ii, ftf, cc;
   char           text[40], *pp;

   BJreset();                                                              //  reset job data

   zdialog_fetch(zd,"entdvd",text,19);                                     //  get DVD device
   strncpy0(BJdvd,text,19);
   pp = strchr(BJdvd,' ');
   if (pp) *pp = 0;

   zdialog_fetch(zd,"entmp",text,39);                                      //  mount point
   strncpy0(BJmp,text,39);
   strTrim(BJmp);
   BJmpcc = strlen(BJmp);
   if (BJmpcc && (BJmp[BJmpcc-1] == '/')) BJmp[BJmpcc--] = 0;              //  remove trailing /
   
   zdialog_fetch(zd,"entcap",BJcap);                                       //  capacity GB

   zdialog_fetch(zd,"bmrb1",ii); if (ii) strcpy(BJbmode,"full");           //  backup mode
   zdialog_fetch(zd,"bmrb2",ii); if (ii) strcpy(BJbmode,"incremental");
   zdialog_fetch(zd,"bmrb3",ii); if (ii) strcpy(BJbmode,"accumulate");

   zdialog_fetch(zd,"vmrb1",ii); if (ii) strcpy(BJvmode,"full");           //  verify mode
   zdialog_fetch(zd,"vmrb2",ii); if (ii) strcpy(BJvmode,"incremental");
   zdialog_fetch(zd,"vmrb3",ii); if (ii) strcpy(BJvmode,"thorough");

   for (ftf = 1;;)
   {
      pp = wscanf(editwidget,ftf);                                         //  include/exclude recs.
      if (! pp) break;
      cc = strTrim(pp);                                                    //  remove trailing blanks
      BJinex[BJnx] = zmalloc(cc+1);                                        //  allocate memory
      strcpy(BJinex[BJnx],pp);                                             //  copy new record
      if (++BJnx >= maxnx) {
         wprintf(mLog," *** exceed %d include/exclude recs \n",maxnx); 
         break;
      }
   }
   
   BJmod++;                                                                //  job modified
   BJvalidate(0);                                                          //  check for errors, set BJval   
   return 0;
}


//  edit dialog response function

int BJedit_compl(zdialog *zd, int zstat)
{   
   if (zstat == 1) {                                                       //  do file-chooser dialog
      fc_dialog("/home");
      return 0;
   }

   if (zstat == 4) {
      wclear(editwidget);                                                  //  clear include/exclude recs
      return 0;
   }

   if (zstat != 2) {                                                       //  ?? response
      zdialog_destroy(zd);                                                 //  quit dialog
      --Fdialog;
      return 0;
   }
                                                                           //  OK response
   BJedit_fetch(zd);                                                       //  get all job data from dialog widgets
   
   if (! BJval) commFail++;

   zdialog_destroy(zd);                                                    //  destroy dialog
   --Fdialog;
   return 0;
}


//  edit dialog event function

int BJedit_event(zdialog * zd, const char * event)
{
   int      err = 0;

   if (strEqu(event,"bopen")) {
      err = fileOpen("");                                                  //  get job file from user
      if (! err) BJedit_stuff(zd);                                         //  stuff dialog widgets
   }

   if (strEqu(event,"bdvd")) {
      err = BJvload("");                                                   //  get job file on DVD
      if (! err) BJedit_stuff(zd);                                         //  stuff dialog widgets
   }

   if (strEqu(event,"bsave")) {
      BJedit_fetch(zd);                                                    //  get job data from dialog widgets
      fileSave("");                                                        //  save to file
   }

   return 0;
}


//  thread function: perform DVD backup using growisofs utility
//  (see technical notes in user guide for more information)

int Backup(char * menu)
{
   strcpy(mbmode,"");
   strcpy(mvmode,"");

   if (strcmpv(menu,"full","incremental","accumulate",0))                  //  backup only
      strcpy(mbmode,menu);

   if (strEqu(menu,"run DVD")) BJvload(null);                              //  load job file from DVD if req.

   if (strcmpv(menu,"run job","run DVD",0)) {                              //  if run job or job on DVD,
      if (BJval) {                                                         //   and valid job file,
         strcpy(mbmode,BJbmode);                                           //    use job file backup & verify modes
         strcpy(mvmode,BJvmode);
      }
   }

   if (! BJval) {                                                          //  check for errors
      wprintf(mLog," *** no valid backup job \n");
      goto backup_done;
   }

   if (strEqu(mbmode,"full")) FullBackup(mvmode);                          //  full backup (+ verify)
   else  IncrBackup(mbmode,mvmode);                                        //  incremental / accumulate (+ verify)

backup_done:
   wprintf(mLog,"ready \n");
   return 0;
}


//  full backup using multiple DVD media if required

int FullBackup(char * BJvmode)
{
   FILE           *fid = 0;
   int            cc, err, gerr, ii, yn;
   char           command[200];
   char           *dfile, vfile[maxfcc];
   double         secs, bspeed;
   timeval        time0;

   dGetFiles();                                                            //  get backup file set
   if (Dnf == 0) {
      wprintf(mLog," *** nothing to back-up \n");
      goto backup_fail;
   }

   vFilesReset();                                                          //  reset DVD files data

   wprintx(mLog,0,"\n""begin full backup \n",boldfont);
   wprintf(mLog," files: %d  bytes: %.0f \n",Dnf,Dbytes);                  //  files and bytes to copy
   
   if (! *newlabel) strcpy(newlabel,oldlabel);                             //  use old label if no new one defined
   if (! *newlabel) strcpy(newlabel,"dkop");                               //  if neither, use default "dkop"

   cc = strlen(newlabel);                                                  //  label:X  >>  label        v.32
   if (newlabel[cc-2] == ':') newlabel[cc-2] = 0;                          //  (stop label:A:A:A ...)

   for (dvdnum = 1; dvdnum <= BJndvd; dvdnum++)                            //  loop for each DVD
   {
      if (BJndvd > 1) zmessageACK("insert DVD medium no. %d",dvdnum);
      
      if (dvdnum == 1) strcat(newlabel,":A");                              //  append :A to :Z to label        v.30
      else  ++newlabel[strlen(newlabel)-1];
      
      err = mountDVDn(1);                                                  //  mount to get medium stats
      if (err == EPERM) goto backup_fail;                                  //  no permission    v.20
      if (err) wprintf(mLog," continuing anyway ... \n");                  //  v.20

      BJstore(TFjobfile,dvdnum);                                           //  copy job file (this DVD) to temp file
      save_filepoop();                                                     //  + owner and permissions to temp file
      writeDT();                                                           //  create date-time & usage temp file

      fid = fopen(TFdiskfiles,"w");                                        //  temp file for growisofs path-list
      if (! fid) {
         wprintf(mLog," *** cannot open /tmp scratch file \n");
         goto backup_fail;
      }

      fprintf(fid,"%s=%s\n",V_JOBFILE +1,TFjobfile);                       //  add job file to growisofs list
      fprintf(fid,"%s=%s\n",V_FILEPOOP +1,TFfilepoop);                     //  add directory poop file
      fprintf(fid,"%s=%s\n",V_DATETIME +1,TFdatetime);                     //  add date-time file

      Dbytes2 = 0.0;
      for (ii = 0; ii < Dnf; ii++)                                         //  process all files in backup set
      {
         if (Drec[ii].dvd != dvdnum) continue;                             //  screen for DVD no.
         dfile = Drec[ii].file;                                            //  add to growisofs path-list
         repl_1str(dfile,vfile,"=","\\\\=");                               //  replace "=" with "\\=" in file name
         fprintf(fid,"%s=%s\n",vfile+1,dfile);                             //  directories/file=/directories/file
         Dbytes2 += Drec[ii].size;
      }

      fclose(fid);

      wprintf(mLog," writing DVD medium %d of %d, %.0f MB \n",
                           dvdnum, BJndvd, Dbytes2/1000000.0);
      start_timer(&time0);                                                 //  start timer for growisofs

      sprintf(command,"/usr/bin/growisofs -Z %s -R -graft-points "         //  build growisofs command line
                      "-iso-level 4 -gui -V %s %s -path-list %s 2>&1",
                      BJdvd,newlabel,gforce,TFdiskfiles);

      gerr = do_shell("growisofs", command);                               //  do growisofs, echo outputs

      if (gerr) {                                                          //  growisofs failed
         wprintf(mLog," *** growisofs failed \n");
         if (dvdnum == 1) goto backup_fail;                                //  if 1st DVD, nothing to lose, quit
         yn = zmessageYN("Write DVD medium no. %d again? \n"
                         " (else abort whole job)", dvdnum);               //  v.28
         if (! yn) goto backup_fail;
         --dvdnum; 
         continue;                                                         //  repeat same DVD, salvage job
      }

      if (checkKillPause()) goto backup_fail;                              //  killed by user

      secs = get_timer(&time0);                                            //  output statistics
      wprintf(mLog," backup time: %.0f secs \n",secs);
      bspeed = Dbytes2/1000000.0/secs;
      wprintf(mLog," backup speed: %.2f MB/sec \n",bspeed);
      wprintf(mLog," backup complete \n");
      if (BJndvd > 1) wprintf(mLog," (DVD medium no. %d) \n",dvdnum);

      ejectDVD("");                                                        //  v.27   compensate flakey driver

      if (*BJvmode) {
         Verify(BJvmode);                                                  //  do verify if req.
         ejectDVD("");                                                     //  eject after verify
         if (commFail) {
            if (dvdnum == 1) goto backup_fail;                             //  if 1st DVD, nothing to lose, quit
            yn = zmessageYN("Write DVD medium no. %d again? \n"
                            " (else abort whole job)", dvdnum);            //  v.28
            if (! yn) goto backup_fail;
            --dvdnum; 
            commFail = 0;
            continue;                                                      //  repeat same DVD, salvage job
         }
      }

      createBackupHist();                                                  //  create backup history file     v.29
   }

   return 0;

backup_fail:
   commFail++;
   wprintx(mLog,0," *** BACKUP FAILED \n",boldfont);
   return 0;
}


//  incremental / accumulate backup (one DVD only)

int IncrBackup(char * BJbmode, char * BJvmode)
{
   FILE           *fid = 0;
   int            err, gerr, ii;
   char           command[200];
   char           *dfile, vfile[maxfcc], disp;
   double         secs, bspeed;
   timeval        time0;

   err = mountDVDn(3);                                                     //  requires successful mount
   if (! dvdmtd) goto backup_fail;

   dGetFiles();                                                            //  get backup files
   vGetFiles();                                                            //  get DVD files
   setFileDisps();                                                         //  file disps: new mod del unch

   if (! Dnf) {
      wprintf(mLog," *** no backup files \n");
      goto backup_fail;
   }

   if (! Vnf) {
      wprintf(mLog," *** no DVD files \n");
      goto backup_fail;
   }

   snprintf(command,99,"\n""begin %s backup \n",BJbmode);
   wprintx(mLog,0,command,boldfont);
   wprintf(mLog," files: %d  bytes: %.0f \n",Mfiles,Mbytes);               //  files and bytes to copy

   if (Mfiles == 0) {                                                      //  nothing to back up
      wprintf(mLog," nothing to back-up \n");
      return 0;
   }

   if (! *newlabel) strcpy(newlabel,oldlabel);                             //  use old label if no new one defined
   if (! *newlabel) strcpy(newlabel,"dkop");                               //  if neither, use default "dkop"

   fid = fopen(TFdiskfiles,"w");                                           //  temp file for growisofs path-list
   if (! fid) {
      wprintf(mLog," *** cannot open /tmp scratch file \n");
      goto backup_fail;
   }

   BJstore(TFjobfile);                                                     //  copy job file to temp file
   save_filepoop();                                                        //  + file owner & permissions
   writeDT();                                                              //  create date-time & usage temp file

   fprintf(fid,"%s=%s\n",V_JOBFILE +1,TFjobfile);                          //  add job file to growisofs list
   fprintf(fid,"%s=%s\n",V_FILEPOOP +1,TFfilepoop);                        //  add directory poop file
   fprintf(fid,"%s=%s\n",V_DATETIME +1,TFdatetime);                        //  add date-time file

   for (ii = 0; ii < Dnf; ii++) {                                          //  process new and modified disk files
      disp = Drec[ii].disp;
      if ((disp == 'n') || (disp == 'm')) {                                //  new or modified file
         dfile = Drec[ii].file;                                            //  add to growisofs path-list
         repl_1str(dfile,vfile,"=","\\\\=");                               //  replace "=" with "\\=" in file name
         fprintf(fid,"%s=%s\n",vfile+1,dfile);                             //  directories/file=/directories/file
         Drec[ii].ivf = 1;                                                 //  set flag for incr. verify
      }
   }

   if (strEqu(BJbmode,"incremental")) {                                    //  incremental backup (not accumulate)
      for (ii = 0; ii < Vnf; ii++) {                                       //  process deleted files still on DVD
         if (Vrec[ii].disp == 'd') {
            dfile = Vrec[ii].file;                                         //  add to growisofs path-list
            repl_1str(dfile,vfile,"=","\\\\=");                            //  replace "=" with "\\=" in file name
            fprintf(fid,"%s=%s\n",vfile+1,"/dev/null");                    //  directories/file=/dev/null
         }
      }
   }

   fclose(fid);

   start_timer(&time0);                                                    //  start timer for growisofs

   sprintf(command,"/usr/bin/growisofs -M %s -R -graft-points "            //  build growisofs command line
                     "-iso-level 4 -gui -V %s %s -path-list %s 2>&1",
                     BJdvd,newlabel,gforce,TFdiskfiles);

   gerr = do_shell("growisofs", command);                                  //  do growisofs, echo outputs

   if (gerr) {                                                             //  growisofs failed
      wprintx(mLog,0," *** growisofs failed \n",boldfont);
      goto backup_fail;
   }

   if (checkKillPause()) goto backup_fail;                                 //  killed by user

   secs = get_timer(&time0);                                               //  output statistics
   wprintf(mLog," backup time: %.0f secs \n",secs);
   bspeed = Mbytes/1000000.0/secs;
   wprintf(mLog," backup speed: %.2f MB/sec \n",bspeed);
   wprintf(mLog," backup complete \n");

   ejectDVD("");                                                           //  v.27   compensate flakey driver
   vFilesReset();                                                          //  reset DVD files

   if (*BJvmode) Verify(BJvmode);                                          //  do verify if req.
   else wprintf(mLog," ready \n");
   if (! commFail) createBackupHist();                                     //  create backup history file     v.29
   return 0;

backup_fail:
   commFail++;
   wprintx(mLog,0," *** BACKUP FAILED \n",boldfont);
   vFilesReset();
   return 0;
}


//  thread function, verify DVD medium integrity

int Verify(char * menu)
{
   int            ii, vfiles, verrs = 0, cerrs = 0;
   char           *dfile, *errmess = 0;
   double         secs, dcc1, vbytes, vspeed;
   timeval        time0;

   vGetFiles();                                                            //  get DVD files
   wprintf(mLog," %d files on DVD \n",Vnf);
   if (! Vnf) goto verify_exit;

   vfiles = verrs = cerrs = 0;
   vbytes = 0.0;

   start_timer(&time0);

   if (strEqu(menu,"full"))                                                //  verify all files are readable
   {
      wprintx(mLog,0,"\n""verify ALL files on DVD \n\n\n",boldfont);

      for (ii = 0; ii < Vnf; ii++)
      {
         if (checkKillPause()) goto verify_exit;

         dfile = Vrec[ii].file;                                            //  /home/.../file.ext
         track_filespec(dfile);                                            //  track progress on screen
         errmess = checkFile(dfile,0,dcc1);                                //  check file, get length
         if (errmess) track_filespec_err(dfile,errmess);                   //  log errors
         if (errmess) verrs++;
         vfiles++;
         vbytes += dcc1;

         if (verrs + cerrs > 100) {
            wprintx(mLog,0," *** OVER 100 ERRORS, GIVING UP *** \n",boldfont);
            goto verify_exit;
         }
      }
   }
   
   if (strEqu(menu,"incremental"))                                         //  verify files in prior incr. backup
   {
      wprintx(mLog,0,"\n""verify files in prior incremental backup \n",boldfont);

      for (ii = 0; ii < Dnf; ii++)
      {
         if (checkKillPause()) goto verify_exit;
         if (! Drec[ii].ivf) continue;                                     //  skip if not in prior incr. backup

         dfile = Drec[ii].file;
         wprintf(mLog,"  %s \n",kleenex(dfile));                           //  output filespec
         errmess = checkFile(dfile,0,dcc1);                                //  check file on DVD, get length
         if (errmess) wprintf(mLog,"  *** %s \n",errmess);
         if (errmess) verrs++;
         vfiles++;
         vbytes += dcc1;

         if (verrs + cerrs > 100) {
            wprintx(mLog,0," *** OVER 100 ERRORS, GIVING UP *** \n",boldfont);
            goto verify_exit;
         }
      }
   }
   
   if (strEqu(menu,"thorough"))                                            //  compare backup files to disk files
   {
      setFileDisps();                                                      //  file disps: new mod del unch

      wprintx(mLog,0,"\n""thorough verify ALL files on DVD \n",boldfont);
      wprintf(mLog," bytewise compare %d files and check %d others \n\n\n",nunc,nmod+ndel);
      
      for (ii = 0; ii < Vnf; ii++)                                         //  process DVD files
      {
         if (checkKillPause()) goto verify_exit;

         dfile = Vrec[ii].file;
         track_filespec(dfile);                                            //  track progress on screen

         if (Vrec[ii].disp == 'u') {                                       //  unchanged file
            errmess = checkFile(dfile,1,dcc1);                             //  compare disk and DVD files
            if (errmess) track_filespec_err(dfile,errmess);                //  log errors
         }
         else {                                                            //  modified or deleted file
            errmess = checkFile(dfile,0,dcc1);                             //  check DVD file
            if (errmess) track_filespec_err(dfile,errmess);                //  log errors
         }

         if (errmess) {
            if (strstr(errmess,"compare")) cerrs++;                        //  file compare error
            else  verrs++;
         }

         vfiles++;
         vbytes += dcc1;

         if (verrs + cerrs > 100) {
            wprintx(mLog,0," *** OVER 100 ERRORS, GIVING UP *** \n",boldfont);
            goto verify_exit;
         }
      }
   }

   wprintf(mLog," files: %d  bytes: %.0f \n",vfiles,vbytes);
   wprintf(mLog," file read errors: %d \n",verrs);
   if (strEqu(menu,"thorough")) wprintf(mLog," compare failures: %d \n",cerrs);

   secs = get_timer(&time0);
   wprintf(mLog," verify time: %.0f secs \n",secs);
   vspeed = vbytes/1000000.0/secs;
   wprintf(mLog," verify speed: %.2f MB/sec \n",vspeed);

   if (verrs + cerrs) wprintx(mLog,0," *** THERE WERE ERRORS *** \n",boldfont);
   else wprintf(mLog," NO ERRORS \n");

verify_exit:
   if (! Vnf) wprintf(mLog," *** no files on DVD \n");
   if (! Vnf) commFail++;
   if (verrs + cerrs) commFail++;
   wprintf(mLog," ready \n");
   return 0;
}


//  Reports menu function 

int Report(char * menu)
{
   if (strEqu((char *) menu, "get backup files")) get_backup_files(0);
   if (strEqu((char *) menu, "diffs summary")) report_summary_diffs(0);
   if (strEqu((char *) menu, "diffs by directory")) report_directory_diffs(0);
   if (strEqu((char *) menu, "diffs by file")) report_file_diffs(0);
   if (strEqu((char *) menu, "list backup files")) list_backup_files(0);
   if (strEqu((char *) menu, "list DVD files")) list_DVD_files(0);
   if (strEqu((char *) menu, "find files")) find_files(0);
   if (strEqu((char *) menu, "view backup hist")) view_backup_hist(0);
   return 0;
}


//  refresh backup files and report summary statistics per include/exclude statement

int get_backup_files(char *menu)
{
   char           chs;
   int            ii;

   dFilesReset();                                                          //  force refresh
   dGetFiles();                                                            //  get disk files

   if (! BJval) {
      wprintf(mLog," *** backup job is invalid \n");
      goto report_exit;
   }

   wprintx(mLog,0,"\n  files    Kbytes  DVD  include/exclude filespec \n",boldfont);

   for (ii = 0; ii < BJnx; ii++)                                           //  v.24
   {
      if (BJfiles[ii] > 0) wprintf(mLog," %6d %9.0f  %2d", BJfiles[ii], BJbytes[ii]/1000, BJdvdno[ii]);
      if (BJfiles[ii] < 0) wprintf(mLog," %6d %9.0f    ", BJfiles[ii], BJbytes[ii]/1000);
      if (BJfiles[ii] == 0) wprintf(mLog,"                     ");
      wprintf(mLog,"   %s \n",BJinex[ii]);
   }

   if (BJndvd > 1) chs = 's'; else chs = ' ';
   wprintf(mLog," %6d %9.0f       TOTAL  %.0f MB  %d DVD%c \n", 
                  Dnf, Dbytes/1000, Dbytes/1000000, BJndvd, chs);

report_exit:
   wprintf(mLog," ready \n");
   return 0;
}


//  report disk:DVD differences summary

int report_summary_diffs(char *menu)
{
   if (! BJval) {
      wprintf(mLog," *** backup job is invalid \n");
      goto report_exit;
   }

   dGetFiles();
   vGetFiles();
   setFileDisps();

   wprintf(mLog,"\n disk files: %d  DVD files: %d \n",Dnf,Vnf);
   wprintf(mLog,"\n Differences between DVD and files on disk: \n");
   wprintf(mLog,"  %d  files on disk, not on DVD (new files) \n",nnew);
   wprintf(mLog,"  %d  files with different data (modified files) \n",nmod);
   wprintf(mLog,"  %d  files on DVD, not on disk (deleted files) \n",ndel);
   wprintf(mLog,"  %d  files with identical data (unchanged files) \n",nunc);
   wprintf(mLog," Total differences: %d files  %.0f KB \n",nnew+ndel+nmod,Mbytes/1000);

report_exit:
   wprintf(mLog," ready \n");
   return 0;
}


//  report disk:DVD differences by directory, summary statistics

int report_directory_diffs(char *menu)
{
   int         kfiles, knew, kdel, kmod;
   int         dii, vii, comp;
   char        *pp, *pdirk, ppdirk[maxfcc];
   double      nbytes;

   if (! BJval) {
      wprintf(mLog," *** backup job is invalid \n");
      goto report_exit;
   }

   dGetFiles();
   vGetFiles();
   setFileDisps();

   SortFileList((char *) Drec, sizeof(dfrec), Dnf, 'D');                   //  re-sort, directories first
   SortFileList((char *) Vrec, sizeof(vfrec), Vnf, 'D');

   wprintf(mLog,"\n Disk:DVD differences by directory \n");

   wprintf(mLog,"   new   mod   del   KBytes  directory \n");
   
   nbytes = kfiles = knew = kmod = kdel = 0;
   dii = vii = 0;

   while ((dii < Dnf) || (vii < Vnf))                                      //  scan disk and DVD files in parallel
   {   
      if ((dii < Dnf) && (vii == Vnf)) comp = -1;
      else if ((dii == Dnf) && (vii < Vnf)) comp = +1;
      else comp = filecomp(Drec[dii].file, Vrec[vii].file);
      
      if (comp > 0) pdirk = Vrec[vii].file;                                //  get file on DVD or disk
      else pdirk = Drec[dii].file;

      pp = strrchr(pdirk,'/');                                             //  isolate directory
      if (pp) *pp = 0;
      if (strNeq(pdirk,ppdirk)) {                                          //  if directory changed, output
         if (kfiles > 0) wprintf(mLog," %5d %5d %5d %8.0f  %s \n",         //    totals from prior directory
                                 knew,kmod,kdel,nbytes/1000,ppdirk);
         nbytes = kfiles = knew = kmod = kdel = 0;                         //  reset totals
         strcpy(ppdirk,pdirk);                                             //  start new directory
      }
      if (pp) *pp = '/';

      if (comp < 0) {                                                      //  unmatched disk file: new
         knew++;                                                           //  count new files
         kfiles++;
         nbytes += Drec[dii].size;
         dii++;
      }

      else if (comp > 0) {                                                 //  unmatched DVD file: deleted
         kdel++;                                                           //  count deleted files
         kfiles++;
         vii++;
      }

      else if (comp == 0) {                                                //  file present on disk and DVD
         if (Drec[dii].disp == 'm') {
            kmod++;                                                        //  count modified files
            kfiles++;
            nbytes += Drec[dii].size;
         }
         dii++;
         vii++;
      }
   }

   if (kfiles > 0) wprintf(mLog," %5d %5d %5d %8.0f  %s \n",               //  totals from last directory
                           knew,kmod,kdel,nbytes/1000,ppdirk);

   SortFileList((char *) Drec, sizeof(dfrec), Dnf, 'A');                   //  restore ascii sort
   SortFileList((char *) Vrec, sizeof(vfrec), Vnf, 'A');

report_exit:
   wprintf(mLog," ready \n");
   return 0;
}


//  report disk:DVD differences by file (new, modified, deleted)

int report_file_diffs(char *menu)
{
   int      ii;

   if (! BJval) {
      wprintf(mLog," *** backup job is invalid \n");
      goto report_exit;
   }

   report_summary_diffs(0);                                                //  report summary first

   wprintf(mLog,"\n Detailed list of disk:DVD differences: \n");

   wprintf(mLog,"\n %d new files (on disk, not on DVD) \n",nnew);

   for (ii = 0; ii < Dnf; ii++) 
   {
      if (Drec[ii].disp != 'n') continue;
      wprintf(mLog,"  %s \n",kleenex(Drec[ii].file));
      if (checkKillPause()) break;
   }

   wprintf(mLog,"\n %d modified files (disk and DVD files are different) \n",nmod);

   for (ii = 0; ii < Dnf; ii++) 
   {
      if (Drec[ii].disp != 'm') continue;
      wprintf(mLog,"  %s \n",kleenex(Drec[ii].file));
      if (checkKillPause()) break;
   }

   wprintf(mLog,"\n %d deleted files (on DVD, not on disk) \n",ndel);

   for (ii = 0; ii < Vnf; ii++) 
   {
      if (Vrec[ii].disp != 'd') continue;
      wprintf(mLog,"  %s \n",kleenex(Vrec[ii].file));
      if (checkKillPause()) break;
   }

report_exit:
   wprintf(mLog," ready \n");
   return 0;
}


//  list all files in backup file set on disk

int list_backup_files(char *menu)
{
   int      ii;

   if (! BJval) {
      wprintf(mLog," *** backup job is invalid \n");
      goto report_exit;
   }

   wprintf(mLog,"\n List all files in backup file set: \n");

   dGetFiles();
   wprintf(mLog,"   %d files found \n",Dnf);

   for (ii = 0; ii < Dnf; ii++)
   {
      if (checkKillPause()) break;
      wprintf(mLog," %s \n",kleenex(Drec[ii].file));
   }

report_exit:
   wprintf(mLog," ready \n");
   return 0;
}


//  list all files on mounted DVD

int list_DVD_files(char *menu)
{
   int      ii;

   wprintf(mLog,"\n List all files on DVD: \n");

   vGetFiles();
   wprintf(mLog,"   %d files found \n",Vnf);

   for (ii = 0; ii < Vnf; ii++)
   {
      if (checkKillPause()) break;
      wprintf(mLog," %s \n",kleenex(Vrec[ii].file));
   }

   return 0;
}


//  find desired files on disk, on mounted DVD, and in history files

int find_files(char *menu)
{
   int            ii, ftf, nn;
   const char     *fspec1, *hfile1;
   static char    fspec2[200] = "/home/*/file*";
   char           hfile[200], buff[1000], *pp;
   FILE           *fid;
   pvlist         *flist = 0;

   dGetFiles();                                                            //  get disk and DVD files
   if (dvdmtd) vGetFiles();
   else wprintf(mLog," DVD not mounted \n");

   wprintf(mLog,"\n find files matching wildcard pattern \n");
   
   fspec1 = dialogText("enter (wildcard) filespec:",fspec2);               //  get search pattern
   if (is_blank_null(fspec1)) goto report_exit;
   strncpy0(fspec2,fspec1,199);
   strTrim(fspec2);
   wprintf(mLog," search pattern: %s \n",fspec2);

   wprintx(mLog,0,"\n matching files on disk: \n",boldfont);

   for (ii = 0; ii < Dnf; ii++)                                            //  search disk files
   {
      if (checkKillPause()) break;
      if (MatchWild(fspec2,Drec[ii].file) == 0) 
            wprintf(mLog,"  %s \n",kleenex(Drec[ii].file));
   }

   wprintx(mLog,0,"\n matching files on DVD: \n",boldfont);

   for (ii = 0; ii < Vnf; ii++)                                            //  search DVD files
   {
      if (checkKillPause()) break;
      if (MatchWild(fspec2,Vrec[ii].file) == 0) 
            wprintf(mLog,"  %s \n",kleenex(Vrec[ii].file));
   }
   
   wprintx(mLog,0,"\n matching files in backup history: \n",boldfont);     //  v.29
   
   flist = pvlist_create(maxhist);
   snprintf(hfile,199,"%s/dkop-hist-*",getz_appdirk());                    //  find all backup history files
   ftf = 1;                                                                //    /home/user/.dkop/dkop-hist-*
   nn = 0;

   while (true)
   {
      hfile1 = SearchWild(hfile,ftf);
      if (! hfile1) break;
      if (nn == maxhist) break;
      pvlist_append(flist,hfile1);                                         //  add to list
      nn++;
   }

   if (nn == 0) wprintf(mLog," no history files found \n");
   if (nn == maxhist) wprintf(mLog," *** too many history files, please purge");
   if (nn == 0 || nn == maxhist) goto report_exit;

   pvlist_sort(flist);                                                     //  sort list ascending
   
   for (ii = 0; ii < nn; ii++)                                             //  loop all history files
   {
      hfile1 = pvlist_get(flist,ii);
      wprintf(mLog,"  %s \n",hfile1);

      fid = fopen(hfile1,"r");                                             //  next history file
      if (! fid) {
         wprintf(mLog,"   *** file open error \n");
         continue;
      }

      while (true)                                                         //  read and search for match
      {
         pp = fgets_trim(buff,999,fid,1);
         if (! pp) break;
         if (MatchWild(fspec2,buff) == 0) 
               wprintf(mLog,"    %s \n",buff);
      }
      
      fclose(fid);
   }

report_exit:
   if (flist) pvlist_free(flist);
   wprintf(mLog," ready \n");
   return 0;
}


//  list available backup history files, select one to view                v.29

int view_backup_hist(char *menu)
{
   const char     *fspec1;
   char           fspec2[200], histfile[200], command[200];
   char           *pp;
   int            ii, jj, nn;
   int            zstat, ftf;
   zdialog        *zd;
   pvlist         *flist = 0;

   wprintf(mLog," available history files in %s \n",getz_appdirk());

   snprintf(fspec2,199,"%s/dkop-hist-*",getz_appdirk());
   flist = pvlist_create(maxhist);
   ftf = 1;
   nn = 0;

   while (true)
   {
      fspec1 = SearchWild(fspec2,ftf);                                     //  file: dkop-hist-yyyymmdd-hhmm-label
      if (! fspec1) break;
      pp = strrchr(fspec1,'/') + 11;                                       //  get yyyymmdd-hhmm-label
      if (nn == maxhist) break;
      pvlist_append(flist,pp);                                             //  add to list
      nn++;
   }

   if (nn == 0) wprintf(mLog," no history files found \n");
   if (nn == maxhist) wprintf(mLog," *** too many history files, please purge");
   if (nn == 0 || nn == maxhist) goto report_exit;
   
   pvlist_sort(flist);                                                     //  sort list ascending
   
   for (ii = 0; ii < nn; ii++)                                             //  report sorted list
      wprintf(mLog," dkop-hist-%s \n",pvlist_get(flist,ii));

   zd = zdialog_new("choose history file","OK","cancel",0);
   zdialog_add_widget(zd,"label","lab1","dialog","history file date and DVD label");
   zdialog_add_widget(zd,"combo","hfile","dialog");

   jj = nn - 20;
   if (jj < 0) jj = 0;
   for (ii = jj; ii < nn; ii++)                                            //  stuff combo box list with
      zdialog_cb_app(zd,"hfile",pvlist_get(flist,ii));                     //    20 newest hist file IDs
   zdialog_stuff(zd,"hfile",pvlist_get(flist,nn-1));                       //  default entry is newest file
   
   zstat = zdialog_run(zd);                                                //  run dialog
   zdialog_fetch(zd,"hfile",histfile,199);                                 //  get user choice
   zdialog_destroy(zd);
   zdialog_free(zd);
   if (zstat != 1) goto report_exit;                                       //  cancelled

   snprintf(command,199,"%s %s/%s-%s",showfile,                            //  view the file
               getz_appdirk(),"/dkop-hist",histfile);
   system(command);

report_exit:   
   if (flist) pvlist_free(flist);
   wprintf(mLog," ready \n");
   return 0;
}


//  file restore dialog - specify DVD files to be restored

int RJedit(char * menu)
{
   int            ii;
   zdialog        *zd;
   
   wprintf(mLog,"\n Restore files from DVD \n");   

   vGetFiles();                                                            //  get files on DVD
   wprintf(mLog,"   %d files on DVD \n",Vnf);
   if (! Vnf) return 0;

   ++Fdialog;
   
   zd = zdialog_new("copy files from DVD","browse","done","cancel",0);
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=10");              //  v.28
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"label","labdev","vb1","DVD device");             //   DVD device      [___________][v]
   zdialog_add_widget(zd,"label","labmp","vb1","mount point");             //   mount point     [___________][v]
   zdialog_add_widget(zd,"combo","entdvd","vb2",BJdvd);
   zdialog_add_widget(zd,"combo","entmp","vb2",BJmp);

   zdialog_add_widget(zd,"label","labfrom","vb1","copy-from DVD");         //  copy-from DVD    [______________]
   zdialog_add_widget(zd,"label","labto","vb1","copy-to disk");            //  copy-to disk     [______________]
   zdialog_add_widget(zd,"entry","entfrom","vb2",RJfrom);
   zdialog_add_widget(zd,"entry","entto","vb2",RJto);

   zdialog_add_widget(zd,"hsep","hsep1","dialog");
   zdialog_add_widget(zd,"label","labfiles","dialog","files to restore");
   zdialog_add_widget(zd,"frame","framefiles","dialog",0,"expand");
   zdialog_add_widget(zd,"scrwin","scrfiles","framefiles");
   zdialog_add_widget(zd,"edit","editfiles","scrfiles");

   for (ii = 0; ii < ndvds; ii++)                                          //  load curr. data into widgets
         zdialog_cb_app(zd,"entdvd",dvddevdesc[ii]);
   for (ii = 0; ii < nmps; ii++) 
         zdialog_cb_app(zd,"entmp",dvdmps[ii]);

   editwidget = zdialog_widget(zd,"editfiles");
   for (int ii = 0; ii < RJnx; ii++)                                       //  get restore include/exclude recs,
      wprintf(editwidget,"%s""\n",RJinex[ii]);                             //   pack into file selection edit box

   zdialog_resize(zd,400,400);

   int RJedit_compl(zdialog *zd, int zstat);
   zdialog_run(zd,0,RJedit_compl);                                         //  run dialog with response function
   return 0;
}


//  restore dialog completion function

int RJedit_compl(zdialog *zd, int zstat)
{
   char           text[40], *pp, fcfrom[maxfcc];
   int            ftf, cc;

   if (zstat != 1 && zstat != 2) goto end_dialog;                          //  cancel or destroy
   
   RJreset();                                                              //  reset restore job data

   zdialog_fetch(zd,"entdvd",text,19);                                     //  get DVD device
   strncpy0(BJdvd,text,19);
   pp = strchr(BJdvd,' ');
   if (pp) *pp = 0;

   zdialog_fetch(zd,"entmp",text,39);                                      //  DVD mount point
   strncpy0(BJmp,text,39);
   strTrim(BJmp);
   BJmpcc = strlen(BJmp);
   if (BJmpcc && (BJmp[BJmpcc-1] == '/')) BJmp[BJmpcc--] = 0;              //  remove trailing /

   zdialog_fetch(zd,"entfrom",RJfrom,maxfcc);                              //  copy-from location /home/xxx/.../
   strTrim(RJfrom);

   zdialog_fetch(zd,"entto",RJto,maxfcc);                                  //  copy-to location  /home/yyy/.../
   strTrim(RJto);

   ftf = 1;
   while (true)                                                            //  include/exclude recs from edit box
   {
      pp = wscanf(editwidget,ftf);
      if (! pp) break;
      cc = strTrim(pp);                                                    //  remove trailing blanks
      if (cc < 3) continue;                                                //  ignore absurdities
      if (cc > maxfcc-100) continue;
      RJinex[RJnx] = zmalloc(cc+1);                                        //  allocate memory
      strcpy(RJinex[RJnx],pp);                                             //  copy new record
      if (++RJnx == maxnx) {
         wprintf(mLog," *** exceed %d include/exclude recs \n",maxnx); 
         break;
      }
   }

   if (zstat == 1) {                                                       //  do file-chooser dialog
      strcpy(fcfrom,BJmp);                                                 //  start at /media/dvd/home/xxx/
      strcat(fcfrom,RJfrom);
      fc_dialog(fcfrom);
      return 0;                                                            //  dialog continues
   }

   RJvalidate();                                                           //  validate restore job data
   if (RJval) rGetFiles();                                                 //  get files to restore
   else wprintf(mLog," *** correct errors in restore job \n");

end_dialog:
   zdialog_destroy(zd);                                                    //  destroy dialog
   --Fdialog;
   return 0;
}


//  thread function, list and validate DVD files to be restored

int RJlist(char * menu)
{
   int       cc1, cc2;
   char     *file1, file2[maxfcc];
   
   if (! RJval) {
      wprintf(mLog," *** restore job has errors \n");
      goto list_exit;
   }

   wprintf(mLog,"\n copy %d files from DVD: %s \n",Rnf, RJfrom);
   wprintf(mLog,"    to directory: %s \n",RJto);
   wprintf(mLog,"\n resulting files will be the following: \n");
   if (! Rnf) goto list_exit;
   
   cc1 = strlen(RJfrom);                                                   //  from: /home/xxx/.../
   cc2 = strlen(RJto);                                                     //    to: /home/yyy/.../

   for (int ii = 0; ii < Rnf; ii++)
   {
      if (checkKillPause()) break;

      file1 = Rrec[ii].file;

      if (! strnEqu(file1,RJfrom,cc1)) {
         wprintf(mLog," *** not within copy-from: %s \n",kleenex(file1));
         RJval = 0;
         continue;
      }
      
      strcpy(file2,RJto);
      strcpy(file2+cc2,file1+cc1);
      wprintf(mLog," %s \n",kleenex(file2));
   }

list_exit:      
   wprintf(mLog," ready \n");
   return 0;
}


//  thread function, restore files based on data from restore dialog

int Restore(char * menu)
{
   int         ii, nn, ccf;
   char        dfile[maxfcc], *errmess;

   if (! RJval || ! Rnf) {
      wprintf(mLog," *** restore job has errors \n");
      goto restore_exit;
   }

   nn = zmessageYN("Restore %d files from: %s%s \n     to: %s \n"
                   "Proceed with file restore ?",Rnf,BJmp,RJfrom,RJto);
   if (! nn) goto restore_exit;

   snprintf(dfile,maxfcc-1,"\n""begin restore of %d files to: %s \n",Rnf,RJto);
   wprintx(mLog,0,dfile,boldfont);

   ccf = strlen(RJfrom);                                                   //  from: /media/xxx/filespec

   for (ii = 0; ii < Rnf; ii++)
   {
      if (checkKillPause()) goto restore_exit;
      strcpy(dfile,RJto);                                                  //  to: /destination/filespec
      strcat(dfile,Rrec[ii].file+ccf);
      wprintf(mLog," %s \n",kleenex(dfile));
      errmess = copyFile(Rrec[ii].file,dfile);                             //  v.23
      if (errmess) wprintf(mLog," *** %s \n",errmess);
   }

   restore_filepoop();                                                     //  restore owner/permissions

   dFilesReset();                                                          //  reset disk file data

restore_exit:   
   wprintf(mLog," ready \n");
   return 0;
}


//  get available DVD devices and mount points

int getDVDs()
{
   int      err, eof, ii, dvdrw = 0;
   char     buff[100], *pp;

   ndvds = nmps = 0;

   strcpy(dvddevs[0],"/dev/dvd");                                          //  prepare default DVD, mount point
   strcpy(dvddesc[0],"default DVD");
   strcpy(dvddevdesc[0],"/dev/dvd  default DVD");
   strcpy(dvdmps[0],"/media/dvd");

   err = createProc("udevinfo -e");                                        //  get hardware disk info
   if (err) {
      zmessageACK("unable to execute udevinfo command");
      ndvds = 1;
      goto get_mps;
   }

   while (true)
   {
      eof = getProcOutput(buff,99,err);                                    //  read udevinfo output
      if (eof) break;

      if (strnEqu(buff,"P:",2)) {                                          //  start next device
         dvdrw = 0;                                                        //  is DVDRW not known
         if (ndvds == maxdvds) continue;
         pp = strstr(buff,"/block/");
         if (! pp) continue;                                               //  not a disk
         strcpy(dvddevs[ndvds],"/dev/");
         strncat(dvddevs[ndvds],pp+7,14);                                  //  prepare /dev/devID
         strcpy(dvddesc[ndvds],"?");                                       //  description not known
      }
         
      if (strnEqu(buff,"S:",2)) {
         pp = strstr(buff,"dvdrw");
         if (! pp) continue;
         if (ndvds == maxdvds) continue;
         dvdrw = 1;                                                        //  device is a DVDRW
         ndvds++;                                                          //  now we can count it
      }
         
      if (strnEqu(buff,"E:",2)) {
         if (! dvdrw) continue;
         pp = strstr(buff,"ID_MODEL=");
         if (! pp) continue;
         strncpy0(dvddesc[ndvds-1],pp+9,40);                               //  description
      }
   }
   
   if (ndvds == 0) ndvds = 1;                                              //  if none found, leave a default
   
   for (ii = 0; ii < ndvds; ii++)                                          //  combine devices and descriptions
   {                                                                       //    for use in GUI chooser list
      strcpy(dvddevdesc[ii],dvddevs[ii]);
      strcat(dvddevdesc[ii],"  ");
      strcat(dvddevdesc[ii],dvddesc[ii]);
   }

get_mps:   

   err = createProc("ls --format=single-column /media");                   //  get /media/xxx mount points
   if (err) {
      zmessageACK("unable to execute ls --format command");
      nmps = 1;
      return 0;
   }

   while (true)
   {
      eof = getProcOutput(buff,99,err);                                    //  read ls command output
      if (eof) return 0;                                                   //  error or EOF
      if (nmps < maxmps) {
         strcpy(dvdmps[nmps],"/media/");
         strncat(dvdmps[nmps],buff,32);                                    //  add /media/xxx to list
         nmps++;
      }
   }
}


//  change DVD device and mount point

int setDVDdevice(char *menu)                                               //  v.15
{
   const char     *pp1;
   char           *pp2, text[60];
   int            ii, Nth, zstat;
   zdialog        *zd;

   if (*scriptParam) {                                                     //  script
      Nth = 1;                                                             //  parse: /dev/dvd /media/dvd
      pp1 = strField(scriptParam,' ',Nth++);
      if (pp1) strncpy0(BJdvd,pp1,19);
      pp1 = strField(scriptParam,' ',Nth++);
      if (pp1) strncpy0(BJmp,pp1,39);
      *scriptParam = 0;
      return 0;
   }
   
   zd = zdialog_new("select DVD drive","OK","cancel",0);                   //  dialog to select DVD and mount point
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");               //  v.28
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"label","labdvd","vb1","DVD device");
   zdialog_add_widget(zd,"label","labmp","vb1","mount point");
   zdialog_add_widget(zd,"combo","entdvd","vb2",BJdvd);
   zdialog_add_widget(zd,"combo","entmp","vb2",BJmp);

   for (ii = 0; ii < ndvds; ii++)                                          //  stuff avail. DVDs, mount points
         zdialog_cb_app(zd,"entdvd",dvddevdesc[ii]);
   for (ii = 0; ii < nmps; ii++) 
         zdialog_cb_app(zd,"entmp",dvdmps[ii]);
   
   zstat = zdialog_run(zd);
   zdialog_destroy(zd);
   if (zstat != 1) return 0;
   
   zstat = zdialog_fetch(zd,"entdvd",text,60);                             //  get selected DVD
   strncpy0(BJdvd,text,19);
   pp2 = strchr(BJdvd,' ');
   if (pp2) *pp2 = 0;

   zstat = zdialog_fetch(zd,"entmp",text,60);                              //  get selected mount point
   strncpy0(BJmp,text,39);
   pp2 = strchr(BJmp,' ');
   if (pp2) *pp2 = 0;
   BJmpcc = strlen(BJmp);
   if (BJmpcc && (BJmp[BJmpcc-1] == '/')) BJmp[BJmpcc--] = 0;              //  remove trailing /

   wprintf(mLog," DVD and mount point: %s %s \n",BJdvd,BJmp);
   wprintf(mLog," (not mounted) \n");                                      //  v.20
   wprintf(mLog," ready \n");
   return 0;
}


//  set label for next DVD backup via growisofs                            v.30

int setDVDlabel(char *menu)
{
   const char     *pp;

   if (*oldlabel) wprintf(mLog," DVD label: %s \n",oldlabel);
   pp = dialogText("set DVD label for next backup",oldlabel);
   if (is_blank_null(pp)) pp = "dkop";
   strncpy0(newlabel,pp,28);                                               //  leave room for :A appendage
   wprintf(mLog," new DVD label for next backup: %s \n",newlabel);
   return 1;
}


//  mount DVD with message feedback to window
//  note: Linux is pretty stupid about mounting DVDs, and sometimes produces the wrong status, 
//  e.g. EBUSY for a blank medium. will make changes if or when Linux straightens itself out.

int mountDVD(char *menu)                                                   //  menu mount function    v.12
{
   mountDVDn(1);
   wprintf(mLog," ready \n");
   return 0;
}

int mountDVDn(int ntry)                                                    //  internal mount function   v.12
{
   int            err;
   char           command[100], mbuff[100], *pp = 0;
   FILE           *fid;
   struct stat    dstat;
   
   if (dvdmtd) {
      err = stat(BJmp,&dstat);
      if ((! err) && (dvdtime == dstat.st_ctime)) return 0;                //  medium unchanged, do nothing  v.08
   }

   dvdtime = -1;
   dvdmtd = 0;                                                             //  set DVD not mounted
   strcpy(mediumDT,"unknown");
   mediumNB = 0;
   *mediumDT = 0;
   *oldlabel = 0;
   vFilesReset();                                                          //  reset DVD files

   while (true)                                                            //  mount DVD with retries  v.12
   {
      sprintf(mbuff,"mount %s %s",BJdvd,BJmp);                             //  back to using bash   v.27
      err = do_shell("mount",mbuff);
      if (! err) break;
      if (err == EPERM) goto done;                                         //  no permission (need root)
      if (err == 32) wprintf(mLog," blank or missing disk? \n");
      if (--ntry == 0) goto done;
      if (checkKillPause()) goto done;
      wprintf(mLog," retrying mount ...");
      sleep(3);                                                            //  compensate flakey driver
   }

   err = stat(BJmp,&dstat);                                                //  really mounted?
   if (err) goto done;                                                     //  no

   dvdmtd = 1;                                                             //  DVD is mounted
   dvdtime = dstat.st_ctime;                                               //  set DVD ID = mod time

   snprintf(command,99,"volname %s",BJdvd);                                //  get DVD label   v.30
   fid = popen(command,"r");
   if (fid) {
      pp = fgets_trim(mbuff,99,fid,1);
      if (pp) strncpy0(oldlabel,pp,31);
      pclose(fid);
   }

   strcpy(mbuff,BJmp);
   strcat(mbuff,V_DATETIME);                                               //  get medium usage stats if poss.
   fid = fopen(mbuff,"r");
   if (fid) {
      pp = fgets_trim(mbuff,99,fid,1);
      if (pp) strncpy0(mediumDT,pp,15);
      pp = fgets_trim(mbuff,99,fid,1);
      pp = strstr(mbuff,"count:");
      if (pp) mediumNB = atoi(pp+6);
      if (mediumNB < 0) mediumNB = 0;
      fclose(fid);
   }

   wprintf(mLog," DVD label: %s  dkop usage: %d  last: %s \n",oldlabel,mediumNB,mediumDT);

done:
   if (! dvdmtd) commFail++;
   return err;
}


//  eject DVD with message feedback to window

int ejectDVD(char * menu)
{
   int      err;
   char     command[60];
   
   vFilesReset();
   dvdmtd = 0;
   dvdtime = -1;
                                                                           //  remove unmount (makes eject fail)
   sprintf(command,"eject %s 2>&1",BJdvd);
   err = do_shell("eject",command);
   wprintf(mLog," ready \n");
   return 0;
}


//  wait for DVD and reset hardware (get over lockups after growisofs)

int resetDVD(char * menu)                                                  //  v.12
{
   int      dvdfd, err = 0;
   char     command[60];

   vFilesReset();
   dvdmtd = 0;
   dvdtime = -1;

   commFail++;                                                             //  whatever is running, fail

   if (*subprocName) {                                                     //  try to kill it
      signalProc(subprocName,"resume");
      signalProc(subprocName,"kill");
      sleep(1);
   }

   dvdfd = open(BJdvd,O_RDONLY);
   if (dvdfd == -1) {
      wprintf(mLog," *** DVD device open failure \n");
      err = 1;
   }
   else {
      err = ioctl(dvdfd,CDROMRESET);
      close(dvdfd);
      if (err) wprintf(mLog," *** DVD reset err: %s \n",syserrText());
   }
   
   if (err) {
      sprintf(command,"wodim -abort dev=%s 2>&1",BJdvd);                   //  v.21  new tool avail.
      err = do_shell("wodim",command);
   }

   wprintf(mLog," ready \n");
   return 0;
}


//  thread function, erase DVD medium by filling it with zeros

int eraseDVD(char * menu)
{
   char        command[200];
   int         nstat;
   
   nstat = zmessageYN("Erase DVD. This will take some time. \n Continue?");
   if (! nstat) goto erase_exit;

   vFilesReset();                                                          //  reset DVD file data

   sprintf(command,"growisofs -Z %s=/dev/zero %s 2>&1",BJdvd,gforce);
   do_shell("growisofs", command);                                         //  do growisofs, echo outputs

erase_exit:
   wprintf(mLog," ready \n");
   return 0;
}


//  thread function, format DVD (2-4 minutes)

int formatDVD(char * menu)                                                 //  v.12
{
   char        command[60];
   int         nstat;
   
   nstat = zmessageYN("Format DVD. This will take 2-4 minutes. \n Continue?");
   if (! nstat) goto format_exit;

   vFilesReset();                                                          //  reset DVD file data
   resetDVD(null);                                                         //  tray in

   sprintf(command,"dvd+rw-format -force %s 2>&1",BJdvd);
   do_shell("dvd+rw-format", command);

format_exit:
   wprintf(mLog," ready \n");
   return 0;
}


//  save logging window as text file

int saveScreen(char * menu)
{
   if (*scriptParam) {                                                     //  v.11
      wfiledump(mLog, scriptParam);
      *scriptParam = 0;
      return 0;
   }
   
   wfilesave(mLog);
   return 0;
}


//  thread function to display help/about or help/contents

int helpFunc(char * menu)
{
   if (strEqu((char *) menu,"about")) {
      wprintf(mLog," %s \n",dkop_title);
      wprintf(mLog," free software: %s \n",dkop_license);
   }

   if (strEqu((char *) menu,"contents")) showz_helpfile();                 //  help file in new process   v.27

   return 0;
}


//  construct file-chooser dialog box 
//  note: Fdialog unnecessary: this dialog called from other dialogs

int fc_dialog(char * dirk)
{
   fc_dialogbox = gtk_dialog_new_with_buttons("choose files", 
                  GTK_WINDOW(mWin), GTK_DIALOG_MODAL, "hidden",100, 
                  "include",101, "exclude",102, "done",103, null);

   gtk_window_set_default_size(GTK_WINDOW(fc_dialogbox),600,500);
   G_SIGNAL(fc_dialogbox,"response",fc_response,0)

   fc_widget = gtk_file_chooser_widget_new(GTK_FILE_CHOOSER_ACTION_OPEN);
   gtk_container_add(GTK_CONTAINER(GTK_DIALOG(fc_dialogbox)->vbox),fc_widget);

   gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(fc_widget),dirk);
   gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(fc_widget),1);
   gtk_file_chooser_set_show_hidden(GTK_FILE_CHOOSER(fc_widget),0);

   gtk_widget_show_all(fc_dialogbox);
   return 0;
}


//  file-chooser dialog handler (file selection, OK, Cancel, Kill)

int fc_response(GtkDialog *dwin, int arg, void *data)
{
   GtkTextBuffer  *textBuff;
   GSList         *flist = 0;
   char           *file1, *file2, *ppf;
   char           *errmess;
   int             ii, err, hide;
   struct stat64   filestat;
   
   if (arg == 103 || arg == -4)                                            //  done, cancel
   {
      gtk_widget_destroy(GTK_WIDGET(dwin));
      return 0;
   }
   
   if (arg == 100)                                                         //  hidden
   {
      hide = gtk_file_chooser_get_show_hidden(GTK_FILE_CHOOSER(fc_widget));
      hide = 1 - hide;
      gtk_file_chooser_set_show_hidden(GTK_FILE_CHOOSER(fc_widget),hide);
   }
   
   if (arg == 101 || arg == 102)                                           //  include, exclude
   {
      flist = gtk_file_chooser_get_filenames(GTK_FILE_CHOOSER(fc_widget));

      for (ii = 0; ; ii++)                                                 //  process selected files
      {
         file1 = (char *) g_slist_nth_data(flist,ii);
         if (! file1) break;

         file2 = zmalloc(strlen(file1)+4);
         strcpy(file2,file1);
         g_free(file1);

         err = stat64(file2,&filestat);
         if (err) {
            errmess = syserrText();
            wprintf(mLog," *** error: %s  file: %s \n",errmess,kleenex(file2));
         }
         
         if (S_ISDIR(filestat.st_mode)) strcat(file2,"/*");                //  if directory, append wildcard
         
         ppf = file2;
         if (strnEqu(ppf,BJmp,BJmpcc)) ppf += BJmpcc;                      //  omit DVD mount point   v.19

         textBuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(editwidget));
         if (arg == 101) wprintf(editwidget,"include %s""\n",ppf);
         if (arg == 102) wprintf(editwidget,"exclude %s""\n",ppf);

         zfree(file2);                                                     //  v.19 leak
      }
   }

   gtk_file_chooser_unselect_all(GTK_FILE_CHOOSER(fc_widget));
   g_slist_free(flist);
   return 0;
}


//  backup helper function
//  set nominal backup date/time
//  write date/time and updated medium use count to temp file

int writeDT()
{
   time_t      dt1;
   struct tm   dt2;                                                        //  year/month/day/hour/min/sec
   FILE        *fid;

   dt1 = time(0);
   dt2 = *localtime(&dt1);

   snprintf(backupDT,15,"%4d%02d%02d-%02d%02d",dt2.tm_year+1900,           //  yyyymmdd-hhmm
            dt2.tm_mon+1, dt2.tm_mday, dt2.tm_hour, dt2.tm_min);
   
   strcpy(mediumDT,backupDT);
   mediumNB++;                                                             //  incr. medium usage count

   fid = fopen(TFdatetime,"w");
   if (! fid) {
      wprintf(mLog," *** cannot open /tmp scratch file \n");
      commFail++;
      return 0;
   }

   fprintf(fid,"%s \n",mediumDT);                                          //  write date/time and medium count
   fprintf(fid,"usage count: %d \n",mediumNB);                             //    to temp file
   fclose(fid);
   return 0;
}


//  backup helper function
//  save all file and directory owner and permission data to temp file

int save_filepoop()                                                        //  all files, not just directories
{
   int            ii, cc, err;
   FILE           *fid;
   char           file[maxfcc], dirk[maxfcc], pdirk[maxfcc];
   char           *pp, *errmess;
   struct stat64  dstat;

   fid = fopen(TFfilepoop,"w");
   if (! fid) {
      wprintf(mLog," *** cannot open /tmp scratch file \n");
      commFail++;
      return 0;
   }

   *pdirk = 0;                                                             //  no prior
   
   for (ii = 0; ii < Dnf; ii++)
   {
      strcpy(dirk,Drec[ii].file);                                          //  next file on disk
      pp = dirk;

      while (true)
      {
         pp = strchr(pp+1,'/');                                            //  next (last) directory level
         if (! pp) break;
         cc = pp - dirk + 1;                                               //  cc incl. '/'
         if (strncmp(dirk,pdirk,cc) == 0) continue;                        //  matches prior, skip
         
         *pp = 0;                                                          //  terminate this directory level

         err = stat64(dirk,&dstat);                                        //  get owner and permissions
         if (err) {
            errmess = syserrText();
            wprintf(mLog," *** error: %s  file: %s \n",errmess,kleenex(dirk));
            break;
         }

         dstat.st_mode = dstat.st_mode & 0777;

         fprintf(fid,"%4d:%4d %3o %s\n",                                   //  output uid:gid permissions directory
              dstat.st_uid, dstat.st_gid, dstat.st_mode, dirk);            //                  (octal)
         
         *pp = '/';                                                        //  restore '/'
      }
      
      strcpy(pdirk,dirk);                                                  //  prior = this directory
      
      strcpy(file,Drec[ii].file);                                          //  disk file, again

      err = stat64(file,&dstat);                                           //  get owner and permissions
      if (err) {
         errmess = syserrText();
         wprintf(mLog," *** error: %s  file: %s \n",errmess,kleenex(file));
         continue;
      }

      dstat.st_mode = dstat.st_mode & 0777;

      fprintf(fid,"%4d:%4d %3o %s\n",                                      //  output uid:gid permissions file
            dstat.st_uid, dstat.st_gid, dstat.st_mode, file);              //                  (octal)
   }

   fclose(fid);
   return 0;
}


//  restore helper function
//  restore original owner and permissions for restored files and directories

int restore_filepoop()                                                     //  v.23 
{
   FILE        *fid;
   int         cc1, cc2, ccf, nn, ii, err;
   int         uid, gid, perms;
   char        file1[maxfcc], file2[maxfcc];
   char        poopfile[100];
   
   wprintf(mLog,"\n restore directory owner and permissions \n");
   wprintf(mLog,"  for directories anchored at: %s \n",RJto);
   
   cc1 = strlen(RJfrom);                                                   //  from: /home/xxx/.../
   cc2 = strlen(RJto);                                                     //    to: /home/yyy/.../

   strcpy(poopfile,BJmp);                                                  //  DVD file with owner & permissions
   strcat(poopfile,V_FILEPOOP);

   fid = fopen(poopfile,"r");
   if (! fid) {
      wprintf(mLog," *** cannot open DVD file: %s \n",poopfile);
      return 0;
   }
   
   ii = 0;

   while (true)
   {
      nn = fscanf(fid,"%d:%d %o %[^\n]",&uid,&gid,&perms,file1);           //  uid, gid, permissions, file
      if (nn == EOF) break;                                                //  (nnn:nnn)   (octal)
      if (nn != 4) continue;

      ccf = strlen(file1);                                                 //  match directories too
      if (ccf < cc1) continue;

      while (ii < Rnf)                                                     //  v.02
      {
         nn = strncmp(Rrec[ii].file,file1,ccf);                            //  file in restored file list?
         if (nn >= 0) break;                                               //  (logic depends on sorted lists)
         ii++;
      }

      if (ii == Rnf) break;
      if (nn > 0) continue;                                                //  no

      strcpy(file2,RJto);                                                  //  copy-to location
      strcpy(file2 + cc2, file1 + cc1);                                    //  + org. file, less copy-from part
      wprintf(mLog," owner: %4d:%4d  permissions: %3o  file: %s \n",
                                    uid, gid, perms, kleenex(file2));
      err = chown(file2,uid,gid);
      if (err) wprintf(mLog," *** error: %s \n",syserrText());
      err = chmod(file2,perms);
      if (err) wprintf(mLog," *** error: %s \n",syserrText());
   }
   
   fclose(fid);
   return 0;
}


//  create backup history file after successful backup                     v.29

int createBackupHist()
{
   int         ii, err;
   FILE        *fid;
   char        backupfile[200], buff[230];
   char        disp;

   snprintf(backupfile,199,"%s/dkop-hist-%s-%s",                           //  create history file name:
               getz_appdirk(),backupDT,newlabel);                          //    dkop-hist-yyyymmdd-hhmm-dvdlabel
   
   snprintf(buff,229,"\n""create history file: %s \n",backupfile);
   wprintx(mLog,0,buff,boldfont);

   fid = fopen(backupfile,"w");
   if (! fid) {
      wprintf(mLog," *** cannot open dkop-hist file \n");
      return 0;
   }
   
   fprintf(fid,"%s (%s backup) \n\n",backupfile,mbmode);

   for (ii = 0; ii < BJnx; ii++)                                           //  output include/exclude recs   v.31
      fprintf(fid," %s \n",BJinex[ii]);
   fprintf(fid,"\n");
   
   if (strEqu(mbmode,"full"))
   {
      for (ii = 0; ii < Dnf; ii++)                                         //  output all files in backup set
      {
         if (Drec[ii].dvd != dvdnum) continue;                             //  screen for DVD number
         fprintf(fid,"%s\n",Drec[ii].file);                                        
      }
   }
   
   else {
      for (ii = 0; ii < Dnf; ii++) {                                       //  output new and modified disk files
         disp = Drec[ii].disp;
         if ((disp == 'n') || (disp == 'm')) 
               fprintf(fid,"%s\n",Drec[ii].file);
      }
   }

   err = fclose(fid);
   if (err) wprintf(mLog," *** dkop-hist file error %s \n",syserrText());
   return 0;
}


//  parse an include/exclude filespec statement
//  return: 0=comment  1=OK  2=parse-error  3=fspec-error

int inexParse(char * rec, char *& rtype, char *& fspec)
{
   char    *pp1, *pp2;
   int      ii;

   rtype = fspec = 0;
   
   if (rec[0] == '#') return 0;                                            //  comment recs.  v.31
   if (strlen(rec) < 3) return 0;
   strTrim(rec);
   
   ii = 0;
   while ((rec[ii] == ' ') && (ii < 30)) ii++;                             //  find 1st non-blank
   if (rec[ii] == 0) return 0;
   if (ii == 30) return 0;                                                 //  blank record
   
   rtype = rec + ii;                                                       //  include/exclude

   while ((rec[ii] > ' ') && (ii < 30)) ii++;                              //  look for next blank or null
   if (ii == 30) return 2;
   
   if (rec[ii] == ' ') { rec[ii] = 0; ii++; }                              //  end of rtype
   if (strlen(rtype) > 7) return 2;

   while ((rec[ii] == ' ') && (ii < 30)) ii++;                             //  find next non-blank
   if (ii == 30) return 2;

   fspec = rec + ii;                                                       //  filespec (wildcards)
   if (strlen(fspec) < 4) return 3;
   if (strlen(fspec) > maxfcc-100) return 3;

   if (strEqu(rtype,"exclude")) return 1;                                  //  exclude, done
   if (strNeq(rtype,"include")) return 2;                                  //  must be include

   if (fspec[0] != '/') return 3;                                          //  must have at least /topdirk/
   pp1 = strchr(fspec+1,'/');
   if (!pp1) return 3;
   if (pp1-fspec < 2) return 3;
   pp2 = strchr(fspec+1,'*');                                              //  any wildcards must be later
   if (pp2 && (pp2 < pp1)) return 3;
   pp2 = strchr(fspec+1,'%');
   if (pp2 && (pp2 < pp1)) return 3;
   return 1;                                                               //  include + legit. fspec
}


//  list backup job data and validate as much as practical

int BJvalidate(char * menu)
{
   int            err, nerr = 0;
   struct stat    dstat;
   
   wprintx(mLog,0,"\n""Validate backup job data \n",boldfont);

   BJval = 0;

   if (! BJnx) {
      wprintf(mLog," *** no job data present \n");
      commFail++;
      return 0;
   }

   wprintf(mLog," DVD device: %s \n",BJdvd);
   wprintf(mLog," mount point: %s \n",BJmp);
   wprintf(mLog," capacity GB: %.1f \n",BJcap);
   
   err = stat(BJdvd,&dstat);
   if (err || ! S_ISBLK(dstat.st_mode)) {
      wprintf(mLog," *** %s is apparently invalid \n",BJdvd);
      nerr++;
   }
   
   err = stat(BJmp,&dstat);
   if (err || ! S_ISDIR(dstat.st_mode)) {
      wprintf(mLog," *** %s is apparently invalid \n",BJmp);
      nerr++;
   }

   if (BJcap < dvdcapmin || BJcap > dvdcapmax) {
      wprintf(mLog," *** DVD medium capacity: %.1f GB \n",BJcap);
      wprintf(mLog,"     outside range %.1f to %.1f \n",dvdcapmin,dvdcapmax);
      nerr++;
   }

   wprintf(mLog," backup %s \n",BJbmode);
   if (! strcmpv(BJbmode,"full","incremental","accumulate",0)) {
      wprintf(mLog," *** backup mode not full/incremental/accumulate \n");
      nerr++;
   }
   
   wprintf(mLog," verify %s \n",BJvmode);
   if (! strcmpv(BJvmode,"full","incremental","thorough",0)) {
      wprintf(mLog," *** verify mode not full/incremental/thorough \n");
      nerr++;
   }
   
   nerr += nxValidate(BJinex,BJnx);                                        //  validate include/exclude recs

   wprintf(mLog," *** %d errors \n",nerr);
   if (nerr) commFail++;
   else BJval = 1;
   return 0;
}


//  validate restore job data

int RJvalidate()
{
   int      cc, nerr = 0;
   char     rdirk[maxfcc];
   DIR      *pdirk;

   if (RJval) return 1;   

   wprintf(mLog,"\n Validate restore job data \n");

   if (! RJnx) {
      wprintf(mLog," *** no job data present \n");
      return 0;
   }

   wprintf(mLog," copy-from: %s \n",RJfrom);
   strcpy(rdirk,BJmp);                                                     //  validate copy-from location
   strcat(rdirk,RJfrom);                                                   //  /media/dvd/home/...
   pdirk = opendir(rdirk);
   if (! pdirk) {
      wprintf(mLog," *** invalid copy-from location \n");
      nerr++;
   }
   else closedir(pdirk);

   cc = strlen(RJfrom);                                                    //  insure '/' at end
   if (RJfrom[cc-1] != '/') strcat(RJfrom,"/");

   wprintf(mLog,"   copy-to: %s \n",RJto);
   pdirk = opendir(RJto);                                                  //  validate copy-to location
   if (! pdirk) {
      wprintf(mLog," *** invalid copy-to location \n");
      nerr++;
   }
   else closedir(pdirk);

   cc = strlen(RJto);                                                      //  insure '/' at end
   if (RJto[cc-1] != '/') strcat(RJto,"/");

   nerr += nxValidate(RJinex,RJnx);                                        //  validate include/exclude recs

   wprintf(mLog," %d errors \n",nerr);
   if (! nerr) RJval = 1;
   else RJval = 0;
   return RJval;
}


//  list and validate a set of include/exclude recs

int nxValidate(char **inexrecs, int nrecs)
{
   char    *rtype, *fspec, nxrec[maxfcc];
   int      ii, nstat, errs = 0;

   for (ii = 0; ii < nrecs; ii++)                                          //  process include/exclude recs
   {
      strcpy(nxrec,inexrecs[ii]);
      wprintf(mLog," %s \n",nxrec);                                        //  output
      
      nstat = inexParse(nxrec,rtype,fspec);                                //  parse
      if (nstat == 0) continue;                                            //  comment
      if (nstat == 1) continue;                                            //  OK

      if (nstat == 2) {
         wprintf(mLog," *** cannot parse \n");                             //  cannot parse
         errs++;
         continue;
      }

      if (nstat == 3) {                                                    //  bad filespec
         wprintf(mLog," *** invalid filespec \n");
         errs++;
         continue;
      }
   }
   
   return errs;
}


//  get all backup files specified by include/exclude records
//  save in Drec[] array

int dGetFiles()
{
   const char     *fsp;
   char           *rtype, *fspec, bjrec[maxfcc], *errmess;
   int            ftf, cc, nstat, wstat, err;
   int            ii, jj, nfiles, ndvd, toobig;
   double         nbytes, dvdbytes;
   struct stat64  filestat;

   if (! BJval) {                                                          //  validate job data if needed
      dFilesReset(); 
      BJvalidate(0); 
      if (! BJval) return 0;                                               //  job has errors
   }

   if (Dnf > 0) return 0;                                                  //  avoid refresh

   wprintx(mLog,0,"\n""generating backup file set \n",boldfont);
   
   for (ii = 0; ii < BJnx; ii++)                                           //  process include/exclude recs
   {
      BJfiles[ii] = 0;                                                     //  initz. include/exclude rec stats
      BJbytes[ii] = 0.0;
      BJdvdno[ii] = 0;
      
      strcpy(bjrec,BJinex[ii]);                                            //  next record
      nstat = inexParse(bjrec,rtype,fspec);                                //  parse

      if (nstat == 0) continue;                                            //  comment

      if (strEqu(rtype,"include"))                                         //  include filespec
      {
         ftf = 1;

         while (1)
         {
            fsp = SearchWild(fspec,ftf);                                   //  find matching files
            if (! fsp) break;

            cc = strlen(fsp);
            if (cc > maxfcc-100) zappcrash("file cc: %d, %99s...",cc,fsp);

            Drec[Dnf].file = zmalloc(cc+1);
            strcpy(Drec[Dnf].file,fsp);

            err = stat64(fsp,&filestat);                                   //  check accessibility
            if (err == 0) {
               if (S_ISDIR(filestat.st_mode)) continue;                    //  skip directories
               if (! S_ISREG(filestat.st_mode)) continue;                  //  omit pipes, devices ...
            }                                                              //  (simlinks to reg. files included)

            Drec[Dnf].stat = err;                                          //  save file status
            Drec[Dnf].inclx = ii;                                          //  save pointer to include rec
            Drec[Dnf].size = filestat.st_size;                             //  save file size
            Drec[Dnf].mtime = filestat.st_mtime                            //  save last mod time
                            + filestat.st_mtim.tv_nsec * nano;             //  (nanosec resolution)  v.23
            if (err) Drec[Dnf].size = Drec[Dnf].mtime = 0;
            Drec[Dnf].disp = Drec[Dnf].ivf = 0;                            //  initialize

            BJfiles[ii]++;                                                 //  count included files and bytes
            BJbytes[ii] += Drec[Dnf].size;

            if (++Dnf == maxfs) {
               wprintf(mLog," *** exceeded %d files \n",maxfs);
               goto errret;
            }
         }
      }
         
      if (strEqu(rtype,"exclude"))                                         //  exclude filespec
      {
         for (jj = 0; jj < Dnf; jj++)                                      //  check included files (SO FAR) v.09
         {
            if (! Drec[jj].file) continue;
            wstat = MatchWild(fspec,Drec[jj].file);
            if (wstat != 0) continue;
            BJfiles[ii]--;                                                 //  un-count excluded file and bytes
            BJbytes[ii] -= Drec[jj].size;
            zfree(Drec[jj].file);                                          //  clear file data in array
            Drec[jj].file = 0;
            Drec[jj].stat = 0;                                             //  bugfix  v.24
         }
      }
   }                                                                       //  end of include/exclude recs

   for (ii = 0; ii < Dnf; ii++)                                            //  list and remove error files
   {                                                                       //  (after excluded files removed)  v.10
      if (Drec[ii].stat)
      {
         stat64(Drec[ii].file,&filestat);
         errmess = syserrText();
         wprintf(mLog," *** %s  omit: %s \n",errmess,kleenex(Drec[ii].file));
         jj = Drec[ii].inclx;
         BJfiles[jj]--;                                                    //  un-count file and bytes
         BJbytes[jj] -= Drec[ii].size;
         zfree(Drec[ii].file);
         Drec[ii].file = 0;
      }
   }

   for (ii = 0; ii < Dnf; ii++)                                            //  list and remove too-big files  v.14
   {
      if (Drec[ii].size > BJcap * giga)
      {
         wprintf(mLog," *** omit file too big: %s \n",kleenex(Drec[ii].file));
         jj = Drec[ii].inclx;
         BJfiles[jj]--;                                                    //  un-count file and bytes
         BJbytes[jj] -= Drec[ii].size;
         zfree(Drec[ii].file);
         Drec[ii].file = 0;
      }
   }

   ii = jj = 0;                                                            //  repack file arrays after deletions
   while (ii < Dnf)
   {
      if (Drec[ii].file == 0) ii++;
      else {
         if (ii > jj) {
            if (Drec[jj].file) zfree(Drec[jj].file);
            Drec[jj] = Drec[ii];                                           //  v.23
            Drec[ii].file = 0;
         }
         ii++;
         jj++;
      }
   }

   Dnf = jj;                                                               //  final file count in backup set
   
   Dbytes = 0.0;
   for (ii = 0; ii < Dnf; ii++) Dbytes += Drec[ii].size;                   //  compute total bytes from files

   nfiles = 0;
   nbytes = 0.0;

   for (ii = 0; ii < BJnx; ii++)                                           //  compute total files and bytes
   {                                                                       //    from include/exclude recs
      nfiles += BJfiles[ii];
      nbytes += BJbytes[ii];
   }

   wprintf(mLog," backup files: %d  %.1f MB \n",nfiles,nbytes/1000000);
   
   if ((nfiles != Dnf) || (Dbytes != nbytes)) {                            //  must match
      wprintf(mLog," *** bug: nfiles: %d  Dnf: %d \n",nfiles,Dnf);
      wprintf(mLog," *** bug: nbytes: %.0f  Dbytes: %.0f \n",nbytes,Dbytes);
      goto errret;
   }
   
   //  assign DVD sequence number to all files, under constraint that            v.14
   //  all files from same include record are on same DVD, if possible
      
   ndvd = 1;                                                               //  1st DVD sequence no.
   dvdbytes = 0.0;                                                         //  DVD bytes so far
   toobig = 0;
   
   for (ii = jj = 0; ii < Dnf; ii++)                                       //  loop all files
   {
      if (Drec[ii].inclx != Drec[jj].inclx) jj = ii;                       //  start of include group

      if (dvdbytes + Drec[ii].size > BJcap * giga) {                       //  exceeded DVD capacity in this group
         if (jj > 0 && Drec[jj].dvd == Drec[jj-1].dvd) ii = jj;            //  if same DVD as prior, restart group
         else toobig++;
         ndvd++;                                                           //  next DVD no.
         dvdbytes = 0.0;                                                   //  reset byte counter
      }

      Drec[ii].dvd = ndvd;                                                 //  set DVD no. for file
      dvdbytes += Drec[ii].size;                                           //  accum. bytes for this DVD no.
      
      if (ii == jj) BJdvdno[Drec[ii].inclx] = ndvd;                        //  set 1st DVD no. for include rec
   }

   BJndvd = ndvd;                                                          //  final DVD media count
      
   if (toobig) wprintf(mLog," *** warning: single include set exceeds DVD capacity \n");

   SortFileList((char *) Drec,sizeof(dfrec),Dnf,'A');                      //  sort Drec[Dnf] by Drec[].file
   
   for (ii = 1; ii < Dnf; ii++)                                            //  look for duplicate files  v.10
      if (strEqu(Drec[ii].file,Drec[ii-1].file)) {
         wprintf(mLog," *** duplicate file: %s \n",kleenex(Drec[ii].file));
         BJval = 0;                                                        //  invalidate backup job
      }

   if (! BJval) goto errret;
   return 0;

errret:
   dFilesReset();
   BJval = 0;
   return 0;
}


//  get existing files on DVD medium, save in Vrec[] array 
//  (the shell command "find ... -type f" does not find the
//   files "deleted" via copy from /dev/null in growisofs)

int vGetFiles()
{
   int            cc, gcc, err;
   char           command[100], *pp, *errmess;
   char           fspec1[maxfcc], fspec2[maxfcc];
   FILE           *fid;
   struct stat64  filestat;

   if (Vnf) return 0;                                                      //  avoid refresh

   mountDVDn(3);                                                           //  mount with retries
   if (! dvdmtd) return 0;                                                 //  cannot mount

   wprintx(mLog,0,"\n""generating DVD file set \n",boldfont);

   sprintf(command,"find -L %s -type f >%s",BJmp,TFdvdfiles);              //  -L: include symlinks
   wprintf(mLog," %s \n",command);
   
   err = system(command);                                                  //  list all DVD files to temp file
   if (err) {
      errmess = syserrText();
      wprintf(mLog," *** find command failed: %s \n",errmess);
      commFail++;
      return 0;
   }

   fid = fopen(TFdvdfiles,"r");                                            //  read file list
   if (! fid) {
      wprintf(mLog," *** cannot open /tmp scratch file \n");
      commFail++;
      return 0;
   }

   gcc = strlen(V_DKOPDIRK);
   
   while (1)
   {
      pp = fgets_trim(fspec1,maxfcc-2,fid);                                //  get next file
      if (! pp) break;                                                     //  eof
      
      cc = strlen(pp);                                                     //  absurdly long file name
      if (cc > maxfcc-100) {
         wprintf(mLog," *** absurd file skipped: %300s (etc.) \n",kleenex(pp));
         continue;
      }

      if (strnEqu(fspec1+BJmpcc,V_DKOPDIRK,gcc)) continue;                 //  ignore special dkop files

      repl_1str(fspec1,fspec2,"\\=","=");                                  //  replace "\=" with "=" in file name
      
      cc = strlen(fspec2) - BJmpcc;                                        //  save file in Vrec[] array
      Vrec[Vnf].file = zmalloc(cc+2);                                      //  (without DVD mount point)
      strcpy(Vrec[Vnf].file,fspec2 + BJmpcc);

      err = stat64(fspec1,&filestat);                                      //  check accessibility
      Vrec[Vnf].stat = err;                                                //  save file status
      Vrec[Vnf].size = filestat.st_size;                                   //  save file size
      Vrec[Vnf].mtime = filestat.st_mtime                                  //  save last mod time
                      + filestat.st_mtim.tv_nsec * nano;                   //  v.23
      if (err) Vrec[Vnf].size = Vrec[Vnf].mtime = 0;

      Vnf++;
      if (Vnf == maxfs) zappcrash("exceed %d files",maxfs);
   }

   fclose (fid);

   SortFileList((char *) Vrec,sizeof(vfrec),Vnf,'A');                      //  sort Vrec[Vnf] by Vrec[].file
   
   wprintf(mLog," DVD files: %d \n",Vnf);
   return 0;
}


//  get all DVD restore files specified by include/exclude records

int rGetFiles()
{
   char       *rtype, *fspec, fspecx[maxfcc], rjrec[maxfcc];
   int         ii, jj, cc, nstat, wstat, ninc, nexc;

   if (! RJval) return 0;

   rFilesReset();                                                          //  clear restore files
   vGetFiles();                                                            //  get DVD files
   if (! Vnf) return 0;

   wprintf(mLog,"\n""generating DVD restore file set \n");
   
   for (ii = 0; ii < RJnx; ii++)                                           //  process include/exclude recs
   {
      strcpy(rjrec,RJinex[ii]);                                            //  next record
      wprintf(mLog," %s \n",rjrec);                                        //  output
      
      nstat = inexParse(rjrec,rtype,fspec);                                //  parse
      if (nstat == 0) continue;                                            //  comment
      
      repl_1str(fspec,fspecx,"\\=","=");                                   //  replace "\=" with "=" in file name

      if (strEqu(rtype,"include"))                                         //  include filespec
      {
         ninc = 0;                                                         //  count of included files

         for (jj = 0; jj < Vnf; jj++)                                      //  screen all DVD files
         {
            wstat = MatchWild(fspecx,Vrec[jj].file);
            if (wstat != 0) continue;
            cc = strlen(Vrec[jj].file);                                    //  add matching files
            Rrec[Rnf].file = zmalloc(cc+1);
            strcpy(Rrec[Rnf].file,Vrec[jj].file);
            Rnf++; ninc++;
            if (Rnf == maxfs) zappcrash("exceed %d files",maxfs);
         }
            
         wprintf(mLog,"  %d files added \n",ninc);
      }

      if (strEqu(rtype,"exclude"))                                         //  exclude filespec
      {
         nexc = 0;

         for (jj = 0; jj < Rnf; jj++)                                      //  check included files (SO FAR) v.09
         {
            if (! Rrec[jj].file) continue;

            wstat = MatchWild(fspecx,Rrec[jj].file);
            if (wstat != 0) continue;
            zfree(Rrec[jj].file);                                          //  remove matching files
            Rrec[jj].file = 0;
            nexc++;
         }

         wprintf(mLog,"  %d files removed \n",nexc);
      }
   }

   ii = jj = 0;                                                            //  repack after deletions
   while (ii < Rnf)
   {
      if (Rrec[ii].file == 0) ii++;
      else
      {
         if (ii > jj) 
         {
            if (Rrec[jj].file) zfree(Rrec[jj].file);
            Rrec[jj].file = Rrec[ii].file;
            Rrec[ii].file = 0;
         }
         ii++;
         jj++;
      }
   }

   Rnf = jj;
   wprintf(mLog," total file count: %d \n",Rnf);

   cc = strlen(RJfrom);                                                    //  copy from: /home/.../

   for (ii = 0; ii < Rnf; ii++)                                            //  get selected DVD files to restore
   {
      if (! strnEqu(Rrec[ii].file,RJfrom,cc)) {
         wprintf(mLog," *** not under copy-from; %s \n",Rrec[ii].file);
         RJval = 0;                                                        //  mark restore job invalid
         continue;
      }
   }

   SortFileList((char *) Rrec,sizeof(rfrec),Rnf,'A');                      //  sort Rrec[Rnf] by Rrec[].file
   return 0;
}


//  helper function for backups and reports
//
//  compare disk and DVD files, set dispositions in Drec[] and Vrec[] arrays
//       n  new         on disk, not on DVD
//       d  deleted     on DVD, not on disk
//       m  modified    on both, but not equal
//       u  unchanged   on both, and equal

int setFileDisps()
{
   int            dii, vii, comp;
   char           disp;
   double         diff;
   
   dii = vii = 0;
   nnew = nmod = nunc = ndel = comp = 0;
   Mbytes = 0.0;                                                           //  total bytes, new and modified files
   
   while ((dii < Dnf) || (vii < Vnf))                                      //  scan disk and DVD files in parallel
   {   
      if ((dii < Dnf) && (vii == Vnf)) comp = -1;
      else if ((dii == Dnf) && (vii < Vnf)) comp = +1;
      else comp = strcmp(Drec[dii].file, Vrec[vii].file);

      if (comp < 0)
      {                                                                    //  unmatched disk file: new on disk
         Drec[dii].disp = 'n';
         Mbytes += Drec[dii].size;                                         //  accumulate Mbytes
         nnew++;                                                           //  count new files
         dii++;
      }

      else if (comp > 0)
      {                                                                    //  unmatched DVD file: deleted on disk
         Vrec[vii].disp = 'd';
         ndel++;                                                           //  count deleted files
         vii++;
      }

      else if (comp == 0)                                                  //  file present on disk and DVD
      {
         disp = 'u';                                                       //  set initially unchanged
         if (Drec[dii].stat != Vrec[vii].stat) disp = 'm';                 //  fstat() statuses are different
         diff = fabs(Drec[dii].size - Vrec[vii].size);
         if (diff > 0) disp = 'm';                                         //  sizes are different
         diff = fabs(Drec[dii].mtime - Vrec[vii].mtime);
         if (diff > modtimetolr) disp = 'm';                               //  mod times are different   v.23
         Drec[dii].disp = Vrec[vii].disp = disp;

         if (disp == 'u') nunc++;                                          //  count unchanged files
         if (disp == 'm') nmod++;                                          //  count modified files
         if (disp == 'm') Mbytes += Drec[dii].size;                        //    and accumulate Mbytes

         dii++;
         vii++;
      }
   }
   
   Mfiles = nnew + nmod + ndel;
   return 0;
}


//  Sort file list in memory (disk files, DVD files, restore files).
//  Sort ascii sequence, or sort subdirectories in a directory before files.

int SortFileList(char * recs, int RL, int NR, char sort)                   //  v.02
{
   HeapSortUcomp fcompA, fcompD;                                           //  compare filespecs functions
   if (sort == 'A') HeapSort(recs,RL,NR,fcompA);                           //  normal ascii compare
   if (sort == 'D') HeapSort(recs,RL,NR,fcompD);                           //  compare directories first  v.24
   return 0;
}

int fcompA(const char * rec1, const char * rec2)                           //  ascii comparison
{
   dfrec  *r1 = (dfrec *) rec1;
   dfrec  *r2 = (dfrec *) rec2;
   return strcmp(r1->file,r2->file);
}

int fcompD(const char * rec1, const char * rec2)                           //  special compare filenames
{                                                                          //  subdirectories in a directory are
   dfrec  *r1 = (dfrec *) rec1;                                            //    less than files in the directory
   dfrec  *r2 = (dfrec *) rec2;
   return filecomp(r1->file,r2->file);
}

int filecomp(const char *file1, const char *file2)                         //  special compare filenames   v.24
{                                                                          //  subdirectories compare before files
   const char  *pp1, *pp10, *pp2, *pp20;
   char        slash = '/';
   int         cc1, cc2, comp;
   
   pp1 = file1;                                                            //  first directory level or file
   pp2 = file2;

   while (true)
   {
      pp10 = strchr(pp1,slash);                                            //  find next slash
      pp20 = strchr(pp2,slash);
      
      if (pp10 && pp20) {                                                  //  both are directories
         cc1 = pp10 - pp1;
         cc2 = pp20 - pp2;
         if (cc1 < cc2) comp = strncmp(pp1,pp2,cc1);                       //  compare the directories
         else comp = strncmp(pp1,pp2,cc2);
         if (comp) return comp;
         else if (cc1 != cc2) return (cc1 - cc2);
         pp1 = pp10 + 1;                                                   //  equal, check next level
         pp2 = pp20 + 1;
         continue;
      }
      
      if (pp10 && ! pp20) return -1;                                       //  only one is a directory,
      if (pp20 && ! pp10) return 1;                                        //    the directory is first
      
      comp = strcmp(pp1,pp2);                                              //  both are files, compare
      return comp;
   }
}


//  reset all backup job data and free allocated memory

int BJreset()
{
   for (int ii = 0; ii < BJnx; ii++) zfree(BJinex[ii]);
   BJnx = 0;
   *BJbmode = *BJvmode = 0;
   BJval = BJmod = 0;
   dFilesReset();                                                          //  reset dependent disk file data
   return 0;
}


//  reset all restore job data and free allocated memory

int RJreset()                                                              //  v.08
{
   for (int ii = 0; ii < RJnx; ii++) zfree(RJinex[ii]);
   RJnx = 0;
   RJval = 0;
   rFilesReset();                                                          //  reset dependent disk file data
   return 0;
}


//  reset all file data and free allocated memory

int dFilesReset()
{                                                                          //  disk files data
   for (int ii = 0; ii < Dnf; ii++) 
   {
      zfree(Drec[ii].file);
      Drec[ii].file = 0;
   }

   Dnf = 0;
   Dbytes = Dbytes2 = Mbytes = 0.0;
   return 0;
}

int vFilesReset()
{                                                                          //  DVD files data
   for (int ii = 0; ii < Vnf; ii++) 
   {
      zfree(Vrec[ii].file);
      Vrec[ii].file = 0;
   }

   Vnf = 0;
   Vbytes = Mbytes = 0.0;
   return 0;
}

int rFilesReset()
{                                                                          //  DVD restore files data
   for (int ii = 0; ii < Rnf; ii++) 
   {
      zfree(Rrec[ii].file);
      Rrec[ii].file = 0;
   }

   Rnf = 0;
   return 0;
}


//  helper function to copy a file from DVD to disk

char * copyFile(char * vfile, char *dfile)                                 //  v.23
{
   char              vfile1[maxfcc], vfilex[maxfcc];
   int               fid1, fid2, err, rcc;
   char              *pp, *errmess, buff[vrcc];
   struct stat64     fstat;
   struct timeval    ftimes[2];

   strcpy(vfile1,BJmp);                                                    //  prepend DVD mount point
   strcat(vfile1,vfile);
   repl_1str(vfile1,vfilex,"=","\\=");                                     //  replace "=" with "\=" in DVD file
   
   fid1 = open(vfilex,O_RDONLY+O_NOATIME+O_LARGEFILE);                     //  open input file
   if (fid1 == -1) return syserrText();

   fid2 = open(dfile,O_WRONLY+O_CREAT+O_TRUNC+O_LARGEFILE);                //  open output file
   if (fid2 == -1 && errno == ENOENT) {
      pp = dfile;
      while (true) {                                                       //  create one or more directories,
         pp = strchr(pp+1,'/');                                            //    one level at a time
         if (! pp) break;
         *pp = 0;
         err = mkdir(dfile,0700);
         if (! err) chmod(dfile,0700);
         *pp = '/';
         if (err) {
            if (errno == EEXIST) continue;
            errmess = syserrText();
            close(fid1);
            return errmess;
         }
      }
      fid2 = open(dfile,O_WRONLY+O_CREAT+O_TRUNC+O_LARGEFILE);             //  open output file again
   }

   if (fid2 == -1) {
      errmess = syserrText();
      close(fid1);
      return errmess;
   }
   
   while (true)
   {
      rcc = read(fid1,buff,vrcc);                                          //  read huge blocks
      if (rcc == 0) break;
      if (rcc == -1) {
         errmess = syserrText();
         close(fid1);
         close(fid2);
         return errmess;
      }

      rcc = write(fid2,buff,rcc);                                          //  write blocks
      if (rcc == -1) {
         errmess = syserrText();
         close(fid1);
         close(fid2);
         return errmess;
      }
   }

   close(fid1);
   close(fid2);

   stat64(vfilex,&fstat);                                                  //  get input file attributes

   ftimes[0].tv_sec = fstat.st_atime;                                      //  conv. access times to microsecs
   ftimes[0].tv_usec = fstat.st_atim.tv_nsec / 1000;
   ftimes[1].tv_sec = fstat.st_mtime;
   ftimes[1].tv_usec = fstat.st_mtim.tv_nsec / 1000;

   chmod(dfile,fstat.st_mode);                                             //  set output file attributes
   chown(dfile,fstat.st_uid,fstat.st_gid);                                 //  (if supported by file system)
   utimes(dfile,ftimes);

   return 0;
}


//  Verify helper function
//  Verify that file on backup medium is readable, return its length.
//  Optionally compare backup file to disk file, byte for byte.
//  return:  0: OK  1: open error  2: read error  3: compare fail

char * checkFile(char * dfile, int compf, double &tcc)                     //  v.24
{
   int            vfid = 0, dfid = 0;
   int            err, vcc, dcc, cmperr = 0;
   int            open_flags = O_RDONLY+O_NOATIME+O_LARGEFILE;             //  O_DIRECT not allowed for DVD
   char           vfile[maxfcc], *vbuff = 0, *dbuff = 0;
   char           *errmess = 0;
   double         dtime, vtime;
   struct stat64  filestat;

   tcc = 0.0;

   strcpy(vfile,BJmp);                                                     //  prepend mount point
   repl_1str(dfile,vfile+BJmpcc,"=","\\=");                                //  replace "=" with "\=" in DVD file

   if (compf) goto comparefiles;
   
   vfid = open(vfile,open_flags);                      
   if (vfid == -1) return syserrText();

   err = posix_memalign((void**) &vbuff,512,vrcc);
   if (err) zappcrash("memory allocation failure");

   while (1)
   {
      vcc = read(vfid,vbuff,vrcc);
      if (vcc == 0) break;
      if (vcc == -1) { errmess = syserrText(); break; }
      tcc += vcc;                                                          //  accumulate length      
      if (checkKillPause()) break;
   }
   goto cleanup;

comparefiles:

   vfid = open(vfile,open_flags);
   if (vfid == -1) return syserrText();

   dfid = open(dfile,open_flags);
   if (dfid == -1) { errmess = syserrText(); goto cleanup; }

   err = posix_memalign((void**) &vbuff,512,vrcc);
   if (err) zappcrash("memory allocation failure");
   err = posix_memalign((void**) &dbuff,512,vrcc);
   if (err) zappcrash("memory allocation failure");

   while (1)
   {
      vcc = read(vfid,vbuff,vrcc);                                         //  read two files
      if (vcc == -1) { errmess = syserrText(); goto cleanup; }

      dcc = read(dfid,dbuff,vrcc);
      if (dcc == -1) { errmess = syserrText(); goto cleanup; }

      if (vcc != dcc) cmperr++;                                            //  compare buffers 
      if (memcmp(vbuff,dbuff,vcc)) cmperr++;                               //  *** bugfix ***     v.31

      tcc += vcc;                                                          //  accumulate length
      if (vcc == 0) break;
      if (dcc == 0) break;
      if (checkKillPause()) break;
   }

   if (vcc != dcc) cmperr++;

   if (cmperr) {                                                           //  compare error
      stat64(dfile,&filestat);
      dtime = filestat.st_mtime + filestat.st_mtim.tv_nsec * nano;         //  file modified since snapshot?
      stat64(vfile,&filestat);
      vtime = filestat.st_mtime + filestat.st_mtim.tv_nsec * nano;
      if (fabs(dtime-vtime) < modtimetolr) errmess = "compare error";      //  no, a real compare error
   }

cleanup:
   if (vfid) close(vfid);
   if (dfid) close(dfid);
   if (vbuff) free(vbuff);
   if (dbuff) free(dbuff);
   return errmess;
}


//  track current /directory/.../filename.ext  on logging window 
//  display directory and file names in overlay mode (no scrolling)

int track_filespec(char * filespec)                                        //  v.23
{
   int         cc;
   char        pdirk[300], pfile[300], *pp;
   
   pp = strrchr(filespec+1,'/');                                           //  parse directory/filename
   if (pp) {
      cc = pp - filespec + 2;
      strncpy0(pdirk,filespec,cc);
      strncpy0(pfile,pp+1,299);
   }
   else {
      strcpy(pdirk," ");
      strncpy0(pfile,filespec,299);
   }

   wprintf(mLog,-3," %s \n",kleenex(pdirk));                               //  output /directory
   wprintf(mLog,-2," %s \n",kleenex(pfile));                               //          filename
   return 0;
}


//  log error message and scroll down to prevent it from being overlaid

int track_filespec_err(char * filespec, char * errmess)                    //  v.23
{
   wprintf(mLog,-3," *** %s  %s \n",errmess,kleenex(filespec));
   wprintf(mLog," \n");
   return 0;
}


//  remove special characters in exotic file names causing havoc in output formatting    v.29

char  * kleenex(const char *name)
{
   static char    name2[1000];

   strncpy0(name2,name,999);

   for (int ii = 0; name2[ii]; ii++)
      if (name2[ii] >= 8 && name2[ii] <= 13)                               //  screen out formatting chars.
         name2[ii] = '?';                                                  //  v.34

   return name2;
}


//  do shell command (subprocess) and echo outputs to log window
//  returns command status: 0 = OK, +N = error
//  compensate for growisofs failure not always indicated as bad status    v.20
//  depends on growisofs output being in english

int do_shell(char * pname, char * command)
{
   int         scroll, pscroll;
   char        buff[1000], *errmess;
   int         err, eof, gerr = 0;

   snprintf(buff,999,"\n""shell: %s \n",command);
   wprintx(mLog,0,buff,boldfont);

   err = createProc(command);
   if (err) {
      errmess = syserrText(err);
      wprintf(mLog," *** createProc() failed: %s \n",errmess);
      goto shell_exit;
   }

   strncpy0(subprocName,pname,20);
   if (strEqu(pname,"growisofs")) track_growisofs_files(0);                //  initialize progress tracker  v.23

   scroll = pscroll = 1;
   
   while (1)
   {
      eof = getProcOutput(buff,999,err);                                   //  v.20
      if (eof) break;
      
      pscroll = scroll;
      scroll = 1;

      if (strEqu(pname,"growisofs")) {                                     //  growisofs output
         if (track_growisofs_files(buff)) scroll = 0;                      //  conv. % done into file position
         else {
            scroll = 1;                                                    //  v.34 revised ****
            if (strstr(buff,"genisoimage:")) gerr = 999;                   //  trap errors not reported in
            if (strstr(buff,"mkisofs:")) gerr = 998;                       //    flakey growisofs status
            if (strstr(buff,"failed")) gerr = 997;
         }
      }

      if (strstr(buff,"formatting")) scroll = 0;                           //  dvd+rw-format output

      if (scroll) {                                                        //  output to next line
         wprintf(mLog," %s: %s \n",pname,kleenex(buff));
         zsleep(0.1);                                                      //  throttle output a little
      }
      else {
         if (pscroll) wprintf(mLog,"\n");                                  //  transition from scroll to overlay
         wprintf(mLog,-2," %s: %s \n",pname,kleenex(buff));                //  output, overlay prior output
      }
   }

   errmess = 0;
   if (err) errmess = syserrText(err);                                     //  more info   v.34
   else if (gerr) { err = gerr; errmess = "growisofs failed"; }
   if (err) wprintf(mLog," %s status: %d %s \n", pname, err, errmess);
   else wprintf(mLog," %s status: OK \n",pname);
   
shell_exit:
   *subprocName = 0;
   if (err) commFail++;
   return err;
}


//  Convert "% done" from growisofs into corresponding position in list of files being copied.
//  Incremental backups start with  % done = (initial DVD space used) / (final DVD space used).

int track_growisofs_files(char * buff)                                     //  v.23
{
   static double     bbytes, gpct0, gpct;
   static int        dii, dii2, err;
   static char       *dfile;
   
   if (! buff) {                                                           //  initialize
      dii = 0;
      dii2 = -1;
      bbytes = 0;
      dfile = "";
      return 0;
   }

   if (! strstr(buff,"% done")) return 0;                                  //  not a % done record

   err = convSD(buff,gpct,0.0,100.0);                                      //  get % done, 0-100
   if (err > 1) return 0;
   
   if (strEqu(mbmode,"full")) {                                            //  full backup, possibly > 1 DVD
      while (dii < Dnf) {
         if (bbytes/Dbytes2 > gpct/100) break;                             //  exit if enough bytes      v.28
         if (Drec[dii].dvd == dvdnum) {
            bbytes += Drec[dii].size;                                      //  sum files matching DVD number
            dii2 = dii;
         }
         dii++;
      }
   }

   else  {                                                                 //  incremental backup
      if (bbytes == 0) gpct0 = gpct;                                       //  establish base % done
      while (dii < Dnf) {
         if (bbytes/Mbytes > (gpct-gpct0)/(100-gpct0)) break;              //  exit if enough bytes
         if (Drec[dii].disp == 'n' || Drec[dii].disp == 'm') {
            bbytes += Drec[dii].size;                                      //  sum new and modified files
            dii2 = dii;
         }
         dii++;
      }
   }

   if (dii2 > -1) dfile = Drec[dii2].file;
   snprintf(buff,999,"%6.1f %c  %s",gpct,'%',dfile);                       //  nn.n %  /directory/.../filename
   return 1;
}



