# NAME

Getopt::Yath - Option processing yath style.

# DESCRIPTION

This is the getopt processor yath uses. It should work perfectly fine outside
of yath as well.

# SYNOPSIS

## DEFINING OPTIONS

    package My::Package;
    use Getopt::Yath;

    # Include options from other modules that use Getopt::Yath
    include_options(
        'Some::Options::Package',
        ...,
    );

    # an option group is basically a way to specify common parameters for all
    # options defined in the codeblock.
    option_group {category => 'Human readable category', group => 'settings_group'} => sub {

        # In addition to the fields specified here, all the fields from the
        # 'option_group' above are included:
        option verbose => (
            type => 'Bool',                 # This is a boolean type, it does not take an argument

            # Optional fields
            short => 'v',                   # Allow -v in addition to --verbose
            default => 0,                   # What value to use if none is specified (booleans default to 0 anyway)
            from_env_vars => ['VERBOSE'],   # If the $VERBOSE environment variable is set, this will be set to true.
            set_env_vars  => ['VERBOSE'],   # If this is set to true it will also set the $VERBOSE environment variable

            description => "This turns on verbose output",
        );

        option username => (
            type => 'Scalar',           # Scalar type, requires an argument

            # Optional
            short => 'U',               # Allow: -U Bob
            alt => ['user', 'uname'],   # Allow: --user Bob, --uname Bob
            from_env_vars => ['USER'],  # Get the value from the $USER env var if it is not provided.
            default => sub { "bob" . rand(100) }, # If none is specified, and the env var is empty, generate a default.

            description => "This sets your username",
        );

        # Other options
        ...
    };

## PARSING OPTIONS

    my $parsed = parse_options(
        ['-v', '--user', 'fred', 'not_an_opt', '--', '--will-not-process'],    # Normally you might pass in \@ARGV
        skip_non_opts => 1,                                                    # Skip non-opts, that is any argument that does not start with a '-' it will just skip.
        stops         => ['--'],                                               # Stop processing
        no_set_env    => 1,                                                    # Do not actually change %ENV
        groups        => { ':{' => '}:' },                                     # Arguemnts between the :{ and }: will be captured into an arrayref, they can be used as option values, or stand-alone
    );

The `$parsed` structure:

    $parsed = {
        'cleared' => {},                       # Options that were cleared with --no-opt
        'skipped' => ['not_an_opt'],           # Skipped non options
        'settings' => {                        # Blessed as Getopt::Yath::Settings
            'settings_group' => {              # Blessed as Getopt::Yath::Settings::Group
                'verbose'  => 1,               # The option and its value
                'username' => 'fred',          # Another option and value
            },
        },
        'stop'    => '--',                     # We stopped at '--', if there was no '--' this would be undef
        'remains' => ['--will-not-process'],   # Stuff after the '--' that we did not process
        'modules' => {'My::Package' => 2},     # Any module that provided options that were seen will be listed
        'env'     => {'VERBOSE' => 1}          # Environment variabvles that would have been set if not for 'no_set_env'
    };

## GENERATING COMMAND LINE HELP OUTPUT:

    sub help {
        print options()->docs('cli');
    }

    help();

Produces:

    Human readable category
     --username ARG,  --username=ARG,  --user ARG,  --user=ARG,  --uname ARG
     --uname=ARG,  -U ARG,  -U=ARG,  --no-username
       This sets your username

       Can also be set with the following environment variables: USER

     --verbose,  -v,  --no-verbose
       This turns on verbose output

       Can also be set with the following environment variables: VERBOSE

       The following environment variables will be set after arguments are processed: VERBOSE

## GENERATING POD:

    sub pod {
        print options()->docs('pod', head => 2); # The '2' specifies what heading level to use
    }

    pod();

Produces:

    =head2 Human readable category

    =over 4

    =item --username ARG

    =item --username=ARG

    =item --user ARG

    =item --user=ARG

    =item --uname ARG

    =item --uname=ARG

    =item -U ARG

    =item -U=ARG

    =item --no-username

    This sets your username

    Can also be set with the following environment variables: C<USER>


    =item --verbose

    =item -v

    =item --no-verbose

    This turns on verbose output

    Can also be set with the following environment variables: C<VERBOSE>

    The following environment variables will be set after arguments are processed: C<VERBOSE>


    =back

# EXPORTS

- $opts = options()

    This will return an [Getopt::Yath::Instance](https://metacpan.org/pod/Getopt%3A%3AYath%3A%3AInstance) object. This object holds all the
    defined options, and does all the real work under the hood.

- $parsed = parse\_options(\\@ARGV)
- $parsed = parse\_options(\\@ARGV, %PARAMS)

    This processes an arrayref of command line arguments into a structure that can
    be easily referenced. If there is a problem parsing, such as invalid options in
    the array, exceptions will be thrown.

    The `$parsed` structure will look like this:

        $parsed = {
            'cleared' => {},                       # Options that were cleared with --no-opt
            'skipped' => ['not_an_opt'],           # Skipped non options
            'settings' => {                        # Blessed as Getopt::Yath::Settings
                'settings_group' => {              # Blessed as Getopt::Yath::Settings::Group
                    'verbose'  => 1,               # The option and its value
                    'username' => 'fred',          # Another option and value
                },
            },
            'stop'    => '--',                     # We stopped at '--', if there was no '--' this would be undef
            'remains' => ['--will-not-process'],   # Stuff after the '--' that we did not process
            'modules' => {'My::Package' => 2},     # Any module that provided options that were seen will be listed
            'env'     => {'VERBOSE' => 1}          # Environment variabvles that would have been set if not for 'no_set_env'
        };

    Available parameters that effect parsing are:

    - stops => \\@STOP\_LIST
    - stops => \['--'\]

        This is a list of string that if encountered should stop the parsing process.
        The string encountered will be put into the `stop` field of the `$parsed`
        structure. Any unparsed arguments after the stop will be put into the
        `remains` key of the `$parsed` structure.

        This is mostly useful for supporting the `--` option.

    - groups => \\%GROUP\_BORDERS
    - groups => { ':{' => '}:' }

        Arguments between the specified start and end tokens will be grouped together into an arrayref.

    - stop\_at\_non\_opts => BOOL

        This will cause parsing to stop at any non-option. A non-option in this case is
        any argument that does not start with a `-`.

        The item stopped at will be placed in the `stop` field of the `$parsed`
        structure with the remaining arguments placed in the `remains` field.

    - skip\_non\_opts => BOOL

        This will skip any non-option encountered. A non-option is any argument that
        does not start with `-`. All skipped items will be placed into the `skipped`
        field of the &lt;$parsed> structure.

    - skip\_invalid\_opts => BOOL

        This will skip any invalid option encountered. This includes any argument that
        starts with `-` but is not a valid option. All skipped items will be placed
        into the `skipped` field of the &lt;$parsed> structure.

    - stop\_at\_invalid\_opts => BOOL

        This will cause parsing to stop at any invalid option. This includes any
        argument that starts with `-` but is not a valid option.

        The item stopped at will be placed in the `stop` field of the `$parsed`
        structure with the remaining arguments placed in the `remains` field.

    - no\_set\_env => BOOL

        Set this to true to prevent any modifications to `%ENV`.

        The `env` key of the `$parsed` structure will contain the environment
        variable changes that would have been made.

        **Note:** The env key is always included even if `%ENV` is modified directly.

- include\_options('Options::Module::A', 'Options::Module::B', ...)

    This allows you to build libraries of `Getopt::Yath` options and include them
    as needed. Options from the specified libraries will be merged into the current
    packages options.

- option\_group \\%fields => sub { ... }
- option\_group {group => 'my\_group'} => sub { option ...; ... }

    Create a group of options with common parameters.

- option TITLE => \\%SPECIFICATION
- option TITLE => (type => '+My::Type', ...)
- option TITLE => (type => 'Getopt::Yath::Option::Type', ...)
- option TITLE => (type => 'Type', ...)

    This is used to define a single option. You must specify an option NAME and
    'type', which must be a valid [Getopt::Yath::Option](https://metacpan.org/pod/Getopt%3A%3AYath%3A%3AOption) subclass.

    The TILE is used to produce default values for the 'field' and 'name' fields,
    both of which can be specidied directly if the automatic values ar enot
    sufficient. 'field' gets the value of title with dashes replaced by
    underscrores. 'name' gets the value of title with underscores replaced with
    dashes.

    Most of the time you can just list the type as the part after the last `::` in
    `Getopt::Yath::Option::TYPE`. You can also specify
    `Getopt::Yath::Option::TYPE` or `Getopt::Yath::Option::TYPE::SubType`
    directly. However if you need to use a module that is not in the
    `Getopt::Yath::Option::` namespace you will need to prefix the module with a
    `+` to indicate that.

        $export{option_post_process} = sub {
            my $cb           = pop;
            my $weight       = shift // 0;
            my ($applicable) = @_;

            $applicable //= $common[-1]->{applicable} if @common;

            croak "You must provide a callback coderef" unless $cb && ref($cb) eq 'CODE';

            $instance->_post([caller()], $weight, $applicable, $cb);
        };

# OPTION TYPES AND SPECIFICATIONS

## REQUIRED WITH NO DEFAULTS

- title

    This is the first argument to `option()`. It is used to build the default
    values for both `field` and `name`.

- group => "group\_name"

    Name of the field to use in the options hash under which the option will be
    listed:

    `$parsed->{options}->{$group}->{$field_name} = $val`

- type => 'TypeName'
- type => 'Getopt::Yath::Option::TypeName'
- type => '+My::Custom::Type'

    This must be a valid [Getopt::Yath::Option](https://metacpan.org/pod/Getopt%3A%3AYath%3A%3AOption) subclass:

    - Scalar

        Takes a scalar value. A value is required. Can be used as `--opt VAL` or
        `--opt=val`. `--no-opt` can be used to clear the value.

    - Bool

        Is either on or off. `--opt` will turn it onn. `--no-opt` will turn it off.
        Default is off unless the `default` is parameter is provided.

    - Count

        Is an integer value, default is to start at `0`. `--opt` increments the
        counter. `--no-opt` resets the counter. `--opt=VAL` can be used to specify a
        desired count.

    - List

        Can take multiple values. `--opt VAL` appends a value to the list. `--no-opt`
        will empty the list. If a `split_on` parameter is provided then a single use
        can set multiple values. For example if `split_on` is set to `,` then
        `--opt foo,bar` is provided, then `foo` and `bar` will both be added to the
        list.

    - Map

        Expects all values to be `key=value` pairs and produces a hashref.
        `--opt foo=bar` will set `$h{foo} = 'bar'`. If a `split_on` parameter is
        provided then a single use can set multiple values. For example if `split_on`
        is set to `,` then `--opt foo=bar,baz=bat` is provided, then the result will
        have `$h{foo} = 'bar'; $h{baz} = 'bat'`.

    - Auto

        This type has an 'autofill' value that is used if no argument is provided to
        the parameter, IE `--opt`. But can also be given a specific value using
        `--opt=val`. It **DOES NOT** support `--opt VAL` which will most likely result
        in an exception.

    - AutoList

        This is a combination of 'Auto' and 'List' types. The no-arg form `--opt` will
        add the default values(s) to the list. The `--opt=VAL` form will add
        additional values.

    - AutoMap

        This is a combination of 'Auto' and 'Map' types. The no-arg form `--opt` will
        add the default key+value pairs to the hash. The `--opt=KEY=VAL` form will add
        additional values.

## REQUIRED WITH SANE DEFAULTS

- field => "field\_name"

    Name of the field to use in the group hash for the result of parsing arguments.

    `$parsed->{options}->{$group}->{$field_name} = $val`

    Default is to take the `title` value and replace any dashes with underscores.

- name => "option-name"

    Primary name for the option `--option-name`.

    Default is to take the `title` value and replace any underscores with dashes.

- trace => \[$caller, $file, $line\]

    This normally resolves to the place `option()` was called. You can manually
    override it with a custom value, but you should rarely ever need to.

- category => "Human Readable documentation category"

    When producing POD or command line documentation, options are put into
    "categories" which should be the human readabvle version of the `group` field.

    Default is "NO CATEGORY - FIX ME".

- description => "Explanation of what the option controls"

    Document what the option controls or does.

    Default is 'NO DESCRIPTION - FIX ME'.

## OPTIONAL

- short => 's'

    Specify a short flag to use. This is how you provide single-dash single-letter options.

    - `-s`

        If no argument is required this form is available.

    - `-s=VAL`

        If an argument is allowed this form is available

    - `-sVAL`

        If an argument is allowed, and this form is not directly disabled by the type
        (Types can override `allows_shortval()` to return false to forbid this form.
        Currently [Getopt::Yath::Option::Bool](https://metacpan.org/pod/Getopt%3A%3AYath%3A%3AOption%3A%3ABool) and [Getopt::Yath::Option::Count](https://metacpan.org/pod/Getopt%3A%3AYath%3A%3AOption%3A%3ACount)
        disable this form.

    - `-sss`

        So far only the [Getopt::Yath::Option::Count](https://metacpan.org/pod/Getopt%3A%3AYath%3A%3AOption%3A%3ACount) type makes use of this. It
        allows you to add the flag multiple times after a single dash to increment the
        count.

- alt => \\@LIST
- alt => \['alt1', 'alt2'\]

    Specify alternate or alias names that can be used to set or toggle a field.

    `--alt1` `--alt2 foo`

- prefix => "a-prefix"

    Specify a prefix to attach to the name, and to any alternate names. This is mainly useful when specifying an option group:

        option_group {prefix => 'foo'} => sub {
            option bar => (
                type => "Bool",
            );
        };

    This would then be used as `--foo-bar`

- module => 'My::Module'

    Specify the module the argument should be associated with. This defaults to the
    caller, so usually you do not need to specify it.

    This is mainly used in the case of plugins we only want to load if the option
    is used.

- no\_module => BOOL

    Default is 0. When this is set to true the module name is not used.

- applicable => sub { my $options = shift; ... ? 1 : 0 }

    This can be used to dynamically show/hide options. When this returns false the
    option will not be available.

- initialize => $scalar
- initialize => sub { ... }

    Initialize the value to this before any arguments are parsed. This is mainly
    used so that [Getopt::Yath::Option::Map](https://metacpan.org/pod/Getopt%3A%3AYath%3A%3AOption%3A%3AMap) can start with an empty hash, and
    [Getopt::Yath::Option::List](https://metacpan.org/pod/Getopt%3A%3AYath%3A%3AOption%3A%3AList) can be initialized to an empty arrayref.

    This can be a simple scalar (string or number, not a reference), or it may be a
    codeblock that returns anything you want. Only 1 item should be returned, extra
    values will result in undefined behavior. For a map this should return an empty
    hashref, for a list it should return an empty arrayref.

- clear => $scalar
- clear => sub { ... }

    Similar to `initialize`, but this is used when clearing the value. For things
    like 'Map' this should return a hashref, etc.

- default => $scalar
- default => sub { ... }

    Set a default to use if no value is provided at the command line.

    This can be a simple scalar (string or number, not a reference), or it may be a
    codeblock that returns anything you want.

    Most options will only accept a single default value.
    [Getopt::Yath::Option::Map](https://metacpan.org/pod/Getopt%3A%3AYath%3A%3AOption%3A%3AMap) and [Getopt::Yath::Option::List](https://metacpan.org/pod/Getopt%3A%3AYath%3A%3AOption%3A%3AList) support a list
    of defaults for setting key/value pairs, or adding items to an array.

    These are valid for anything:

        default => 'foo',
        default => 123,
        default => sub { "hi" }

    This is valid for an [Getopt::Yath::Option::Map](https://metacpan.org/pod/Getopt%3A%3AYath%3A%3AOption%3A%3AMap):

        default => sub { return ('foo' => 'bar') }

    This is valid for a [Getopt::Yath::Option::List](https://metacpan.org/pod/Getopt%3A%3AYath%3A%3AOption%3A%3AList):

        default => sub { return (1, 2, 3, 4) }

- autofill => $scalar
- autofill => sub { ... }

    This is used for [Getopt::Yath::Option::Auto](https://metacpan.org/pod/Getopt%3A%3AYath%3A%3AOption%3A%3AAuto) and similar. This is the value
    used if the command line option is provided, but no value is provided with it.

    This can be a simple scalar (string or number, not a reference), or it may be a
    codeblock that returns anything you want.

    Most options will only accept a single autofill value.
    [Getopt::Yath::Option::Map](https://metacpan.org/pod/Getopt%3A%3AYath%3A%3AOption%3A%3AMap) and [Getopt::Yath::Option::List](https://metacpan.org/pod/Getopt%3A%3AYath%3A%3AOption%3A%3AList) support a list
    of autofill data for setting key/value pairs, or adding items to an array.

    These are valid for anything:

        autofill => 'foo',
        autofill => 123,
        autofill => sub { "hi" }

    This is valid for an [Getopt::Yath::Option::Map](https://metacpan.org/pod/Getopt%3A%3AYath%3A%3AOption%3A%3AMap):

        autofill => sub { return ('foo' => 'bar') }

    This is valid for a [Getopt::Yath::Option::List](https://metacpan.org/pod/Getopt%3A%3AYath%3A%3AOption%3A%3AList):

        autofill => sub { return (1, 2, 3, 4) }

- normalize => sub { my ($input) = @\_; ...; return $output }

    If you wish to normalize or transform a value then you use this hook. The sub
    will get the option and the input value as its arguments. You should return the
    new value to set, or the input value if it does not need to change.

- trigger => sub { my ($opt, %params) = @\_; ... }

    This will be called any time the option is parsed from the command line, or
    whenever the command line clears the option.

    **NOTE:** It will not run when initial, autofill, or default values are set.

    The `%params` passed into the sub look like this:

        (
            # If this trigger is called because the value is cleared via --no-OPT:
            action => 'clear',
            val    => undef,

            # If a value is set because of --opt being parsed:
            action   => 'set',
            val      => [...],
            ref      => $ref,
            state    => $state,
            options  => $self,
            settings => $settings,
            group    => $group,
        );

    Note that val is always passed in as an arrayref. For simple scalar type
    options this will only ever have 1 value. For list or map types it may have
    multiple values, also note that for such types the trigger will only see the
    newly added values in the 'val' arrayref, not the values already included,
    which is important as list and map types can be built over several assignments.

- from\_env\_vars => \\@LIST

    A list of environment variables that will be used to populate the option's
    initial value. These will be checked in order, the first one that is set is the
    one that will be used, others will not be checked once a value is found. This
    will prevent the default value from being used, but using the option on the
    command line will override it.

    **Note:** that an environment variable can be prefixed with a `!` to indicate
    the value should be boolean-inverted. This means that an option like `quiet`
    can have `from_env_vars => ['!VERBOSE']` to be set to true when the
    VERBOSE env var is false. This also works when setting a variable, so you could
    have `set_env_vars => ['!VERBOSE']`.

- clear\_env\_vars => \\@LIST

    A list of enviornment variables to clear after the options are all populated.
    This is useful if you want to use an env var to set an option, but want to make
    sure no child proceses see the environemnt variable.

- set\_env\_vars => \\@LIST

    A list of environment variables that will be set to the value of this option
    (if it is set) when argument processing is complete.

    **Note:** This is only supported in types that have a single value, maps and
    lists are not supported.

    **Note:** that an environment variable can be prefixed with a `!` to indicate
    the value should be boolean-inverted. This means that an option like `quiet`
    can have `from_env_vars => ['!VERBOSE']` to be set to true when the
    VERBOSE env var is false. This also works when setting a variable, so you could
    have `set_env_vars => ['!VERBOSE']`.

- short\_examples => \\@LIST
- short\_examples => \['', 'ARG', '=ARG'\]
- short\_examples => \[' ARG', '=ARG'\]

    Override the default list of arguments when generating docs. This is used for
    the short form (single dash followed by a single letter and then a value
    `-Ilib`, `-I lib`, `-I=lib`, `-v`, `-vv`, `-vvv...`) documentation.

- long\_examples => \\@LIST
- long\_examples => \['', '=ARG'\]
- long\_examples => \[' ARG', '=ARG'\]

    Override the default list of arguments when generating docs. This is used for
    the long form (double-dash and option name and then a value `--include`,
    `--include=lib`, `--include lib`) documentation.

# SOURCE

The source code repository for Getopt-Yath can be found at
[http://github.com/Test-More/Getopt-Yath/](http://github.com/Test-More/Getopt-Yath/).

# MAINTAINERS

- Chad Granum <exodist@cpan.org>

# AUTHORS

- Chad Granum <exodist@cpan.org>

# COPYRIGHT

Copyright Chad Granum <exodist7@gmail.com>.

This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

See [http://dev.perl.org/licenses/](http://dev.perl.org/licenses/)
