/*

   Spinbox demo
   author: Charles S. Kerr

*/



#include <assert.h>
#include <stdlib.h>
#include <stdio.h>

#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/RowColumn.h>

#include "../src/Spinbox.h"



static Display *master_display;
static Widget master_parent;
static XtAppContext master_context;

static void set_style_cb ( Widget, XtPointer, XtPointer );
static void set_type_cb ( Widget, XtPointer, XtPointer );
static void set_arrow_cb ( Widget, XtPointer, XtPointer );
static void set_decimal_cb ( Widget, XtPointer, XtPointer );


static String fallbacks[] = {
   "SpinboxDemo*fontList:	-adobe-helvetica-bold-r-*-*-*-120-*-*-*-*-iso8859-1",
   "SpinboxDemo*caption.XmpSpinbox.columns:	16",
   NULL
};

static Widget
   spinbox,
   decimal_sb,
   decimal_lb;

static String style_names[] = {
   "Stacked",
   "Stacked Right",
   "Stacked Left",
   "Left",
   "Right",
   "Separate" };

static int style_values[] = {
   XmSPINBOX_STACKED,
   XmSPINBOX_STACKED_RIGHT,
   XmSPINBOX_STACKED_LEFT,
   XmSPINBOX_LEFT,
   XmSPINBOX_RIGHT,
   XmSPINBOX_SEPARATE };

static String type_names[] = {
   "Number",
   "Clock (hh:mm)",
   "Clock (hh:mm:ss)",
   "Dollars" };
static int type_values[] = {
   XmSPINBOX_NUMBER,
   XmSPINBOX_CLOCK_HM,
   XmSPINBOX_CLOCK_HMS,
   XmSPINBOX_DOLLARS };

static String arrow_names[] = {
   "Vertical",
   "Horizontal" };
static int arrow_values[] = {
   XmVERTICAL,
   XmHORIZONTAL };


static const char * tricks[] = {
   "* Try hitting the right mouse button on the arrow buttons",
   "* Try hitting the up/down key in the text field",
   "* Try hitting the page up/page down key in the text field",
   "* Try holding down one of the arrow buttons" };

int main (
   int argc,
   char *argv[] )
{
   Widget w, caption, rc;

#ifdef USE_EDITRES
    extern void _XEditResCheckMessages();
#endif

   assert ( XtNumber(style_names) == XtNumber(style_values) );
   assert ( XtNumber(type_names) == XtNumber(type_values) );
   assert ( XtNumber(arrow_names) == XtNumber(arrow_values) );

   master_parent = XtVaAppInitialize (
      &master_context, "SpinboxDemo", NULL, 0, &argc, argv, fallbacks,
      XtNtitle, "Spinbox Demo",
      NULL );

#ifdef USE_EDITRES
    XtAddEventHandler ( master_parent,
       (EventMask)0, True,
       _XEditResCheckMessages, NULL);
#endif

   master_display = XtDisplay ( master_parent );
   if ( !master_display ) {
      (void) fprintf ( stderr, "\nXmpSpinbox demo can't open display.\n" );
      return ( -1 );
   }

   rc = XtVaCreateManagedWidget ( "rc", xmRowColumnWidgetClass, master_parent,
      XmNorientation, XmVERTICAL,
      XmNmarginWidth, 10,
      XmNmarginHeight, 10,
      XmNspacing, 10,
      XmNallowResize, True,
      NULL );

   caption = XtVaCreateManagedWidget ( "caption", xmRowColumnWidgetClass, rc,
      XmNnumColumns, 2,
      XmNpacking, XmPACK_COLUMN,
      XmNorientation, XmVERTICAL,
      NULL );

   XtVaCreateManagedWidget ( "Style:", xmLabelWidgetClass, caption,
      NULL );
   XtVaCreateManagedWidget ( "Arrows:", xmLabelWidgetClass, caption,
      NULL );
   XtVaCreateManagedWidget ( "Type:", xmLabelWidgetClass, caption,
      NULL );
   decimal_lb = XtVaCreateManagedWidget ( "Decimal Places:", xmLabelWidgetClass, caption,
      NULL );
   w = XtVaCreateManagedWidget ( "styles_sb", xmpSpinboxWidgetClass, caption,
      XmNspinboxCycle, True,
      XmNspinboxType, XmSPINBOX_STRINGS,
      XmNitems, style_names,
      XmNitemCount, XtNumber(style_names),
      XmNvalue, 0,
      NULL );
   XtAddCallback ( w, XmNvalueChangedCallback, set_style_cb, NULL );

   w = XtVaCreateManagedWidget ( "arrows_sb", xmpSpinboxWidgetClass, caption,
      XmNspinboxCycle, True,
      XmNspinboxType, XmSPINBOX_STRINGS,
      XmNitems, arrow_names,
      XmNitemCount, XtNumber(arrow_names),
      XmNvalue, 0, /* default is vertical, and vertical is arrow_values[1] */
      NULL );
   XtAddCallback ( w, XmNvalueChangedCallback, set_arrow_cb, NULL );

   w = XtVaCreateManagedWidget ( "types_sb", xmpSpinboxWidgetClass, caption,
      XmNspinboxCycle, True,
      XmNspinboxType, XmSPINBOX_STRINGS,
      XmNitems, type_names,
      XmNitemCount, XtNumber(type_names),
      XmNvalue, XmSPINBOX_NUMBER,
      NULL );
   XtAddCallback ( w, XmNvalueChangedCallback, set_type_cb, NULL );

   decimal_sb = w = XtVaCreateManagedWidget ( "decimals_sb", xmpSpinboxWidgetClass, caption,
      XmNspinboxCycle, True,
      XmNvalue, 2,
      XmNminimum, 0,
      XmNmaximum, 8,
      NULL );
   XtAddCallback ( w, XmNvalueChangedCallback, set_decimal_cb, NULL );

   /**
   ***  Tricks
   ***
   ***  For some reason, when I build this on SunOS 4.1.3, X isn't converting \n
   ***  to linefeeds.  So instead I'm building a separate label for each trick...
   **/

   {
      int i;
      Widget tricks_rc;

      w = XtVaCreateManagedWidget ( "tricks_fr", xmFrameWidgetClass, rc,
         XmNshadowType, XmSHADOW_ETCHED_IN,
         XmNmarginWidth, 5,
         XmNmarginHeight, 5,
         XmNshadowThickness, 3,
         NULL );
      tricks_rc = XtVaCreateManagedWidget ( "tricks_rc", xmRowColumnWidgetClass, w,
         XmNorientation, XmVERTICAL,
         NULL );

      for ( i=0; i<XtNumber(tricks); i++ )
         w = XtVaCreateManagedWidget ( tricks[i], xmLabelWidgetClass, tricks_rc,
                XmNalignment, XmALIGNMENT_BEGINNING,
                NULL );
   }


   /**
   ***  Spinbox
   **/

   //w = XtVaCreateManagedWidget ( "spinbox_form", xmFormWidgetClass, rc,
   //   XmNshadowType, XmSHADOW_OUT,
   //   XmNshadowThickness, 5,
   //   NULL );
   spinbox = XtVaCreateManagedWidget ( "spinbox_sb", xmpSpinboxWidgetClass, rc,
      XmNtopAttachment, XmATTACH_FORM,
      XmNtopOffset, 20,
      XmNleftAttachment, XmATTACH_POSITION,
      XmNleftOffset, 20,
      //XmNbottomAttachment, XmATTACH_FORM,
      //XmNbottomOffset, 20,
      //XmNrightAttachment, XmATTACH_POSITION,
      //XmNrightPosition, 70,
      XmNvalue, 1,
      XmNspinboxStyle, XmSPINBOX_STACKED,
      XmNdecimalPoints, 2,
      NULL );

   XtRealizeWidget ( master_parent );
   XtAppMainLoop ( master_context );

   return ( 0 );
}


/**
***
***  CALLBACKS
***
**/

static void set_type_cb (
   Widget w,
   XtPointer client,
   XtPointer call )
{
   XmpSpinboxCallbackStruct *cbs = (XmpSpinboxCallbackStruct*)call;

   /* when we change types, XmNminimum and XmNmaximum changes, too.
      To be predictable, let's always set the value to 0 when changing
      types so that we know that the value will be within the new
      XmNminimum...XmNmaximum range.  This odd requirement only pops
      up when we're changing XmNspinboxType on the fly, as in demos. :) */

   XtVaSetValues ( spinbox,
      XmNspinboxType, type_values[cbs->value],
      XmNvalue, 0,
      NULL );

   /* We only want decimal places sensitive when type==XmSPINBOX_NUMBER */
   XtSetSensitive ( decimal_lb, type_values[cbs->value]==XmSPINBOX_NUMBER );
   XtSetSensitive ( decimal_sb, type_values[cbs->value]==XmSPINBOX_NUMBER );
}

static void set_style_cb (
   Widget w,
   XtPointer client,
   XtPointer call )
{
   XmpSpinboxCallbackStruct *cbs = (XmpSpinboxCallbackStruct*)call;
   XtVaSetValues ( spinbox, XmNspinboxStyle, style_values[cbs->value], NULL );
}

static void set_arrow_cb (
   Widget w,
   XtPointer client,
   XtPointer call )
{
   XmpSpinboxCallbackStruct *cbs = (XmpSpinboxCallbackStruct*)call;
   XtVaSetValues ( spinbox,
      XmNarrowOrientation, arrow_values[cbs->value],
      NULL );
}

static void set_decimal_cb (
   Widget w,
   XtPointer client,
   XtPointer call )
{
   XmpSpinboxCallbackStruct *cbs = (XmpSpinboxCallbackStruct*)call;
   XtVaSetValues ( spinbox,
      XmNdecimalPoints, cbs->value,
      NULL );
      
}
