
/* tick.c for firepower

   This is the main processing module.  All movement, shells, mines, deaths, wins
   losses, etc. are handled here.

   Copyright (c) 1995 Joe Rumsey
   */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "global.h"
#include "defs.h"
#include "struct.h"

#include "map.h"
#include "network.h"
#include "newplayer.h"

static int mapx[MAXPLAYERS], mapy[MAXPLAYERS];

/* check_move(p, oldx, oldy)

   checks if the player pointed at by p is able to move to his current position.
   If not, move him back to oldx, oldy

   Return: none
   */
static void check_move(player * p,
                       int oldx,
                       int oldy)
{

    /* First make sure he's not off the edge of the map */
    if ((p->p_x >> 4) > (GLO_map_info.m_width * GRIDWIDTH) - TANKSIZE / 2) {
        p->p_x = ((GLO_map_info.m_width * GRIDWIDTH) - TANKSIZE / 2) << 4;
    } else if ((p->p_x >> 4) < TANKSIZE / 2) {
        p->p_x = (TANKSIZE / 2) << 4;
    }
    if ((p->p_y >> 4) > (GLO_map_info.m_height * GRIDHEIGHT) - TANKSIZE / 2) {
        p->p_y = ((GLO_map_info.m_height * GRIDHEIGHT) - TANKSIZE / 2) << 4;
    } else if ((p->p_y >> 4) < TANKSIZE / 2) {
        p->p_y = (TANKSIZE / 2) << 4;
    }
    /* Now see if he's moved into a wall */
    if (map_iswall(GLO_map[(p->p_x >> 4) / GRIDWIDTH][(p->p_y >> 4) / GRIDHEIGHT])) {
        if (!map_iswall(GLO_map[(oldx >> 4) / GRIDWIDTH][(oldy >> 4) / GRIDHEIGHT])) {
            /* Yup, in a wall, move back to where he was */
            p->p_x = oldx;
            p->p_y = oldy;
        } else {                /* someone probably built a wall on top of us, skip backwards */
            p->p_x -= GLO_Cos[p->p_dir] * 32;
            p->p_y -= GLO_Sin[p->p_dir] * 32;
        }
    } else if ((GLO_map[(p->p_x >> 4) / GRIDWIDTH][(p->p_y >> 4) / GRIDHEIGHT] == T_TREE) &&
               ABS(((p->p_x >> 4) % GRIDWIDTH) - GRIDWIDTH / 2) < 10 &&
               ABS(((p->p_y >> 4) % GRIDHEIGHT) - GRIDHEIGHT / 2) < 10) {
        /* Ran into the trunk of a tree (the center of the square - it's ok
           to pass through the edges) */
        p->p_x = oldx;
        p->p_y = oldy;
    }
}

/* do_steering(p)

   see if player pointed to by p is turning, turn him if so

   Return: none
   */

static void do_steering(player * p)
{
    int keys = p->p_keys;
    unsigned int min, max;
    int turnamount;

#ifdef STEERING_KEYS
    /* obsolete */
    if (p->p_turnok)
        p->p_turnok--;
    if ((keys & RKEY) && !p->p_turnok) {
        p->p_dir += 16 - ABS(p->p_speed / ((MAXSPEED) / 13));
        p->p_turnok = TURNSPEED;
    } else if ((keys & LKEY) && !p->p_turnok) {
        p->p_dir -= 16 - ABS(p->p_speed / ((MAXSPEED) / 13));
        p->p_turnok = TURNSPEED;
    }
#else
    /* if he's already going the direction he wants to, we're done */
    if (p->p_desdir != p->p_dir) {

        /* see if he can turn yet, this is to make tanks turn slower
           at higher speeds */
        if (p->p_turnok)
            p->p_turnok--;
        if (!p->p_turnok) {

            /* Turn less each tick at higher speeds */
            turnamount = 16 - ABS(p->p_speed / ((MAXSPEED) / 13));

            /* should he turn left or right? */
            if (p->p_dir > p->p_desdir) {
                min = p->p_desdir;
                max = p->p_dir;
            } else {
                min = p->p_dir;
                max = p->p_desdir;
            }

            if ((turnamount > max - min) || (turnamount > 256 - max + min)) {
                /* is he within turnamount of his goal?  If so, set the desired course
               as his real course */
                p->p_dir = p->p_desdir;
            } else if ((unsigned char) ((int) p->p_dir - (int) p->p_desdir) > 127) {
                /* turn right towards goal */
                p->p_dir += turnamount;
            } else {
                /* turn left */
                p->p_dir -= turnamount;
            }
            /* set the counter for how soon he will be turned again */
            p->p_turnok = TURNSPEED;
        }
    }
#endif

    /* semi-obsolete, forward/backward keys.  Still work, but speed requests
       are better */
    if (keys & UKEY) {
        p->p_desspeed = MAXSPEED;
    } else if (keys & DKEY) {
        p->p_desspeed = MINSPEED;
    }
    /* see if he's going the speed he wants to be, accel or decel if not */
    if (p->p_desspeed > p->p_speed) {
        p->p_speed += FORWARDACCEL;
        if (p->p_speed > p->p_desspeed)
            p->p_speed = p->p_desspeed;
    } else if (p->p_desspeed < p->p_speed) {
        p->p_speed -= DECEL;
        if (p->p_speed < p->p_desspeed)
            p->p_speed = p->p_desspeed;
    }
}

/* terrain_mod(x,y)

   Calculate the movement modifier for a square of the map

   Return: the modifier
   */
static double terrain_mod(int x, int y)
{
    switch (GLO_map[(x >> 4) / GRIDWIDTH][(y >> 4) / GRIDHEIGHT]) {
    case T_ROAD:
    case T_BRIDGE:
    case T_BASE1:
    case T_BASE2:
        return 1.5;
    case T_WATER:
        return .2;
    case T_TREE:
        return .75;
    case T_CRATER:
        return .5;
    default:
        if (map_iswall(GLO_map[(x >> 4) / GRIDWIDTH][(y >> 4) / GRIDHEIGHT]))
            return 0;
        return 1;
    }
}

/* check_hidden(p)

   Check and see if the player pointed to by p is hidden by trees

   Return: none (sets the player's flags appropriately)
   */
static void check_hidden(player * p)
{
    int i, j, treecount = 0;

    /* there must be a tree IN your square for you to hide */
    if (GLO_map[MAPX(p->p_x)][MAPY(p->p_y)] != T_TREE) {
        p->p_flags &= ~(PFHIDDEN);
        return;
    }
    /* count how many trees surround your square.
     Check's each square in a 3x3 box around the player */
    for (i = MAPX(p->p_x) - 1; i < MAPX(p->p_x) + 2; i++) {
        for (j = MAPY(p->p_y) - 1; j < MAPY(p->p_y) + 2; j++) {

            if (GLO_map[i][j] == T_TREE) {
                treecount++;
                if (treecount >= TREESTOHIDE + 1) {     /* the +1 is to not count the tree in your square */
                    p->p_flags |= PFHIDDEN;
                    return;
                }
            }
        }
    }

    /* fell through the loop, player is not hidden */
    p->p_flags &= ~(PFHIDDEN);
}

/* kill_player(pl)

   player #pl's damage exceeded MAXDAMAGE, kill him and send him to the motd

   Return: none
   */
static void kill_player(int pl)
{
    int i;

    /* He'll be exploding for this long */
    GLO_players[pl].p_expfuse = EXPFUSE * 2;

    /* his status is DEAD */
    GLO_players[pl].p_status = PDEAD;

    /* if he's carrying a flag, leave it where he died */
    if (GLO_players[pl].p_carryflag >= 0) {
        GLO_flags[GLO_players[pl].p_carryflag].f_carrier = -1;
        GLO_players[pl].p_carryflag = -1;
    }
    /* send this player's status to himself */
    NET_update_one_player(pl, pl);

    /* and to everyone else */
    for (i = 0; i < MAXPLAYERS; i++) {
        if (GLO_players[i].p_status != PEMPTY) {
            NET_update_one_player(i, pl);
            NET_send_playerdata(i, pl);
        }
    }
}

/* set_update_flags(pl)

   set this player to be updated to everyone else

   Return: none
   */
static void set_update_flags(int pl)
{
    int i;

    for (i = 0; i < MAXPLAYERS; i++) {
        GLO_players[i].p_updateplayers |= (1 << pl);
    }
}


/* announce_flag(pl)

   Player #pl brought an enemy flag back to his base, spread the word!

   Return: none
   */
static void announce_flag(int pl)
{
    char buf[80];

    /* Who captured the flag, and what team they're on */
    sprintf(buf, "%s(%d) has captured a flag for team #%d!", GLO_players[pl].p_name, pl, GLO_players[pl].p_team + 1);
    NET_send_message_to_all(buf, M_SRV);

    /* Each team's score */
    sprintf(buf, "Score: Team 1: %d/%d     Team2: %d/%d", GLO_bases[0].b_numflags, GLO_num_flags[0],
            GLO_bases[1].b_numflags, GLO_num_flags[1]);
    NET_send_message_to_all(buf, M_SRV);
}

/* check_flags(pl)

   Check flags, the capturable kind, in relation to player #pl

   Return: none
   */
static void check_flags(int pl)
{
    int i, oteam;

    /* if this player is not active, forget it */
    if (GLO_players[pl].p_status == PEMPTY)
        return;

    /* oteam = the opposite team, ie the team whose flags you are trying
       to capture.  There is no way to affect your own flags directly. */
    oteam = (GLO_players[pl].p_team == 0) ? 1 : 0;

    /* loop through every flag */
    for (i = 0; i < GLO_maxflag; i++) {

        /* if this flag belongs to the other team */
        if (GLO_flags[i].f_team != GLO_players[pl].p_team) {

            /* if the player in question is CARRYING this flag */
            if (GLO_flags[i].f_carrier == pl) {
                if (GLO_players[pl].p_status != PALIVE) {

                    /* The guy carrying it died, set the flag's flags :) */
                    GLO_flags[i].f_carrier = -1;
                    GLO_players[pl].p_carryflag = -1;
                    GLO_flags[i].f_moved = 1;
                } else {
                    /* The guy is carrying this and still alive, move the flag to where he is */
                    GLO_flags[i].f_x = GLO_players[pl].p_x;
                    GLO_flags[i].f_y = GLO_players[pl].p_y;
                    GLO_flags[i].f_moved = 1;

                    /* Check if this guy has brought the flag back to his base */
                    if (((GLO_flags[i].f_x >> 4) / GRIDWIDTH) == GLO_bases[GLO_players[pl].p_team].b_x &&
                        ((GLO_flags[i].f_y >> 4) / GRIDHEIGHT) == GLO_bases[GLO_players[pl].p_team].b_y) {

                        /* Yup!  Increment the number of flags at his base */
                        GLO_bases[GLO_players[pl].p_team].b_numflags++;

                        if (GLO_bases[GLO_players[pl].p_team].b_numflags >= GLO_num_flags[oteam]) {
                            /* THEY WON! (got all the enemy flags to their base)
                               do the game over stuff. */
                            newplayer_game_over(GLO_players[pl].p_team);
                        } else {
                            /* didn't win yet, set this flag's position to the base,
                               make in uncarryable, and tell the world it's been captured */
                            GLO_flags[i].f_x = (GLO_bases[GLO_players[pl].p_team].b_x * GRIDWIDTH +
                                                GRIDWIDTH / 2 + GLO_bases[GLO_players[pl].p_team].b_numflags * 2) << 4;
                            GLO_flags[i].f_y = (GLO_bases[GLO_players[pl].p_team].b_y * GRIDWIDTH + GRIDWIDTH / 2) << 4;

                            GLO_flags[i].f_carrier = -2;        /* -2 means it can't be moved any more. */
                            GLO_players[pl].p_carryflag = -1;
                            announce_flag(pl);
                        }
                    }
                }
            } else if (GLO_flags[i].f_carrier == -1) {
                /* this flag is not carried by anyone */
                if (GLO_players[pl].p_carryflag < 0 &&
                (ABS(GLO_players[pl].p_x - GLO_flags[i].f_x) >> 4) < 8 &&
                (ABS(GLO_players[pl].p_y - GLO_flags[i].f_y) >> 4) < 8) {

                    /* but this guy is close enough to pick it up.  Do so. */
                    GLO_flags[i].f_carrier = pl;
                    GLO_flags[i].f_x = GLO_players[pl].p_x;
                    GLO_flags[i].f_y = GLO_players[pl].p_y;
                    GLO_flags[i].f_moved = 1;
                    GLO_players[pl].p_carryflag = i;
                }
            }
        }
    }
}

/* check_mines(pl)

   Check the positions of all mines relative to player #pl, take appropriate actions.

   Return: none
   */
static void check_mines(int pl)
{
    int i;
    char buf[100];
    int mapx, mapy;

    /* loop through every mine */
    for (i = 0; i < MAXPLAYERS * MINESPERPLAYER; i++) {
        if (GLO_players[i / MINESPERPLAYER].p_status == PEMPTY || i / MINESPERPLAYER == pl) {
            /* skip over mines this player owns, they can't hurt him */
            i += MINESPERPLAYER - 1;
            continue;
        }
        if (GLO_mines[i].mi_status == MILIVE) {
            /* if this mine is active */
            if ((ABS(GLO_mines[i].mi_x - GLO_players[pl].p_x) >> 4) < 15 &&
              (ABS(GLO_mines[i].mi_y - GLO_players[pl].p_y) >> 4) < 15) {
                /* DOH! You ran into a mine, stupid! */
                mapx = (GLO_mines[i].mi_x >> 4) / GRIDWIDTH;
                mapy = (GLO_mines[i].mi_y >> 4) / GRIDHEIGHT;

                if (GLO_map[mapx][mapy] != T_CRATER && GLO_map[mapx][mapy] != T_BASE1 && GLO_map[mapx][mapy] != T_BASE2) {

                    /* change the mine's old map square to a crater */
                    map_crater_square(mapx, mapy);
                }
                /* Set the mine's status to empty */
                GLO_mines[i].mi_status = MIEMPTY;

                /* mark it as needing to be updated to everyone */
                GLO_mines[i].mi_updated = 0;

                if (GLO_players[pl].p_damage + MINEDAMAGE >= MAXDAMAGE) {
                    /* If this guy died as a result of hitting the mine, tell everyone */
                    sprintf(buf, "%s(%c) killed by %s(%c) [mine]", GLO_players[pl].p_name, GLO_players[pl].p_char,
                            GLO_players[i / MINESPERPLAYER].p_name, GLO_players[i / MINESPERPLAYER].p_char);
                    NET_send_message_to_all(buf, M_SRV);

                    /* Give a kill to the guy who laid the mine */
                    GLO_players[i / MINESPERPLAYER].p_kills += (100 + GLO_players[pl].p_kills / 10);

                    /* Take all the kills away from the guy who died */
                    GLO_players[pl].p_kills = 0;

                    /* Set both player's statistics to be updated to everyone */
                    set_update_flags(i / MINESPERPLAYER);
                    set_update_flags(pl);

                    /* kill the player who died */
                    kill_player(pl);
                    return;     /* stop checking for mines */
                } else {
                    /* He's still alive, just add the mine's damage to his damage */
                    GLO_players[pl].p_damage += MINEDAMAGE;
                }

                /* Set the mine to inactive */
                GLO_mines[i].mi_status = MIEMPTY;
                GLO_mines[i].mi_updated = 0;
            } else if (GLO_shells[pl].s_status == SALIVE &&
                       ((ABS(GLO_mines[i].mi_x - GLO_shells[pl].s_x) < (6 << 4) &&
                ABS(GLO_mines[i].mi_y - GLO_shells[pl].s_y) < (6 << 4)) ||
                        (ABS(GLO_mines[i].mi_x -
                             (GLO_shells[pl].s_x - ((GLO_Cos[GLO_shells[pl].s_dir] * SHELLSPEED) / 2))) < (6 << 4) &&
                         ABS(GLO_mines[i].mi_y -
                             (GLO_shells[pl].s_y - ((GLO_Sin[GLO_shells[pl].s_dir] * SHELLSPEED) / 2))) < (6 << 4)))) {
                /* Mines can be shot!  This guy's shell just hit one */
                mapx = (GLO_mines[i].mi_x >> 4) / GRIDWIDTH;
                mapy = (GLO_mines[i].mi_y >> 4) / GRIDHEIGHT;

                if (GLO_map[mapx][mapy] != T_CRATER && GLO_map[mapx][mapy] != T_BASE1 && GLO_map[mapx][mapy] != T_BASE2) {
                    map_crater_square(mapx, mapy);
                }
                /* set the mine to inactive */
                GLO_mines[i].mi_status = MIEMPTY;

                /* and make sure everyone hears about it */
                GLO_mines[i].mi_updated = 0;

                /* tell everyone this shell is exploded, so their clients
                   can draw pretty explosions */
                GLO_shells[pl].s_status = SEXPLODE;
                NET_update_shell(&GLO_shells[pl]);
            }
        }
    }
}

/* tick_do_tick()

   The heart of the tick module.  Loops through every player checking all relative objects,
   movement, etc.

   This is the only tick function visible outside the tick module.

   Return: none
   */
void tick_do_tick()
{
    int i, j, oldx, oldy, ix, iy;
    player *p;
    shell *s;
    char buf[80];

    /* loop through every player */
    for (i = 0; i < MAXPLAYERS; i++) {

        /* p points to the player under consideration */
        p = &GLO_players[i];

        /* This player is an empty slot, go on */
        if (p->p_status == PEMPTY)
            continue;

        /* cool, he's alive, let's do stuff */
        if (p->p_status == PALIVE) {
            int maxspeed;
            int nspeed;

            /* See if he's trying to turn and move */
            do_steering(p);

            /* remember his current position so we can check to make sure his move
               was legal later */
            oldx = p->p_x;
            oldy = p->p_y;

            /* damage will lower his maximum speed */
            maxspeed = (((MAXDAMAGE - p->p_damage) * MAXSPEED) / MAXDAMAGE) + 1;
            if (maxspeed > MAXSPEED) {
                maxspeed = MAXSPEED;
            }
            if (p->p_speed > maxspeed) {
                p->p_speed = maxspeed;
            }
            if (p->p_desspeed > maxspeed) {
                p->p_desspeed = maxspeed;
            }
            nspeed = (p->p_speed * 9) / MAXSPEED;

            /* Take away some gas */
            p->p_fuel -= ABS(nspeed) * FUELRATE;
            if (p->p_fuel < 0) {
                /* DOH! He ran out of fuel! Tell everyone how much he sucks, then kill him. */
                sprintf(buf, "%s(%c) ran out of fuel and died, what a loser!", p->p_name, p->p_char);
                NET_send_message_to_all(buf, M_SRV);

                /* take away his hard won kills */
                p->p_kills = 0;

                /* Make sure everyone hears about his new statistics */
                set_update_flags(p->p_num);

                /* and kill him */
                kill_player(p->p_num);

                /* He's dead, move on to the next player */
                continue;
            }
            /* move him to his new location */
            p->p_x += GLO_Cos[p->p_dir] * (double) p->p_speed * terrain_mod(p->p_x, p->p_y);
            p->p_y += GLO_Sin[p->p_dir] * (double) p->p_speed * terrain_mod(p->p_x, p->p_y);

            /* ok he may have moved somewhere, make sure he didn't hit a wall
               or something */
            check_move(p, oldx, oldy);

            /* If he crossed from one map square to another, check if he's now hidden (or not)*/
            if (mapx[i] != MAPX(p->p_x) || mapy[i] != MAPY(p->p_y)) {
                check_hidden(p);
                mapx[i] = MAPX(p->p_x);
                mapy[i] = MAPY(p->p_y);
            }
            /* Tanks repair damage slowly, see if it's time to repair a bit */
            if (!(GLO_counter % REPAIRRATE) && (p->p_damage > 0)) {
                p->p_damage--;
            }
            /* If he's on his own base, repair, refuel, reload */
            if ((p->p_x >> 4) / GRIDWIDTH == GLO_bases[p->p_team].b_x &&
                (p->p_y >> 4) / GRIDHEIGHT == GLO_bases[p->p_team].b_y) {

                if (p->p_fuel < MAXFUEL) {
                    p->p_fuel += BASEREFUELRATE;
                    if (p->p_fuel > MAXFUEL)
                        p->p_fuel = MAXFUEL;
                }
                if (p->p_ammo < MAXAMMO && !(GLO_counter % BASERELOADTICKS)) {
                    p->p_ammo++;
                }
                if (p->p_damage > 0 && !(GLO_counter % (REPAIRRATE / 2))) {
                    p->p_damage--;
                }
            }
            /* Check all mines relative to this player */
            check_mines(i);
        } else if (p->p_expfuse > 0) {
            /* This guy is in the process of dying */
            p->p_expfuse--;

            if (!p->p_expfuse) {
                /* He's done dying */
                if (GLO_shells[i * SHELLSPERPLAYER].s_status == SALIVE) {
                    /* unless he still has an active shell */
                    p->p_expfuse = 1;
                } else {
                    /* Ok he's really dead, send him to the motd */
                    p->p_status = POUTFIT;

                    /* Set his x,y to his base for when he reenters */
                    newplayer_enter_player_xy(i);

                    /* speed, desired speed to 0 */
                    p->p_speed = 0;
                    p->p_desspeed = 0;

                    /* Tell everyone about his newfound lack of life */
                    NET_update_all_players(i);
                    for (j = 0; j < MAXPLAYERS; j++) {
                        if (GLO_players[j].p_status == PALIVE) {
                            NET_update_one_player(j, i);
                        }
                    }
                }
            }
        }
        /* Check all flags(the kind on a pole) relative to this player */
        check_flags(i);
    }

    /* Ok, now loop through all the shells */
    for (i = 0; i < MAXPLAYERS * SHELLSPERPLAYER; i++) {

        /* s is a shortcut to the current shell */
        s = &GLO_shells[i];

        if (s->s_status == SALIVE) {
            /* This shell is live and moving, do stuff */
            s->s_fuse--;        /* alive for this much longer */

            if (s->s_fuse <= 0) {
                /* Its fuse just ran out, set it to inactive, tell everyone about it */
                s->s_status = SDEAD;
                NET_update_shell(s);
            } else {
                /* ix, iy are the change in position for this shell */
                ix = GLO_Cos[s->s_dir] * (double) SHELLSPEED;
                iy = GLO_Sin[s->s_dir] * (double) SHELLSPEED;

                /* move it */
                s->s_x += ix;
                s->s_y += iy;

                /* check this shell relative to every player, see if it hit */
                for (j = 0; j < MAXPLAYERS; j++) {
                    /* Check if the player is
                       1) alive
                       2) on the other team
                       3) close enough to be hit
                       */
                    if ((GLO_players[j].p_status == PALIVE) &&
                        (GLO_players[j].p_team != GLO_players[s->s_owner].p_team) &&
                        ((((ABS(s->s_x - GLO_players[j].p_x) >> 4) < 8) &&
                        ((ABS(s->s_y - GLO_players[j].p_y) >> 4) < 8)) ||
                         (((ABS((s->s_x - ix / 2) - GLO_players[j].p_x) >> 4) < 8) &&
                          ((ABS((s->s_y - iy / 2) - GLO_players[j].p_y) >> 4) < 8)))) {

                        /* all conditions met, this guy's been hit! */
                        if (GLO_players[j].p_damage + SHELLDAMAGE >= MAXDAMAGE) {
                            /* And he took enough damage to kill him, tell the world */
                            sprintf(buf, "%s(%c) killed by %s(%c)", GLO_players[j].p_name, GLO_players[j].p_char,
                                    GLO_players[s->s_owner].p_name, GLO_players[s->s_owner].p_char);
                            NET_send_message_to_all(buf, M_SRV);

                            /* give him a loss, give the guy who killed him a win and a kill */
                            GLO_players[j].p_losses++;
                            GLO_players[s->s_owner].p_wins++;
                            GLO_players[s->s_owner].p_kills += (100 + GLO_players[j].p_kills / 10);

                            /* set the kills of the guy who died to 0 */
                            GLO_players[j].p_kills = 0;

                            /* Set the statistics of both players to be updated */
                            set_update_flags(s->s_owner);
                            set_update_flags(j);

                            /* and kill the guy who died */
                            kill_player(j);
                        } else {
                            /* Hit, but not dead.  Up his damage. */
                            GLO_players[j].p_damage += SHELLDAMAGE;

                            /* Change his course to some random direction (not more than 90
                               degrees from his present course) */
                            GLO_players[j].p_desdir += (int) ((rand() % 128) - 64);
                            GLO_players[j].p_dir = GLO_players[j].p_desdir;

                            /* and knock him back a bit */
                            GLO_players[j].p_x += (SHELLSPEED / 2) * GLO_Cos[s->s_dir];
                            GLO_players[j].p_y += (SHELLSPEED / 2) * GLO_Sin[s->s_dir];
                        }
                        /* This shell's exploding, tell everyone's client to draw it that way */
                        s->s_status = SEXPLODE;
                        NET_update_shell(s);
                    }
                }
                if (s->s_status == SALIVE) {    /* didn't hit anyone, check against map */
                    /* Get the square it's in */
                    int mapx = (s->s_x >> 4) / GRIDWIDTH, mapy = (s->s_y >> 4) / GRIDHEIGHT;
                    if (map_iswall(GLO_map[mapx][mapy])) {
                        /* This shell hit a wall, add one to that wall's damage count */
                        GLO_mapdamage[mapx][mapy]++;

                        if (GLO_mapdamage[mapx][mapy] >= WALLDESTROY) {
                            /* and if it's too damaged, crater the square */
                            map_crater_square(mapx, mapy);
                        }
                        /* tell everyone to draw an exploding shell */
                        s->s_status = SEXPLODE;
                        NET_update_shell(s);
                    }
                }
            }
        } else {

            /* This shell is inactive, possibly exploding */
            if (s->s_fuse > 0) {

                /* must be exploding */
                s->s_fuse--;

                if (s->s_fuse == 0) {
                    /* done exploding, set it to inactive */
                    s->s_status = SDEAD;
                }
            }
        }
    }
}
