
/* socket.c for firepower

  Includes publically accesible network functions (NET_*)
  all direct handling of sockets is done here.
  Also takes care of the timers, and calls tick_do_tick

These functions are part of the API:

void NET_listenSocket(int port);
        Set up the initial sockets

void NET_send_map(int pl);
        Send the map to the specified player.

void NET_update_shell(shell *s);
        Tell everyone about a change in status for a shell

void NET_send_message_to_indiv(char *msg, int from, int pl);
        Send a text message to a particular player

void NET_warning(char *msg, int pl);
        Send a special server message to a particular player

void NET_send_message_to_team(char *msg, int team, int from);
        Send a text message to everyone on one team

void NET_send_message_to_all(char *msg, int from);
        Send a text message to everyone

void NET_send_playerdata(int pl, int datapl);
        Send data about one player (name, login) to another

void NET_update_one_player(int pl, int updpl);
        Send player status (alive, dead, etc) to one player

void NET_update_all_players(int pl);
        Send everyone else's status to one player

void NET_update_mapsquare(int x, int y);
        Send a change in a particular bit of terrain to everyone

void NET_update_mapsquare_value(int x, int y, int value);
        Similar to update_mapsquare, but use value instead of what's in the global array

void NET_checkSockets(void);
        The main loop, should be called repeatedly until the server is killed.
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/errno.h>
#include <sys/ioctl.h>
#include <sys/file.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#ifdef _AIX
#include <sys/select.h>
#endif
#include "defs.h"
#include "global.h"
#include "packet.h"
#include "struct.h"

#include "newplayer.h"
#include "log.h"
#include "map.h"
#include "tick.h"
#include "life.h"
#include "commands.h"
#include "tick.h"
#include "life.h"

/* how many bytes to buffer on udp sockets */
#define UDPBUFSIZE 1024

/* sockets to listen for new connections and player list requests */
static int listenSock = -1, playerListSock = -1;

/* sockets for each of the players, and their IP addresses */
static int playerSock[MAXPLAYERS], playerUdpSock[MAXPLAYERS], remoteaddr[MAXPLAYERS];

/* buffers for the UDP sockets */
static char udpBuf[MAXPLAYERS][UDPBUFSIZE];
static int udpBufPos[MAXPLAYERS];

/* how long this player's been dead (at the motd) */
static int deadcounter[MAXPLAYERS];

/* each players client version */
static int clientvers[MAXPLAYERS];

static void handle_UdpReq(char *, int);
static int connUdpConn(int pl, int udpClientPort);
static int closeUdpConn(int pl);

/*
 * listenSocket(port)
 *
 * Open up the socket that listens for new player connections
 *
 * Return: none
 */
void NET_listenSocket(int port)
{
    struct sockaddr_in addr;
    int i;

    /* create the socket, exit if we can't get one */
    if ((listenSock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
        perror("socket");
        exit(0);
    }
    /* bind to the specified port.  Allow connections from any address */
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = INADDR_ANY;
    addr.sin_port = htons(port);
    if (bind(listenSock, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
        sleep(10);
        if (bind(listenSock, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
            perror("bind");
            close(listenSock);
            exit(0);
        }
    }
    /* listen for new connections, only allow 1 pending */
    if (listen(listenSock, 1) < 0) {
        perror("listen");
        exit(0);
    }
    /* everything should be non-blocking so the server never waits on
       a particular connection */
    fcntl(listenSock, F_SETFL, O_NONBLOCK);

    /* Set up the player list socket on the main port - 1.
       This is so people can telnet to this port just to see who's
       playing without actually connecting to the game.
       */
    if ((playerListSock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
        perror("socket");
        exit(0);
    }
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = INADDR_ANY;
    addr.sin_port = htons(port - 1);

    if (bind(playerListSock, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
        sleep(10);
        if (bind(playerListSock, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
            perror("bind");
            close(playerListSock);
            exit(0);
        }
    }
    if (listen(playerListSock, 1) < 0) {
        perror("listen");
        exit(0);
    }
    fcntl(playerListSock, F_SETFL, O_NONBLOCK);
    printf("Listening on %d\n", port);
    printf("Player list on %d\n", port - 1);

    /* mark all player sockets unused */
    for (i = 0; i < MAXPLAYERS; i++) {
        playerSock[i] = -1;
        playerUdpSock[i] = -1;
    }
}

/*
 * checkSocket(pl)
 *
 * Get the address of the machine at the other end of a connection
 *
 * pl: player number to check
 *
 * Return: none
 */
static void checkSocket(int pl)
{
    struct sockaddr_in sin;
    int length;

    /* getpeername puts the remote address in sin */
    length = sizeof(sin);
    if (getpeername(playerSock[pl], (struct sockaddr *) &sin, &length) < 0) {
        /* A bad thing. */
        return;
    }
    /* store all the addresses in an array */
    remoteaddr[pl] = sin.sin_addr.s_addr;
}

/*
 * flush_buffer(pl)
 *
 * All data going to each player is buffered up before sending.
 * This is so we only send one big UDP packet instead of lots of little ones
 *
 * Return: none
 */
static void flush_buffer(int pl)
{
    if (udpBufPos[pl] > 0) {
        if (write(playerUdpSock[pl], udpBuf[pl], udpBufPos[pl]) != udpBufPos[pl]) {
            /* If we can't write to this player, assume their connection is dead */
            printf("Bad write to player %d\n", pl);
            newplayer_remove_player(pl);
            close(playerUdpSock[pl]);
            playerUdpSock[pl] = -1;
        }
        udpBufPos[pl] = 0;
    }
}

/*
 * add_to_buffer(pl, pack, len)
 *
 * add to the outgoing buffer for a player.  This should be used in place of a write
 * command for most packet types.
 *
 * Return: none
 */

static void add_to_buffer(int pl,
                          char *pack,
                          int len)
{
    if (playerUdpSock[pl] < 0) {
        /* If this player has only TCP open, just send the packet */
        if (write(playerSock[pl], pack, len) != len) {
            printf("Bad write to player %d\n", pl);
            newplayer_remove_player(pl);
            close(playerSock[pl]);
            playerSock[pl] = -1;
        }
    } else {
        /* if this packet would overflow the buffer, flush it first. */
        if (udpBufPos[pl] + len >= UDPBUFSIZE) {
            flush_buffer(pl);
        }
        memcpy(udpBuf[pl] + udpBufPos[pl], pack, len);

        udpBufPos[pl] += len;
    }
}

/*
 * sendPacket(pack, pl)
 *
 * Take the packet, determine the type and length from the packet itself, and
 * send it to the appropriate socket.  Some packets get sent by UDP, others
 * need to be guaranteed delivery, so they go by TCP.
 *
 * Return: 0 on failure, 1 on success
 */

static int sendPacket(char *pack,
                      int pl)
{
    int len, w;
    int asock;

    /* default to UDP socket if open.  Particular packet types will select
       the TCP socket instead. */
    asock = (playerUdpSock[pl] >= 0) ? playerUdpSock[pl] : playerSock[pl];
    switch (pack[0]) {
    case S_MAPINFO:
        len = sizeof(s_mapinfo);
        asock = playerSock[pl]; /* The Map is essential, send TCP */
        break;
    case S_MAPSQUARE:
        len = sizeof(s_mapsquare);
        asock = playerSock[pl];
        break;
    case S_PERSONAL:
        len = sizeof(s_personal);
        break;
    case S_MESSAGE:
        len = sizeof(s_message);
        asock = playerSock[pl]; /* Messages are not time critical, send TCP */
        break;
    case S_PLAYERDATA:
        len = sizeof(s_playerdata);
        break;
    case S_PLAYERSTATS:
        len = sizeof(s_playerstats);
        break;
    case S_FLAG:
        len = sizeof(s_flag);
        break;
    case S_MOTDLINE:
        len = sizeof(s_motdline);
        asock = playerSock[pl]; /* MOTD packets are not time critical */
        break;
    case S_MINE:
        len = sizeof(s_mine);
        break;
    case S_SHELL:
    case S_SHELLINFO:
    case S_POSITIONS:
    case S_PING:
        len = 4;
        break;
    case S_PLAYERINFO:
    case S_LOGIN:
    case S_TEAMOK:
        len = 4;
        asock = playerSock[pl];
        break;
    default:
        printf("sendPacket: Tried to send unknown packet type %d\n", pack[0]);
        return 0;
    }
    if (asock >= 0) {
        if (asock == playerUdpSock[pl]) {
            /* UDP packets are buffered up for sending so as not to waste
               bandwidth and CPU time sending many small ones
             */
            add_to_buffer(pl, pack, len);
        } else {
            /* TCP packets are not buffered, on the assumption that there
               are relatively few (aside from ones sent when first connecting,
               which are relatively large anyway)
             */
            if (GLO_debug > 2) {
                printf("Writing packet %d\n", pack[0]);
            }
            if ((w = write(asock, pack, len)) != len) {
                /* the socket is full, no room to write this packet.
                   Potential problem here losing packets, but on the
                   systems it's been run on by me (Linux, OSF/1)
                   it hasn't caused any difficulties */
                if (errno == EAGAIN) {
                    perror("sendPacket");
                    return 0;
                }
                /* write failed for some other reason, assume the connection
                   is dead. */
                printf("Bad write to player %d\n", pl);
                newplayer_remove_player(pl);
                close(playerSock[pl]);
                playerSock[pl] = -1;
                return 0;
            }
            if (GLO_debug > 2) {
                printf("Wrote %d bytes to player %d\n", w, pl);
            }
        }
    }
    return 1;
}

/* sendUdpPacket(pack, pl)
 *
 * Mostly obsolete.  Now used only during handling of initial UDP requests.
 * New code should use add_to_buffer instead.
 *
 * Return: none
 */

static void sendUdpPacket(char *pack,
                          int pl)
{
    int len, w;

    if (playerSock[pl] >= 0) {
        if (pack[0] != S_UDPREPLY) {
            printf("sendUdpPacket: bad packet type %d\n", pack[0]);
            return;
        }
        len = sizeof(s_udpreply);

        if (GLO_debug > 2) {
            printf("Writing UDP packet %d\n", pack[0]);
        }
        if ((w = write(playerUdpSock[pl], pack, len)) != len) {
            printf("Bad write to player %d\n", pl);
            newplayer_remove_player(pl);
            close(playerUdpSock[pl]);
            playerUdpSock[pl] = -1;
        }
        if (GLO_debug > 2) {
            printf("Wrote %d bytes to player %d\n", w, pl);
        }
    }
}

/*
 * sendRaw(data, len, pl)
 *
 * Send a chunk of raw data to a player.  This is only used by send_map to send
 * the map body.
 *
 * Return: none
 */

static void sendRaw(void *data, int len, int pl)
{
    int w;

    if (playerSock[pl] >= 0) {
        if ((w = write(playerSock[pl], data, len)) != len) {
            /* only used by send_map, failure here is fatal, kill the guy. */
            printf("Bad write to player %d\n", pl);
            newplayer_remove_player(pl);
            close(playerSock[pl]);
            playerSock[pl] = -1;
        }
        if (GLO_debug > 2) {
            printf("Wrote %d bytes to player %d\n", w, pl);
        }
    }
}

/* send_map(pl)
 *
 * Send the current map description to the player.  This is normally called
 * when a player first logs on, or when the game is won and the map changes
 *
 * Return: none
 */

void NET_send_map(int pl)
{
    s_mapinfo pack;
    int len;
    char *sendmap;
    int x, y;

    /* Every send function has to fill in the packet type.
       Every packet's first byte holds the numerical type. */

    pack.type = S_MAPINFO;

    /* Send width, height, and name to the client */
    pack.width = GLO_map_info.m_width;
    pack.height = GLO_map_info.m_height;
    strcpy(pack.name, GLO_map_info.m_name);

    /* Send the map info packet.  The client will be expecting the raw map data
       immediately following this, but technically it's not part of the same packet
       */
    if (!sendPacket((char *) &pack, pl) && playerSock[pl] >= 0) {
        printf("Bad write to player %d\n", pl);
        newplayer_remove_player(pl);
        close(playerSock[pl]);
        playerSock[pl] = -1;
        return;
    }
    /* create a packed array for the map data.  The internal array is 256*256,
       but most maps are much smaller than that, so we don't need to send
       the whole 64k to everyone. */
    len = GLO_map_info.m_width * GLO_map_info.m_height;
    while ((len % 4)) {         /* need to make sure this is long word aligned */
        ++len;
    }
    sendmap = malloc(len);

    /* copy the map into the new arrary */
    for (x = 0; x < GLO_map_info.m_width; x++) {
        for (y = 0; y < GLO_map_info.m_height; y++) {
            sendmap[y * GLO_map_info.m_width + x] = GLO_map[x][y];
        }
    }

    /* send the raw data and free up the memory we used */
    sendRaw(sendmap, len, pl);
    free(sendmap);
}

/* send_map_positions(pl)
 *
 * send the positions of players who are visible but not on the combat display
 * IE they should appear only on the map display
 *
 * Return: none
 */
static void send_map_positions(int pl)
{
    int i, np, dx, dy, cx, cy;
    player *p, *me;
    s_positions pack;
    s_oneposition pos[MAXPLAYERS];

    /* every send function must fill in the type */
    pack.type = S_POSITIONS;

    /* pack.num gives the number of player positions contained in this packet.
       It must be set correctly so the client will know how large this packet is */
    pack.num = 0;
    np = 0;

    /* set me to the player being sent to so we can pretend to see things from
       his point of view more easily */
    me = &GLO_players[pl];

    /* Determine the upper left corner of this player's combat display
       so we can decide if each other player is on his display or not */
    cx = (me->p_x >> 4) - WINWIDTH / 2;
    cy = (me->p_y >> 4) - WINHEIGHT / 2;
    if (cx < 0) {
        cx = 0;
    } else if (cx > GRIDWIDTH * GLO_map_info.m_width - WINWIDTH) {
        cx = GRIDWIDTH * GLO_map_info.m_width - WINWIDTH;
    }
    if (cy < 0) {
        cy = 0;
    } else if (cy > GRIDHEIGHT * GLO_map_info.m_height - WINHEIGHT) {
        cy = GRIDHEIGHT * GLO_map_info.m_height - WINHEIGHT;
    }
    /* search through every active player checking poisitions against the player
       we're sending to */
    for (i = 0; i < MAXPLAYERS; i++) {
        /* don't send this player his own position here.  Your own position
           gets sent in a special packet with stuff like fuel and ammo */
        if (i == pl) {
            continue;
        }
        p = &GLO_players[i];

        /* If this guy is active */
        if (p->p_status != PEMPTY) {
            dx = (p->p_x >> 4) - cx;
            dy = (p->p_y >> 4) - cy;

            /* if his display x and y relative to the player being sent to
               are off the edge of the combat screen, then this guy gets sent
               as a map position */
            if (!(dx >= 0 && dx <= (WINWIDTH + GRIDWIDTH) &&
                  dy >= 0 && dy <= (WINHEIGHT + GRIDHEIGHT))) {
                pos[np].num = i;

                /* If he's hiding under a tree, send his coordinates as way
                   off in the corner (off the map entirely)
                   This is so whatever position he was in before gets erased.
                   If we didn't send new coordinates when he moved under a tree,
                   the client would keep drawing him at his last known position. */
                if (p->p_flags & PFHIDDEN) {
                    pos[np].lx = 255;
                    pos[np].ly = 255;
                } else {
                    pos[np].lx = (UCHAR) ((p->p_x >> 4) / GRIDWIDTH);
                    pos[np].ly = (UCHAR) ((p->p_y >> 4) / GRIDHEIGHT);
                }
                /* fill in the direction the player is facing.
                   Not really important for map positions, but there
                   was an extra byte needed to keep everything long word aligned,
                   so we might as well use it */
                pos[np].dir = p->p_dir;
                pack.num++;
                np++;
            }
        }
    }

    /* combine the packet header and position structures into one big packet
       and send it off */
    if (pack.num > 0) {
        int len;
        char buf[1024];

        pack.num |= 0x80;
        memcpy(buf, &pack, sizeof(pack));
        memcpy(buf + sizeof(pack), pos, np * sizeof(s_oneposition));
        len = sizeof(pack) + np * sizeof(s_oneposition);
        add_to_buffer(pl, buf, len);
    }
}

/* send_positions(pl, cx, cy)
 *
 * send the positions of any players visible on the player's combat display.
 * cx and cy give the upper left corner of this player's display.
 *
 * Return: none
 */

static void send_positions(int pl,
                           int cx,
                           int cy)
{
    int i, np, dx, dy;
    player *p, *me;
    s_positions pack;
    s_oneposition pos[MAXPLAYERS];

    /* fill int the packet type */
    pack.type = S_POSITIONS;

    /* keep track of how many players are being sent in this packet */
    pack.num = 0;
    np = 0;

    /* use me to be the player being sent to */
    me = &GLO_players[pl];

    for (i = 0; i < MAXPLAYERS; i++) {
        if (i == pl) {
            continue;
        }
        p = &GLO_players[i];

        /* If this guy is active */
        if (p->p_status != PEMPTY) {
            dx = (p->p_x >> 4) - cx;
            dy = (p->p_y >> 4) - cy;

            /* If his display coords are on the combat display of the player
               being sent to */
            if (dx >= 0 && dx <= (WINWIDTH + GRIDWIDTH) &&
                dy >= 0 && dy <= (WINHEIGHT + GRIDHEIGHT)) {
                pos[np].num = i;

                /* If this guy is within the combat display, but hidden under a
                   tree and outside the range at which he would become
                   visible, send his coords as off the edge of the display */
                if ((p->p_flags & PFHIDDEN) &&
                    (hypot((double) (p->p_x - GLO_players[pl].p_x),
                           (double) (p->p_y - GLO_players[pl].p_y)) >
                     ((GRIDWIDTH * HIDEDISTANCE) << 4))) {
                    dx = 511;
                    dy = 511;
                }
                if (dx > 255) {
                    /* Coordinates can be in the range 0-WINWIDTH.  Because WINWIDTH
                       is generally larger than 256, we need more than one byte
                       to hold the coordinates here. However, using a word
                       (16 bits) for each would mean making the position packet
                       at least 8 bytes to hold an x, y, and direction value.  So
                       instead, we take two bits out of the pos.num entry and make
                       our x and y be 9 bit values (0-511) which is enough to hold
                       the full range of positions and still keep the position
                       packet down to only 4 bytes */

                    pos[np].num |= 0x80;
                    dx -= 256;
                }
                if (dy > 255) {
                    pos[np].num |= 0x40;
                    dy -= 256;
                }
                /* lx, ly hold the Lower part of the position, ie the low 8 bits */
                pos[np].lx = (UCHAR) dx;
                pos[np].ly = (UCHAR) dy;
                pos[np].dir = p->p_dir;
                pack.num++;
                np++;
            }
        }
    }

    /* Copy the packet header and position packets into one large buffer and send it */
    if (pack.num > 0) {
        int len;
        char buf[1024];

        memcpy(buf, &pack, sizeof(pack));
        memcpy(buf + sizeof(pack), pos, np * sizeof(s_oneposition));
        len = sizeof(pack) + np * sizeof(s_oneposition);
        add_to_buffer(pl, buf, len);
    }
    /* Update map positions every 10 ticks (once per second */
    if (!((GLO_counter - pl) % 10)) {
        send_map_positions(pl);
    }
}

/* send_one_shell (pl, shell, cx, cy)
 *
 * Check to see if this shell is visible to this player, and send
 * the shell's position if so.
 *
 * Return: none
 */
static void send_one_shell(int pl,
                           int shl,
                           int cx,
                           int cy)
{
    int dx, dy;
    s_shell pack;
    shell *s = &GLO_shells[shl];

    dx = (s->s_x >> 4) - cx;
    dy = (s->s_y >> 4) - cy;
    if (dx >= 0 && dx <= WINWIDTH &&
        dy >= 0 && dy <= WINHEIGHT) {
        /* ok, it's visible, fill in the packet type, number, and position */
        pack.type = S_SHELL;
        pack.num = shl / SHELLSPERPLAYER;

        /* See comment in send_positions as to why we need 9-bit values here */
        if (dx > 255) {
            pack.num |= 0x80;
            dx -= 256;
        }
        if (dy > 255) {
            pack.num |= 0x40;
            dy -= 256;
        }
        pack.lx = dx;
        pack.ly = dy;

        sendPacket((char *) &pack, pl);
    }
}

/* send_shells(pl, cx, cy)
 *
 * for every active shell, update this player as to its position
 * note that send_one_shell determines if the shell is visible or not.
 * This just checks if the shell is active and calls send_one_shell.
 *
 * Return: none
 */
static void send_shells(int pl,
                        int cx,
                        int cy)
{
    int i;

    for (i = 0; i < SHELLSPERPLAYER * MAXPLAYERS; i++) {
        if (GLO_shells[i].s_status != SDEAD) {
            send_one_shell(pl, i, cx, cy);
        }
    }
}

/* send_shellinfo(shell, pl)
 *
 * update this player as to the status (not position) of a particular shell
 * This is so when a shell dies (reaches max range, or hits something) we can
 * tell the client to erase it.
 *
 * Return: none
 */

static void send_shellinfo(int shell,
                           int pl)
{
    s_shellinfo pack;

    pack.type = S_SHELLINFO;
    pack.num = shell;
    pack.status = GLO_shells[shell].s_status;
    sendPacket((char *) &pack, pl);
}

/* NET_update_shell(s)
 *
 * This is called with a pointer a shell whenever the status of that shell
 * changes.  Updates the shell's status to any player to whom it should be visible
 *
 * Return: None
 */
void NET_update_shell(shell * s)
{
    int i, cx, cy, dx, dy;
    player *me;

    /* Client sets SDEAD on any torp off screen, only send shellinfo if the torp is
       on the client's display. Client sets SALIVE for any shell position packet,
       so only send if the status is SDEAD.*/

    for (i = 0; i < MAXPLAYERS; i++) {
        me = &GLO_players[i];
        if (me->p_status != PEMPTY) {
            /* determine the corner of this player's combat display */
            cx = (me->p_x >> 4) - WINWIDTH / 2;
            cy = (me->p_y >> 4) - WINHEIGHT / 2;
            if (cx < 0) {
                cx = 0;
            } else if (cx > GRIDWIDTH * GLO_map_info.m_width - WINWIDTH) {
                cx = GRIDWIDTH * GLO_map_info.m_width - WINWIDTH;
            }
            if (cy < 0) {
                cy = 0;
            } else if (cy > GRIDHEIGHT * GLO_map_info.m_height - WINHEIGHT) {
                cy = GRIDHEIGHT * GLO_map_info.m_height - WINHEIGHT;
            }
            dx = (me->p_x >> 4) - cx;
            dy = (me->p_y >> 4) - cy;

            /* send the position if visible, and the status */
            if (dx >= 0 && dx <= WINWIDTH &&
                dy >= 0 && dy <= WINHEIGHT) {
                send_one_shell(i, s->s_num, cx, cy);
            }
            send_shellinfo(s->s_owner, i);
        }
    }
}

/* send_mines(pl)
 *
 * check all mines to see if they're in visible range of a player, and
 * that player hasn't seen them before.  If so, send them a nice little surprise.
 * Also, if the status has changed to empty (someone blew one up) send the
 * empty status to the player
 *
 * Return: none
 */
static void send_mines(int pl)
{
    s_mine pack;
    int i;
    int pflag = (1 << pl);

    pack.type = S_MINE;

    /* loop through every mine */
    for (i = 0; i < MAXPLAYERS * MINESPERPLAYER; i++) {
        /* If this player hasn't been updated since this mine changed status */
        if (!(GLO_mines[i].mi_updated & pflag)) {
            /* and the mine is within one square of the player */
            if ((ABS(GLO_mines[i].mi_x - GLO_players[pl].p_x) < (GRIDWIDTH << 4) &&
                 ABS(GLO_mines[i].mi_y - GLO_players[pl].p_y) < (GRIDHEIGHT << 4))) {

                /* then send them the mine.
                 * Just send the actual coordinates, mine packets are
                 * relatively rare, so there's no need to keep it down
                 * to 9 bit variables like player and shell positions
                 */
                pack.x = htons(GLO_mines[i].mi_x >> 4);
                pack.y = htons(GLO_mines[i].mi_y >> 4);
                pack.num = htons(i);
                pack.status = GLO_mines[i].mi_status;

                sendPacket((char *) &pack, pl);

                /* mark the mine as updated to this player */
                GLO_mines[i].mi_updated |= pflag;
            } else if (GLO_mines[i].mi_status == MIEMPTY) {
                /* also update this player whenever a mine's status changes
                   to empty (not present) whether it's visible or not */
                pack.status = GLO_mines[i].mi_status;
                pack.num = htons(i);

                sendPacket((char *) &pack, pl);
                /* mark the mine as updated to this player */
                GLO_mines[i].mi_updated |= pflag;
            }
        }
    }
}

/* NET_send_message_to_indiv(msg, from, pl)
 *
 * Send a text message (from another player or from the server)
 * to this player.
 *
 * Return: none
 */

void NET_send_message_to_indiv(char *msg,
                               int from,
                               int pl)
{
    s_message pack;
    char logm[100];

    /* Client adds a header on itself before printing, but for server message logging,
       we add it here too. */
    sprintf(logm, "%3d->%3d %s", from, pl, msg);
    log_log_message(logm);

    /* every send function must fill in the type */
    pack.type = S_MESSAGE;
    pack.from = from;
    pack.flags = 0;
    pack.to = pl;

    /* copy the message into the packet buffer */
    strncpy(pack.message, msg, 80);
    /* make sure it's null terminated - should have been before it got here,
       but better safe than sorry. */
    pack.message[79] = 0;

    sendPacket((char *) &pack, pl);
}

/* NET_warning(msg, pl)
 *
 * Send a special type of message to the player.  Ideally the client should
 * a special window for these messages, but presently it just prints them as
 * SRV-> messages.
 *
 * Return: none
 */
void NET_warning(char *msg,
                 int pl)
{
    NET_send_message_to_indiv(msg, M_SRV, pl);
}

/* NET_send_message_to_team(msg, team, from)
 *
 * Send a message to every player on a team.
 *
 * Return: none
 */
void NET_send_message_to_team(char *msg,
                              int team,
                              int from)
{
    s_message pack;
    int i;
    char logm[100];

    /* Add the header here for logging only.  The client adds it's own header. */
    sprintf(logm, "%3d->%s %s", from, (team == 0) ? "Blu" : "Red", msg);
    log_log_message(logm);

    /* every send function must fill in the type */
    pack.type = S_MESSAGE;
    pack.from = from;
    pack.flags = 0;

    /* there are two teams, numbered 0 and 1.  The message destinations are
       marked M_BLUE and M_RED.  M_BLUE is assumed to come first with M_RED
       coming right after it, so we can add the team requested to M_BLUE
       to get the proper destination.*/
    pack.to = team + M_BLUE;

    /* copy the actual text into the packet buffer */
    strncpy(pack.message, msg, 80);
    pack.message[79] = 0;

    /* send this packet to every player on the appropriate team */
    for (i = 0; i < MAXPLAYERS; i++) {
        if (GLO_players[i].p_status != PEMPTY && GLO_players[i].p_team == team) {
            sendPacket((char *) &pack, i);
        }
    }
}

/* NET_send_message_to_all(msg, from)
 *
 * Send a text message to every player in the game.
 *
 * Return: none
 */

void NET_send_message_to_all(char *msg,
                             int from)
{
    s_message pack;
    int i;
    char logm[100];

    /* Add a header here for logging purposes only */
    sprintf(logm, "%3d->All %s", from, msg);
    log_log_message(logm);

    /* every send function must fill in a type */
    pack.type = S_MESSAGE;
    pack.from = from;
    pack.to = M_ALL;
    pack.flags = 0;

    /* copy the actual text to the packet buffer */
    strncpy(pack.message, msg, 80);
    pack.message[79] = 0;

    /* And send it to every active player */
    for (i = 0; i < MAXPLAYERS; i++) {
        if (GLO_players[i].p_status != PEMPTY)
            sendPacket((char *) &pack, i);
    }
}

/* NET_send_playerdata(pl, datapl)
 *
 * send player data FOR player #datapl TO player #pl
 * Only sends the player's name for now.
 *
 * Return: none
 */
void NET_send_playerdata(int pl,
                         int datapl)
{
    s_playerdata pack;

    /* every send function must fill in the type */
    pack.type = S_PLAYERDATA;
    pack.num = datapl;
    strcpy(pack.name, GLO_players[datapl].p_name);
    sendPacket((char *) &pack, pl);
}


/* NET_update_one_player(pl, updpl)
 *
 * send player status FOR player #udpl TO player #pl
 * Status meaning, alive, dead, empty slot, or at the motd
 *
 * Return: none
 */
void NET_update_one_player(int pl,
                           int updpl)
{
    s_playerinfo pack;

    pack.type = S_PLAYERINFO;
    pack.num = updpl;
    pack.status = GLO_players[updpl].p_status;
    pack.team = GLO_players[updpl].p_team;
    sendPacket((char *) &pack, pl);
}

/* NET_update_all_players(pl)
 *
 * Send the status of every other player in the game to player #pl
 * Presumably this person has just now logged on
 *
 * Return: none
 */
void NET_update_all_players(int pl)
{
    int i;
    s_playerinfo pack;

    for (i = 0; i < MAXPLAYERS; i++) {
        pack.type = S_PLAYERINFO;
        pack.num = i;
        pack.status = GLO_players[i].p_status;
        pack.team = GLO_players[i].p_team;
        sendPacket((char *) &pack, pl);
    }
}

/* socket_send_personal_info(pl)
 *
 * Send data to this player about his own tank and supplies.
 * Including heading, speed, coordinates, and supplies
 *
 * Return: none
 */
void socket_send_personal_info(int pl)
{
    s_personal pack;
    int speed;

    pack.type = S_PERSONAL;
    pack.status = GLO_players[pl].p_status;

    /* Actual coordinates are stored with 4 more bits than
       are needed for display, so shift everything right by
       4 in order to fit coords into short variables */
    pack.x = (USHORT) (htons(GLO_players[pl].p_x >> 4));
    pack.y = (USHORT) (htons(GLO_players[pl].p_y >> 4));

    /* Use the "direction" byte in the packet to send both direction
       and speed.  This can be done because the client only displays
       16 directions of movement, not the full 256 that are actually
       used by the server */
    pack.dir = (GLO_players[pl].p_dir + 8) & 0xf0;
    speed = ABS((GLO_players[pl].p_speed * 9) / MAXSPEED);
    if (speed < 9 && GLO_players[pl].p_speed != 0)
        speed++;
    pack.dir |= speed;

    pack.damage = GLO_players[pl].p_damage;
    pack.ammo = GLO_players[pl].p_ammo;
    pack.fuel = htons(GLO_players[pl].p_fuel);

    /* send mines and trees carried in the same byte as well.
     * Max of 32 mines(5 bits), and 8 trees(3 bits). */
    pack.mines = GLO_players[pl].p_mines & 0x1f;
    pack.mines |= (GLO_players[pl].p_trees << 5);

    sendPacket((char *) &pack, pl);
}

/* send_login(pl)
 *
 * When a new player connects, tell them what slot they're connected to
 *
 * Return: none
 */
static void send_login(int pl)
{
    s_login pack;

    /* every send function must fill in the type */
    pack.type = S_LOGIN;
    pack.num = pl;

    if (!sendPacket((char *) &pack, pl)) {      /* fatal error */
        printf("Bad write to player %d\n", pl);
        newplayer_remove_player(pl);
        close(playerSock[pl]);
        playerSock[pl] = -1;
    }
}

/* send_playerstats(pl, statpl)
 *
 * Send statistics (win/loss/kills) FOR player #statpl TO player #pl
 *
 * Return: none
 */
static void send_playerstats(int pl,
                             int statpl)
{
    s_playerstats pack;

    pack.type = S_PLAYERSTATS;
    pack.num = statpl;

    pack.wins = htonl(GLO_players[statpl].p_wins);
    pack.losses = htonl(GLO_players[statpl].p_losses);
    pack.kills = htonl(GLO_players[statpl].p_kills);
    sendPacket((char *) &pack, pl);
}

/* send_mapsquare(x, y, value, pl)
 *
 * Update player #pl to a change in the terrain at location x,y
 *
 * Return: none
 */
static void send_mapsquare(int x,
                           int y,
                           int value,
                           int pl)
{
    s_mapsquare pack;

    pack.type = S_MAPSQUARE;
    pack.x = x;
    pack.y = y;
    pack.value = value;
    sendPacket((char *) &pack, pl);
}

/* NET_update_mapsquare(x, y)
 *
 * Tell every player about a change in the terrain at (x,y)
 *
 * Return: none
 */
void NET_update_mapsquare(int x,
                          int y)
{
    int i;

    for (i = 0; i < MAXPLAYERS; i++)
        send_mapsquare(x, y, GLO_map[x][y], i);
}

/* NET_update_mapsquare_value
 *
 * Same as update_mapsquare, but send value instead of whatever is stored in
 * the map array
 *
 * Return: none
 */
void NET_update_mapsquare_value(int x,
                                int y,
                                int value)
{
    int i;

    for (i = 0; i < MAXPLAYERS; i++)
        send_mapsquare(x, y, value, i);
}

/* send_flags (pl, cx, cy)
 *
 * Send positions for any visible flags(the wavy fluttery kind) to player #pl
 *
 * Return: none
 */
static void send_flags(int pl,
                       int cx,
                       int cy)
{
    int i, dx, dy;              /* dx, dy are display coords */
    s_flag pack;

    /* for every flag on the map */
    for (i = 0; i < GLO_maxflag; i++) {
        /* Determine what the display coordinates for this player would be */
        dx = (GLO_flags[i].f_x >> 4) - cx;
        dy = (GLO_flags[i].f_y >> 4) - cy;

        /* if the coords are onscreen for this player, send it */
        if ((dx >= 0 && dx <= WINWIDTH &&
             dy >= 0 && dy <= WINHEIGHT && GLO_flags[i].f_moved) || !((GLO_counter - pl) % UPDATEFREQ)) {
            /* every send function must fill in a type */
            pack.type = S_FLAG;

            /* fill in the rest of the data */
            pack.num = i;
            pack.team = GLO_flags[i].f_team;
            pack.x = htons(GLO_flags[i].f_x >> 4);
            pack.y = htons(GLO_flags[i].f_y >> 4);

            sendPacket((char *) &pack, pl);
        }
    }
}

/* send_updates()
 *
 * The Head honcho of send functions.  Called once per tick,
 * calls all the subsidiary send functions for every player, and
 * checks for dead slots.
 *
 * Return: none
 */
static void send_updates()
{
    int i, j, cx, cy;           /* cx, cy are upper left corner for the display of the player
                         being updated */
    player *me;                 /* me is a shortcut to the current player */

    /* loop through every player */
    for (i = 0; i < MAXPLAYERS; i++) {
        /* if this guy is active */
        if (GLO_players[i].p_status != PEMPTY) {
            me = &GLO_players[i];
            /* If he's the MOTD (outfitting) check to see how long he's been there */
            if (me->p_status == POUTFIT) {
                deadcounter[i]++;

                /* If he's been at the MOTD more than 3 minutes, kill him and free
                   up the slot. */
                if (deadcounter[i] > 1800) {    /* 3 minutes */
                    printf("Killing slot %c\n", me->p_char);

                    newplayer_remove_player(i);

                    /* close both of his sockets */
                    close(playerSock[i]);
                    playerSock[i] = -1;
                    if (playerUdpSock[i] >= 0) {
                        close(playerUdpSock[i]);
                        playerUdpSock[i] = -1;
                    }
                    continue;
                }
            } else {
                /* he's not at the motd, keep his counter at 0 */
                deadcounter[i] = 0;
            }
            /* calculate the upper left corner of this guy's display.
               Several other functions need this, do it here for convenince
               and speed */

            cx = (me->p_x >> 4) - WINWIDTH / 2;
            cy = (me->p_y >> 4) - WINHEIGHT / 2;
            if (cx < 0) {
                cx = 0;
            } else if (cx > GRIDWIDTH * GLO_map_info.m_width - WINWIDTH) {
                cx = GRIDWIDTH * GLO_map_info.m_width - WINWIDTH;
            }
            if (cy < 0) {
                cy = 0;
            } else if (cy > GRIDHEIGHT * GLO_map_info.m_height - WINHEIGHT) {
                cy = GRIDHEIGHT * GLO_map_info.m_height - WINHEIGHT;
            }
            /* call all of the send functions that need to be run every tick */
            socket_send_personal_info(i);
            send_positions(i, cx, cy);
            send_shells(i, cx, cy);
            send_flags(i, cx, cy);
            send_mines(i);

            /* statistics (wins/losses/kills) do not need to be updated every
               tick.  If it's time, check to see if each player's stats have
               changed since the last time the player being sent to was updated */
            if (!((GLO_counter + i) % UPDATEFREQ) && (GLO_players[i].p_updateplayers)) {
                for (j = 0; j < MAXPLAYERS; j++) {
                    if (GLO_players[i].p_updateplayers & (1 << j)) {
                        send_playerstats(i, j);
                    }
                }
                GLO_players[i].p_updateplayers = 0;
            }
            /* send all these glorious packets out to the player, finally
               The send_ functions normally just add the data on to a buffer,
               this makes sure it gets sent _right_now_ */
            if (playerUdpSock[i] >= 0)
                flush_buffer(i);
        }
    }

    /* f_moved keeps track of whether this flag has moved each update, and
       so whether the players need to be told of it.  Since we've now finished a tick,
       set all the moved flags to 0. */
    for (i = 0; i < GLO_maxflag; i++) {
        GLO_flags[i].f_moved = 0;
    }
}

/* send_motd(pl)
 *
 * send the motd(Message Of The Day - the title screen in effect) to player #pl
 *
 * Return: none
 */
static void send_motd(int pl)
{
    int i;
    s_motdline pack;            /* this particular packet buffer will be used
                                  once for every line in the motd */

    /* every send function must fill in a type */
    pack.type = S_MOTDLINE;

    for (i = 0; i < GLO_motdLines; i++) {
        pack.line = htons(i);
        strncpy(pack.text, GLO_motd[i], 79);
        pack.text[79] = 0;
        sendPacket((char *) &pack, pl);
    }
}

/* handle_login(buf, pl)
 *
 * Handle a login packet from player #pl
 *
 * Return: none
 */
static void handle_login(char *buf,
                         int pl)
{
    c_login *pack = (c_login *) buf;    /* assign the buffer passed
                                                     to a c_login */
    char msg[80];               /* buffer for messages to be sent */
    int vers = ntohs(pack->version), i; /* the client version this player
                                           is using */

    /* check client version against server version.  If the client is older
       than the server, inform the player they may have problems, but let them
       stay */
    if (vers < SERVERVERSION) {

        sprintf(msg, "Your client is obsolete.  You need version %d.%02d (you have %d.%02d)",
         SERVERVERSION / 100, SERVERVERSION % 100, vers / 10, vers % 10);
        NET_send_message_to_indiv(msg, M_SRV, pl);
        sprintf(msg, "I will let you stay for now, but you may have problems.");
        NET_send_message_to_indiv(msg, M_SRV, pl);
    }
    clientvers[pl] = vers;

    /* grab the player's name and login from the packet */
    strncpy(GLO_players[pl].p_name, pack->name, 16);
    strncpy(GLO_players[pl].p_login, pack->login, 16);

    /* Tell everyone else about this guy joining */
    sprintf(msg, "%s (%s@%s) joining as %d", pack->name, GLO_players[pl].p_login, GLO_players[pl].p_host, pl);
    NET_send_message_to_all(msg, M_SRV);

    /* Send everyone else the relevant data for this guy (name, login) */
    for (i = 0; i < MAXPLAYERS; i++) {
        if (playerSock[i] >= 0) {
            NET_send_playerdata(i, pl);
        }
    }
}

/* handle_steering (buf, pl)
 *
 * This guy is trying to turn.
 * Obsolete, clients should be using c_course packets now instead.
 *
 */
static void handle_steering(char *buf,
                            int pl)
{
    c_steering *pack = (c_steering *) buf;

    GLO_players[pl].p_keys = pack->keys;
}

/* handle_course(buf, pl)
 *
 * This guy sent us a new heading, start him turning towards it.
 *
 */
static void handle_course(char *buf,
                          int pl)
{
    c_course *pack = (c_course *) buf;

    /* desdir is desired direction, tanks don't turn on a dime.
       The movement code will gradually turn this guy to the new course */
    GLO_players[pl].p_desdir = pack->dir;
}

/* handle_speed(buf, pl)
 *
 * Change this guy's speed
 *
 */
static void handle_speed(char *buf,
                         int pl)
{
    c_speed *pack = (c_speed *) buf;

    if (GLO_debug > 2) {
        printf("Handling speed %d for player %d\n", pack->speed, pl);
    }
    /* Set his desired speed to what he asked for.  Requests are from 0-9,
       the actual speeds are from 0 to MAXSPEED, convert it here */
    GLO_players[pl].p_desspeed = pack->speed * MAXSPEED / 9;

    /* check to make sure his speed request is legal! */
    if (GLO_players[pl].p_desspeed > MAXSPEED) {
        GLO_players[pl].p_desspeed = MAXSPEED;
    } else if (GLO_players[pl].p_desspeed < MINSPEED) {
        GLO_players[pl].p_desspeed = MINSPEED;
    }
}

/* handle_shell(buf, pl)
 *
 * This player wants to fire a shell.
 *
 * There is no important data in the packet itself - the fact that we got this packet
 * is all we care about.
 *
 */
static void handle_shell(char *buf,
                         int pl)
{
    shell *s;                   /* the shell we're going to make active */
    player *o;                  /* the player firing it */
    /* don't even need anything from the packet, just getting it means fire a shell */

    if (GLO_debug > 2) {
        printf("Handling shell from pl %d\n", pl);
    }
    s = &GLO_shells[pl * SHELLSPERPLAYER];
    o = &GLO_players[pl];

    /* if this guy has ammo, doesn't already have a shell active, and is alive */
    if (o->p_ammo > 0 && s->s_fuse <= 0 && s->s_status == SDEAD && GLO_players[pl].p_status == PALIVE) {
        /* make it active */
        s->s_status = SALIVE;

        /* put it right on top of the firing player */
        s->s_x = o->p_x;
        s->s_y = o->p_y;

        /* make it travel the direction the guy is facing */
        s->s_dir = o->p_dir;

        /* set it's owner */
        s->s_owner = pl;

        /* fuse == number of updates a shell has left before fizzling out.
           IE, its range. */
        s->s_fuse = SHELLFUSE;

        /* take one shot away from the player */
        o->p_ammo--;

        /* tell everyone in range about this new shell */
        NET_update_shell(s);
    }
}

/* handle_message(buf, pl)
 *
 * Received a message packet from this player, process it
 *
 */
static void handle_message(char *buf,
                           int pl)
{
    c_message *pack = (c_message *) buf;
    char m[100];

    /* make a copy of the text of the message, make sure it's null terminated */
    strncpy(m, pack->message, 80);
    m[79] = 0;

    /* check the destination of the message, call the appropriate send_message
       function */
    if (pack->to == M_ALL)
        NET_send_message_to_all(m, pl);
    else if (pack->to == M_RED || pack->to == M_BLUE)
        NET_send_message_to_team(m, pack->to - M_BLUE, pl);
    else if (pack->to < MAXPLAYERS && GLO_players[pack->to].p_status != PEMPTY) {
        NET_send_message_to_indiv(m, pl, pack->to);

        /* if this player is sending a message to himself, send it to the
           command processor to see if it's a special command */
        if (pack->to == pl) {
            commands_do_command(m, pl);
        }
    }
}

/* send_teamok(pl, ok)
 *
 * Tell this player if his choice of team is legal.  Upon receipt of a confirmed
 * Team request, the client should assume the player is now in the game and
 * active, no longer at the motd
 *
 * Return: none
 */
static void send_teamok(int pl,
                        int ok)
{
    s_teamok pack;

    /* every send function must fill in a type */
    pack.type = S_TEAMOK;
    pack.teamok = ok;
    sendPacket((char *) &pack, pl);
}

/* handle_team(buf, pl)
 *
 * This player has asked to enter the game (by clicking on a team window)
 * Tell him this team is ok, and put him in the game.
 * (eventually this should do checking to make sure the two teams are
 *  balanced numerically - right now it always lets you pick either team)
 *
 * Return: none
 */
static void handle_team(char *buf,
                        int pl)
{
    c_team *pack = (c_team *) buf;
    int i, active_mines;

    if (GLO_players[pl].p_status != POUTFIT) {
        /* err, something's wrong, why is this guy trying to pick a team
           when he's already active? */
        printf("Got team request from player not outfitting\n");
        return;
    }
    /* for now, lets anyone choose either team */
    GLO_players[pl].p_team = pack->team;

    /* set him alive, give him full fuel and ammo, no damage, 0 kills, and 0 trees */
    GLO_players[pl].p_status = PALIVE;
    GLO_players[pl].p_damage = 0;
    GLO_players[pl].p_fuel = MAXFUEL;
    GLO_players[pl].p_ammo = MAXAMMO;
    GLO_players[pl].p_kills = 0;
    GLO_players[pl].p_trees = 0;

    /* Mines belonging to this guy that are still active should not be
       reloaded. */
    for (i = pl * MINESPERPLAYER, active_mines = 0; i < (pl + 1) * MINESPERPLAYER; i++) {
        if (GLO_mines[i].mi_status == MILIVE) {
            active_mines++;
        }
    }

    /* However many weren't active are loaded into this guys tank. */
    GLO_players[pl].p_mines = MINESPERPLAYER - active_mines;

    /* put the player in the game */
    newplayer_enter_player_xy(pl);

    /* send this guy info on himself */
    NET_update_one_player(pl, pl);

    /* send his info to every other active player as well */
    for (i = 0; i < MAXPLAYERS; i++) {
        if (GLO_players[i].p_status != PEMPTY) {
            NET_update_one_player(i, pl);
            NET_send_playerdata(i, pl);
        }
    }

    /* finally tell him he's in! */
    send_teamok(pl, 1);
}

/* handle_mine(buf, pl)
 *
 * This player wants to lay a mine, sneaky isn't he?
 *
 */
static void handle_mine(char *buf,
                        int pl)
{
    int i;

    /* if he has mines left */
    if (GLO_players[pl].p_mines) {
        /* find the first one that's not active */
        for (i = pl * MINESPERPLAYER; i < (pl + 1) * MINESPERPLAYER; i++) {
            if (GLO_mines[i].mi_status == MIEMPTY) {

                /* Make it active, place it under the player, and take one
                   mine out of his inventory */
                GLO_mines[i].mi_status = MILIVE;
                GLO_mines[i].mi_x = GLO_players[pl].p_x;
                GLO_mines[i].mi_y = GLO_players[pl].p_y;
                GLO_mines[i].mi_fuse = MINEARMFUSE;
                GLO_mines[i].mi_updated = 0;
                GLO_players[pl].p_mines--;
                break;
            }
        }
    }
}

/* handle_builde(buf, pl)
 *
 * This player wants to build a wall or road, or harvest a tree.
 *
 */
static void handle_build(char *buf,
                         int pl)
{
    c_build *pack = (c_build *) buf;
    int mapx, mapy, trees_needed;

    /* Can't build anything if moving. */
    if (GLO_players[pl].p_speed != 0 && pack->terrain != T_GRASS) {
        NET_warning("You must be stationary to build!", pl);
        return;
    }
    /* figure out where on the map this guy is */
    mapx = (GLO_players[pl].p_x >> 4) / GRIDWIDTH;
    mapy = (GLO_players[pl].p_y >> 4) / GRIDHEIGHT;

    /* determine how many trees are needed to build what he asked for */
    switch (pack->terrain) {
    case T_GWALL:
        trees_needed = 2;
        break;
    case T_ROAD:
        trees_needed = 1;
        break;
    case T_GRASS:               /* special, means harvest a tree */
        /* if the square he's in has a tree, and he's carrying less than the
             maximum, give him the tree and change this square to grass */
        if (GLO_map[mapx][mapy] == T_TREE && GLO_players[pl].p_trees < MAXTREES) {
            GLO_map[mapx][mapy] = T_GRASS;
            NET_update_mapsquare(mapx, mapy);
            GLO_players[pl].p_trees++;
            /* if he was hidden, he's not any more! */
            GLO_players[pl].p_flags &= ~(PFHIDDEN);
        }
        return;
    default:
        trees_needed = MAXTREES + 1;    /* make sure they can't build anything else */
        break;
    }
    if (trees_needed <= GLO_players[pl].p_trees) {      /* ok he has enough trees */

        /* always builds on the square IN FRONT OF his tank, figure out which
           one that is.  Directions range from 0-255 with 0 being straight north,
           64 due east, 128 due south, and 192 due west.

           Here, we only care about those 4 directions, pick the one
           closest to his current heading. */
        if (GLO_players[pl].p_dir < 32 || GLO_players[pl].p_dir > 224)
            mapy--;
        else if (GLO_players[pl].p_dir < 96)
            mapx++;
        else if (GLO_players[pl].p_dir < 160)
            mapy++;
        else
            mapx--;

        /* if he's pointing off the edge of the map, don't build */
        if (mapx < 0 || mapy < 0 || mapx >= GLO_map_info.m_width || mapy >= GLO_map_info.m_height)
            return;

        /* cost more to build a road(bridge) over water, and can't
           build walls on water. */
        if (GLO_map[mapx][mapy] == T_WATER || GLO_map[mapx][mapy] == T_BRIDGE) {
            trees_needed *= 2;
            if (pack->terrain == T_GWALL) {
                NET_warning("You can not build a wall on water.", pl);
                return;
            }
            if (GLO_players[pl].p_trees < trees_needed) {
                NET_warning("You do not have enough trees to build a bridge.", pl);
                return;
            }
            /* Request from the client is either road or wall,
               we change it to bridge here if appropriate */
            if (pack->terrain == T_ROAD)
                pack->terrain = T_BRIDGE;
        }
        /* can't build over a wall, or either team's base */
        if (!map_iswall(GLO_map[mapx][mapy]) && GLO_map[mapx][mapy] != T_BASE1 && GLO_map[mapx][mapy] != T_BASE2) {
            /* otherwise, this is a legal request, do it! */
            if (pack->terrain == T_GWALL) {
                map_makenewwall(mapx, mapy);
            } else {
                GLO_map[mapx][mapy] = pack->terrain;
            }
            /* If this is a wall, it starts at full strength.
               This is ignored for other terrain types. */
            GLO_mapdamage[mapx][mapy] = 0;

            /* tell everyone about the new construction */
            NET_update_mapsquare(mapx, mapy);

            /* take the trees out of this guy's inventory */
            GLO_players[pl].p_trees -= trees_needed;
        }
    } else {
        /* hey!  You gotta have trees to build stuff! */
        char wbuf[80];
        sprintf(wbuf, "You do not have enough trees to build a %s.",
                (pack->terrain == T_ROAD) ? "road" : "wall");
        NET_warning(wbuf, pl);
        return;
    }
}

/* handle_ping(buf, pl)
 *
 * This player wants to know how bad his lag is.  Client sends a ping packet,
 * and sees how long it takes for the server to send a response.
 */
static void handle_ping(char *buf,
                        int pl)
{
    c_ping *cpack = (c_ping *) buf;
    s_ping pack;

    /* every function that sends a packet must fill in a type */
    pack.type = S_PING;
    pack.seq = cpack->seq;
    sendPacket((char *) &pack, pl);
}

/* c_packetLength(buf)
 *
 * Given a pointer to the start of an unknown packet, figure out what kind
 * it is and return the length
 *
 * Return: length of the packet pointed to, in bytes
 */
static int c_packetLength(char *buf)
{
    /* the first byte of every packet indicates what type it is. */
    switch (buf[0]) {
    case C_LOGIN:
        return sizeof(c_login);
    case C_MESSAGE:
        return sizeof(c_message);
    case C_UDPREQUEST:
        return sizeof(c_udprequest);
    case C_MINE:
        return sizeof(c_mine);
    case C_BUILD:
        return sizeof(c_build);
    case C_PING:
        return sizeof(c_ping);
    case C_STEERING:
    case C_SHELL:
    default:
        return 4;
    }
}


/* do_read(sock, pl)
 *
 * The main function for handling input from the player.  Called
 * once for each socket (TCP or UDP) that has data pending (as determined by
 * a select elsewhere)
 *
 * Return: none
 */
static void do_read(int sock,
                    int pl)
{
    char buf[1024];
    int in, pos, r;             /* in: how many bytes have yet to be processed
                       pos: position currently being processed in the buffer
                       r: how much is read by subsequent reads (to finish
                          reading packets the initial read only got part of)
                     */

    /* read everything pending, up to 1k */
    in = read(sock, buf, 1024);

    if (in <= 0) {
        if (errno == EAGAIN) {
            /* the kernel's telling us it couldn't read for some reason, but that
           we shouldn't give up */
            return;
        }
        /* other errors mean we should bail on this socket. */
        if (sock == playerSock[pl]) {
            /* It was the TCP socket, this is a fatal error for this connection */
            if (GLO_debug) {
                perror("do_read");
                printf("Lost connection to player %d\n", pl);
            }
            /* close both sockets */
            close(sock);
            playerSock[pl] = -1;
            if (playerUdpSock[pl] >= 0) {
                close(playerUdpSock[pl]);
                playerUdpSock[pl] = -1;
            }
            /* take the player out of the game */
            newplayer_remove_player(pl);
            return;
        } else {
            /* It was only the UDP socket we lost, try to continue with just TCP */
            if (GLO_debug)
                printf("Lost UDP connection to player %d\n", pl);
            close(sock);
            playerUdpSock[pl] = -1;
            return;
        }
    }
    pos = 0;                    /* haven't processed anything yet */
    while (pos < in) {
        while (in - pos < c_packetLength(buf + pos)) {
            /* we don't have a complete packet in the buffer yet, read more until
               we do. */
            r = read(sock, buf + in, c_packetLength(buf + pos) - (in - pos));
            if (r <= 0) {
                /* do the same error checking as above, with the exception
                   that EAGAIN is probably fatal here too, so kill the
                   connection no matter what.*/
                if (sock == playerSock[pl]) {
                    if (GLO_debug)
                        printf("Lost connection to player %d\n", pl);
                    close(sock);
                    playerSock[pl] = -1;
                    playerUdpSock[pl] = -1;
                    newplayer_remove_player(pl);
                    return;
                } else {        /* Lost UDP socket only */
                    if (GLO_debug)
                        printf("Lost UDP connection to player %d\n", pl);
                    close(sock);
                    playerUdpSock[pl] = -1;
                    return;
                }
            }
            /* we just added r bytes to the buffer, update our count of
               unprocessed data */
            in += r;
        }

        /* pos is pointing to the first byte of the next packet to be
           processed, figure out what type it is and call an appropriate
           handler */
        switch (buf[pos]) {
        case C_LOGIN:
            handle_login(&buf[pos], pl);
            break;
        case C_STEERING:
            handle_steering(&buf[pos], pl);
            break;
        case C_SHELL:
            handle_shell(&buf[pos], pl);
            break;
        case C_MESSAGE:
            handle_message(&buf[pos], pl);
            break;
        case C_COURSE:
            handle_course(&buf[pos], pl);
            break;
        case C_SPEED:
            handle_speed(&buf[pos], pl);
            break;
        case C_UDPREQUEST:
            handle_UdpReq(&buf[pos], pl);
            break;
        case C_TEAM:
            handle_team(&buf[pos], pl);
            break;
        case C_MINE:
            handle_mine(&buf[pos], pl);
            break;
        case C_BUILD:
            handle_build(&buf[pos], pl);
            break;
        case C_PING:
            handle_ping(&buf[pos], pl);
            break;
        default:
            /* oops!  Got a bad packet! */
            printf("Bad packet type %d from player %d\n", buf[pos], pl);
            break;
        }
        /* we've now processed a packet, point to the byte immediately following
           it and continue processing */
        pos += c_packetLength(buf + pos);
    }
}

/* send_playerlist(sock)
 *
 * This function sends a plain text player list to people telnetting in on
 * a special port set aside for this purpose.
 *
 */
static void send_playerlist(int sock)
{
    char buf[80];
    int i;

    /* don't want people getting the playerlist to make the server hang.
     * In practice, most OS's won't hang here anyway, but for those that do
     * the playerlist will just be missing some lines.  Not desirable,
     * but this is not essential to the game at all.
     */
    fcntl(sock, F_SETFL, O_NONBLOCK);

    sprintf(buf, "\nXFirepower player list\n");
    strcat(buf, "----------------------\n");
    write(sock, buf, strlen(buf));

    /* For every active player, send a line of the form
       No: Name (login@host)
       */
    for (i = 0; i < MAXPLAYERS; i++) {
        if (GLO_players[i].p_status != PEMPTY) {
            sprintf(buf, "%2d: %s (%s@%s)\n", i, GLO_players[i].p_name, GLO_players[i].p_login, GLO_players[i].p_host);
            write(sock, buf, strlen(buf));
        }
    }
    sprintf(buf, "----------------------\nThat's it!\n");
    write(sock, buf, strlen(buf));
}


/* NET_checkSockets()
 *
 * This is the big one.  Checks for pending connections on all sockets
 * (listen socket, player list socket, and TCP/UDP sockets for every player
 * calls the tick code (tick.c - updates positions, fuel, etc.) and keeps
 * track of the timer
 *
 * return: None
 */
void NET_checkSockets()
{
    struct sockaddr_in naddr;
    fd_set read_fds;
    int st, newsock, len, i;
    static struct timeval tv, otv;
    static int init = 0;
    int ut;

    /* get the current time if this is the first time we've been called */
    if (!init) {
        gettimeofday(&otv, 0);
        init = 1;
    }
    /* for every socket we have open, get ready to check for pending
       data or connections */
    FD_ZERO(&read_fds);
    FD_SET(listenSock, &read_fds);
    FD_SET(playerListSock, &read_fds);
    for (i = 0; i < MAXPLAYERS; i++) {
        /* for every player, check both the TCP and UDP sockets */
        if (playerSock[i] >= 0) {
            FD_SET(playerSock[i], &read_fds);
        }
        if (playerUdpSock[i] >= 0) {
            FD_SET(playerUdpSock[i], &read_fds);
        }
    }

    /* stay in this loop until there's data pending  from some socket */
    while (1) {
        /* determine number of microseconds that have passed since our last
           update */
        gettimeofday(&tv, 0);
        if (tv.tv_sec > otv.tv_sec)
            tv.tv_usec += 1000000;
        ut = tv.tv_usec - otv.tv_usec;
        if (GLO_debug > 2)
            printf("ut = %d\n", ut);

        /* if less than UTIMER, wait until it's been long enough,
           or we have data pending on some sockets */
        if (UTIMER - ut > 0) {
            struct timeval tv;
            unsigned long usecs = UTIMER - ut;
            tv.tv_sec = usecs / 1000000L;
            tv.tv_usec = usecs % 1000000L;
            st = select(70, &read_fds, 0, 0, &tv);
            break;
        } else {
            if (GLO_debug > 2)
                printf("Timer! \n");

            /* do movement, shells, etc. */
            tick_do_tick();

            /* tell everyone what happened this tick */
            send_updates();

            /* count the number of ticks we've done */
            GLO_counter++;
#ifdef LIFE
            /* If it's time to make the trees grow, do it */
            if (!(GLO_counter % LIFEFREQ)) {
                life_do_life();
            }
#endif
            /* mark the time at which we finished this tick */
            gettimeofday(&otv, 0);
        }
    }

    /* Once we get here, some socket probably needs reading,
       see which one it is and do it. */
    if (FD_ISSET(listenSock, &read_fds)) {
        /* A new player connection! Let them in.*/

        len = sizeof(naddr);
        printf("Accepting new connection on main port\n");
        newsock = accept(listenSock, (struct sockaddr *) &naddr, &len);

        if (newsock < 0) {
            /* Err? */
            perror("accept");
            /* shouldn't have to do anything to recover from this, no
               player struct was filled in at all. */
        } else {
            printf("Got new connection!\n");

            /* find the first empty player slot */
            for (i = 0; i < MAXPLAYERS; i++) {

                /* If the socket isn't marked as empty (< 0) something's
                   wrong here anyway, better not let them in at all. */
                if (playerSock[i] < 0) {

                    /* set the new socket to nonblocking, set this player's
                       socket to the new socket. */
                    fcntl(newsock, F_SETFL, O_NONBLOCK);
                    playerSock[i] = newsock;

                    /* find out this guy's host name for logging and telling
                       everyone else. */
                    newplayer_find_host(newsock, i);

                    /* send this guy the stuff he needs to know */
                    send_login(i);
                    NET_send_map(i);

                    /* fill in his player struct */
                    newplayer_new_player(i);

                    /* get this guy's IP address */
                    checkSocket(i);

                    /* send him the title screen (MOTD) */
                    send_motd(i);

                    /* clear his UDP buffer (note that he does not
                       actually have a UDP socket at all at this point */
                    udpBufPos[i] = 0;

                    /* give him the full amount of time to pick a team
                       before kicking him off */
                    deadcounter[i] = 0;
                    break;
                }
            }
        }
    }
    if (FD_ISSET(playerListSock, &read_fds)) {
        /* someone is telnetting in requesting the current player list */
        len = sizeof(naddr);
        printf("Accepting connection on player list port\n");
        newsock = accept(playerListSock, (struct sockaddr *) &naddr, &len);

        if (newsock < 0) {
            /* Hmm. */
            perror("accept");
        } else {
            /* find, send it and close the socket */
            printf("Sending playerlist\n");
            send_playerlist(newsock);
            close(newsock);
        }
    }
    /* Check and see which players have sent us packets this update */
    for (i = 0; i < MAXPLAYERS; i++) {

        /* First the TCP socket */
        if (playerSock[i] >= 0 && FD_ISSET(playerSock[i], &read_fds)) {
            if (GLO_debug > 2)
                printf("data from player %d\n", i);
            do_read(playerSock[i], i);
        }
        /* Then the UDP */
        if (playerUdpSock[i] >= 0 && FD_ISSET(playerUdpSock[i], &read_fds)) {
            if (GLO_debug > 2)
                printf("UDP data from player %d\n", i);
            do_read(playerUdpSock[i], i);
        }
    }
}

/* handle_UdpReq(buf, pl)
 *
 * This player has requested we open a UDP connection for him
 *
 * Return: none
 */
static void handle_UdpReq(char *buf,
                          int pl)
{
    c_udprequest *pack = (c_udprequest *) buf;
    s_udpreply response;
    int mode, port, localPort;

    /* every function that sends a packet must fill in the type */
    response.type = S_UDPREPLY;

    /* Is he requesting we open or close the UDP connection? */
    mode = pack->mode;

    printf("Handling UDP request\n");
    if (mode != MODE_TCP && mode != MODE_UDP) {
        printf("Bad UDP request from player %d\n", pl);
        return;
    }
    /* okay, we have a request to change modes */
    if (mode == MODE_UDP) {
        port = ntohs(pack->port);       /* where to connect to */
        printf("Client UDP port is %d\n", port);
        if (playerUdpSock[pl] >= 0) {
            /* we have a socket open, but the client doesn't seem aware */
            /* (probably because our UDP verify got lost down the line) */
            closeUdpConn(pl);
        }
        /* (note no openUdpConn(); we go straight to connect) */
        if ((localPort = connUdpConn(pl, port)) == -1) {
            response.reply = SWITCH_DENIED;
            response.port = 0;
            sendPacket((char *) &response, pl);
        }
        printf("Local UDP port is %d\n", localPort);
        /* we are now connected to the client, but he's merely bound */
        /* don't switch to UDP mode yet; wait until client connects */
        response.reply = SWITCH_UDP_OK;
        response.port = htons(localPort);
        sendUdpPacket((char *) &response, pl);
    }
}


/* connUdpConn(pl, udpclientport)
 *
 * Create a new udp socket for player #pl, and connect it to the client
 * on the port specified
 *
 * return: local udp port used, -1 on error
 */
static int connUdpConn(int pl,
                       int udpClientPort)
{
    struct sockaddr_in addr, caddr;
    int len;
    int udpSock;
    int udpLocalPort;

    /* Create a DGRAM (aka UDP) socket */
    if ((udpSock = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
        perror("xfireserv: unable to create DGRAM socket");
        return (-1);
    }
    /* accept connections from any address, let the system pick a port for us */
    addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = INADDR_ANY;
    addr.sin_port = 0;

    if (bind(udpSock, (struct sockaddr *) &addr, sizeof(addr)) < 0) {
        perror("xfireserv: cannot bind to local port");
        close(udpSock);
        udpSock = -1;
        return (-1);
    }
    /* determine what our port is */
    len = sizeof(addr);
    if (getsockname(udpSock, (struct sockaddr *) &addr, &len) < 0) {
        perror("xfireserv: unable to getsockname(UDP)");
        close(udpSock);
        udpSock = -1;
        return (-1);
    }
    udpLocalPort = (int) ntohs(addr.sin_port);

    /* Connect our new socket to the client's (hopefully) waiting
     * port.
     */
    caddr.sin_family = AF_INET;
    caddr.sin_addr.s_addr = remoteaddr[pl];     /* addr of our client */
    caddr.sin_port = htons(udpClientPort);      /* client's port */

    if (connect(udpSock, (struct sockaddr *) &caddr, sizeof(caddr)) < 0) {
        /* Apparently it wasn't waiting for a connect, give up */
        perror("xfireserv: connect to client UDP port");
        close(udpSock);
        udpSock = -1;
        return (-1);
    }
    /* determine what our port is */
    len = sizeof(addr);
    if (getsockname(udpSock, (struct sockaddr *) &addr, &len) < 0) {
        perror("xfireserv: unable to getsockname(UDP)");
        close(udpSock);
        udpSock = -1;
        return (-1);
    }
    udpLocalPort = (int) ntohs(addr.sin_port);

    /* everything worked, fill in this guy's UDP socket with our new one */
    playerUdpSock[pl] = udpSock;

    /* And set it nonblocking */
    fcntl(playerUdpSock[pl], F_SETFL, O_NONBLOCK);
    return (udpLocalPort);
}

/* closeUdpConn(pl)
 *
 * Close a player's UDP connection.
 *
 * Return: -1 if he didn't have a connection open, 0 otherwise
 */
static int closeUdpConn(int pl)
{
    if (playerUdpSock[pl] < 0) {
        return (-1);
    }
    shutdown(playerUdpSock[pl], 2);     /* wham */
    close(playerUdpSock[pl]);   /* bam */
    playerUdpSock[pl] = -1;     /* (nah) */

    return (0);
}
