/*
 *   cddbd - CD Database Protocol Server
 *
 *   Copyright (C) 1996  Steve Scherf
 *   Email: steve@moonsoft.com
 *   Moondog Software Productions - makers of fine public domain software.
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifndef LINT
static char *_cddbd_c_ident_ = "@(#)$Id: cddbd.c,v 1.7 1996/12/22 01:48:54 steve Exp $";
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>
#include <ctype.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <dirent.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <time.h>
#include <sys/time.h>
#include <stdio.h>
#include "patchlevel.h"
#include "access.h"
#include "list.h"
#include "cddbd.h"


/* Prototypes. */

void cddbd_access_err(int);
void cddbd_access_to(void);
void cddbd_check_access(void);
void cddbd_check_nus(void);
void cddbd_connect_to(void);
void cddbd_db_init(void);
void cddbd_init(void);
void cddbd_input_to(void);
void cddbd_lock_free_lk(void *);
void cddbd_timer_init(ctimer_t *);
void do_cddb(arg_t *);
void do_cddb_hello(arg_t *);
void do_cddb_query(arg_t *);
void do_cddb_read(arg_t *);
void do_cddb_srch(arg_t *);
void do_cddb_write(arg_t *);
void do_get(arg_t *);
void do_help(arg_t *);
void do_log(arg_t *);
void do_motd(arg_t *);
void do_proto(arg_t *);
void do_quit(arg_t *);
void do_sites(arg_t *);
void do_stat(arg_t *);
void do_update(arg_t *);
void do_ver(arg_t *);
void do_whom(arg_t *);
void free_user(void *);
void sighand(int);
void trunc_log(void);
void usage(char *);

int cddbd_locked(clck_t *);
int cddbd_nus(int);
int cddbd_parse_access(arg_t *, int);
int cddbd_strcasecmp(char *, char *);
int comp_client(void *, void *);
int comp_user(void *, void *);
int cvt_date(char *, char *);
int get_conv(char **, char *);
int make_year(int);

clck_t *lock_log;
clck_t *lock_nus;
clck_t *lock_hash;
clck_t *lock_hist;
clck_t *lock_tfuzz;
clck_t *lock_update;


/* Global variables. */

int build_fuzzy;			/* Build the fuzzy match hash file.*/
int categcnt;				/* Category count. */
int check;				/* Check the database. */
int child;				/* True if we are a child process. */
int client_cnt[NCLIENT];		/* Client connect count. */
int curpid;				/* Our current pid. */
int db_access;				/* Database was accessed. */
int db_dir_mode;			/* DB directory creation mode. */
int db_file_mode;			/* DB file creation mode. */
int db_gid;				/* Effective DB group-id. */
int db_uid;				/* Effective DB user-id. */
int debug;				/* Debug mode if true. */
int skip_log;				/* Checked the log if true. */
int dir_mode;				/* Directory creation mode. */
int dup_ok;				/* Dup directory exists if true. */
int dup_policy;				/* How to treat duplicate entries. */
int file_mode;				/* File creation mode. */
int fuzzy_div;				/* Fuzzy matching divisor. */
int fuzzy_factor;			/* Fuzzy matching factor. */
int get_ok;				/* Maint files obtainable if true. */
int gid;				/* Effective group-id. */
int hello;				/* Handshake done if true. */
int level = DEF_PROTO_LEVEL;		/* Protocol level we're running at. */
int lock_initted;			/* Locking is initialized. */
int logging;				/* Logging in progress. */
int log_hiwat;				/* Log file high water mark. */
int log_lowat;				/* Log file low water mark. */
int mail;				/* Mail filter mode, if true. */
int max_lines;				/* Max # of lines in a cddb entry. */
int max_users;				/* Max # of simultaneous users. */
int max_xmits;				/* Max # of simultaneous transmits. */
int mode;				/* Mode of operation. */
int quiet;				/* Don't log messages, if true. */
int standalone;				/* Standalone mode, if true. */
int test_mail;				/* Mail server test mode. */
int uid;				/* Effective user-id. */
int update;				/* Update database with new entries. */
int update_ok;				/* DB updates are allowed if true. */
int verbose;				/* Verbose mode if true. */
int write_ok;				/* Posting is allowed if true. */
int xmit_time;				/* Transmit timeout in seconds. */

char *categlist[CDDBMAXDBDIR + 1];
char admin_email[CDDBBUFSIZ];
char bounce_email[CDDBBUFSIZ];
char cddbdir[CDDBBUFSIZ];
char client[NCLIENT][CDDBBUFSIZ];
char dupdir[CDDBBUFSIZ];
char hashdir[CDDBBUFSIZ];
char histfile[CDDBBUFSIZ];
char host[CDDBHOSTNAMELEN + 1];
char lockdir[CDDBBUFSIZ];
char lockfile[CDDBBUFSIZ];
char logfile[CDDBBUFSIZ];
char motdfile[CDDBBUFSIZ];
char postdir[CDDBBUFSIZ];
char rhost[CDDBHOSTNAMELEN + 1];
char sitefile[CDDBBUFSIZ];
char smtphost[CDDBBUFSIZ];
char thistfile[CDDBBUFSIZ];
char uhistfile[CDDBBUFSIZ];
char user[CDDBBUFSIZ];

char *accessfile = ACCESSFILE;
char *lockprefix = "db_lck";
char *mailprefix = "db_mail";
char *respprefix = "db_re";
char *tmpdir = TMPDIR;

char *hellostr = "Connect: %s@%s (%s) using %s %s";
char *hellosstr =
    "%*s %*s %*s Connect: %127[^@]@%*s %*[(]%127[^)]%*s using %127s";
char *verstr =
    "cddbd v%sPL%d Copyright (c) 1996 Steve Scherf (steve@moonsoft.com)";
char *verstr2 =
    "cddbd v%sPL%d Copyright (c) 1996 Steve Scherf";

short log_flags;			/* Logging flags. */

long lastsecs;

lhead_t *lock_head;			/* Lock list head. */


/* Access file fields. */
access_t acctab[] = {
	"logfile",	logfile,       AC_PATH,    0,	     (int)TMPDIR"/log",
	"motdfile",	motdfile,      AC_PATH,    0,	     (int)"",
	"sitefile",	sitefile,      AC_PATH,    0,	     (int)"",
	"histfile",	histfile,      AC_PATH,    0,	     (int)"",
	"lockdir",	lockdir,       AC_PATH,    0,	     (int)"",
	"hashdir",	hashdir,       AC_PATH,    0,	     (int)"",
	"cddbdir",	cddbdir,       AC_PATH,    0,	     (int)"",
	"dupdir",	dupdir,	       AC_PATH,    0,	     (int)"",
	"postdir",	postdir,       AC_PATH,    0,	     (int)"",
	"smtphost",	smtphost,      AC_STRING,  0,        (int)"localhost",
	"admin_email",	admin_email,   AC_STRING,  AF_NODEF, 0,
	"bounce_email",	bounce_email,  AC_STRING,  0,	     (int)"",
	"xmit_time",	&xmit_time,    AC_NUMERIC, AF_ZERO,  DEF_XMIT_TO,
	"transmits",	&max_xmits,    AC_NUMERIC, AF_ZERO,  DEF_MAX_XMITS,
	"post_lines",	&max_lines,    AC_NUMERIC, AF_ZERO,  DEF_MAX_LINES,
	"users",	&max_users,    AC_NUMERIC, AF_ZERO,  DEF_MAX_USERS,
	"fuzzy_factor",	&fuzzy_factor, AC_NUMERIC, AF_ZERO,  DEF_FUZZ_FACT,
	"fuzzy_div",	&fuzzy_div,    AC_NUMERIC, AF_NONZ,  DEF_FUZZ_DIV,
	"log_hiwat",	&log_hiwat,    AC_NUMERIC, AF_ZERO,  DEF_LOG_HIWAT,
	"log_lowat",	&log_lowat,    AC_NUMERIC, AF_ZERO,  DEF_LOG_HIWAT,
	"user",		&uid,	       AC_USER,    AF_NODEF, 0,
	"group",	&gid,	       AC_GROUP,   AF_NODEF, 0,
	"file_mode",	&file_mode,    AC_MODE,    0,	     DEF_FILE_MODE,
	"dir_mode",	&dir_mode,     AC_MODE,    0,	     DEF_DIR_MODE,
	"db_user",	&db_uid,       AC_USER,    AF_NODEF, 0,
	"db_group",	&db_gid,       AC_GROUP,   AF_NODEF, 0,
	"db_file_mode",	&db_file_mode, AC_MODE,    0,	     DEF_FILE_MODE,
	"db_dir_mode",	&db_dir_mode,  AC_MODE,    0,	     DEF_DIR_MODE,
	{ 0 }
};


/* Protocol flags. */
proto_t proto_flags[MAX_PROTO_LEVEL + 1] = {
	0,
	0,
	P_QUOTE
};


/* Log flags. */
log_t log[] = {
	{ LOG_NONE,   0, 0,        "none",   0, },
	{ LOG_INPUT,  0, 0,        "input",  0, },
	{ LOG_INFO,   0, 0,        "info",   0, },
	{ LOG_ALL,    0, 0,        "all",    0, },
	{ LOG_ACCESS, 0, 0,        "access", "Accesses", },
	{ LOG_QUERY,  0, 0,        "",       "    Successful queries", },
	{ LOG_UQUERY, 0, 0,        "",       "    Unsuccessful queries", },
	{ LOG_FUZZY,  0, 0,        "",       "    Fuzzy matches", },
	{ LOG_READ,   0, 0,        "",       "    Entries read", },
	{ LOG_WRITE,  0, L_NOSHOW, "post",   "    Entries posted", },
	{ LOG_ERR,    0, 0,        "errors", "Errors", },
	{ LOG_XMIT ,  0, L_NOSHOW, "",       "    Transmit errors", },
	{ LOG_NET ,   0, L_NOSHOW, "",       "    Network errors", },
	{ LOG_LOCK,   0, L_NOSHOW, "",       "    Locking errors", },
	{ LOG_HASH,   0, L_NOSHOW, "",       "    Hash file errors", },
	{ LOG_UPDATE, 0, L_NOSHOW, "",       "    Update errors", },
	{ LOG_MAIL,   0, L_NOSHOW, "",       "    Mail errors", },
	{ LOG_HELLO,  0, 0,        "hello",  "Connections", },
	{ 0 }
};


/* Timer array. */
ctimer_t timers[] = {
	{ "input",   cddbd_input_to,   CT_INPUT_RST,  DEF_INPUT_TO,   0 },
	{ "access",  cddbd_access_to,  CT_ACCESS_RST, DEF_ACCESS_TO,  0 },
	{ "connect", cddbd_connect_to, CT_WRITE_DIS,  DEF_CONNECT_TO, 0 },
	{ 0 }
};


/* Help strings. */
char **help[] = {
	cddb_help,
	get_help,
	help_help,
	log_help,
	motd_help,
	proto_help,
	quit_help,
	sites_help,
	stat_help,
	update_help,
	ver_help,
	whom_help,
	0
};

char **chelp[] = {
	hello_help,
	query_help,
	read_help,
	srch_help,
	write_help,
	0
};


/* Command definitions. */

cmd_t cmd[] = {
	{ "cddb",   do_cddb,  	cddb_help,   CF_SUBCMD },
	{ "get",    do_get,  	get_help,    0 },
	{ "help",   do_help,  	help_help,   0 },
	{ "log",    do_log,  	log_help,    0 },
	{ "motd",   do_motd,  	motd_help,   0 },
	{ "proto",  do_proto,  	proto_help,  0 },
	{ "quit",   do_quit,  	quit_help,   0 },
	{ "sites",  do_sites,  	sites_help,  0 },
	{ "stat",   do_stat,  	stat_help,   0 },
	{ "update", do_update, 	update_help, 0 },
	{ "ver",    do_ver,  	ver_help,    0 },
	{ "whom",   do_whom,  	whom_help,   0 },
	{ 0 }
};

cmd_t cddb_cmd[] = {
	{ "hello", do_cddb_hello, hello_help, 0 },
	{ "query", do_cddb_query, query_help, (CF_HELLO | CF_ACCESS) },
	{ "read",  do_cddb_read,  read_help,  (CF_HELLO | CF_ACCESS) },
	{ "srch",  do_cddb_srch,  srch_help,  (CF_HELLO | CF_ACCESS) },
	{ "write", do_cddb_write, write_help, (CF_HELLO | CF_ACCESS) },
	{ 0 }
};


/* Month table. */
month_t month[] = {
	{ "Jan", 31 },
	{ "Feb", 29 },
	{ "Mar", 31 },
	{ "Apr", 30 },
	{ "May", 31 },
	{ "Jun", 30 },
	{ "Jul", 31 },
	{ "Aug", 31 },
	{ "Sep", 30 },
	{ "Oct", 31 },
	{ "Nov", 30 },
	{ "Dec", 31 }
};


/* Day table. */
char *day[] = { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };


/* ARGSUSED */
void
main(int argc, char **argv)
{
	int i;
	int fix;
	int port;
	int nmode;
	int rmt_op;
	int nrmt_op;
	int check_level;
	char *p;
	char *name;
	char *remote;
	char when[CDDBBUFSIZ];

	name = argv[0];

	mode = MODE_NONE;
	nmode = MODE_NONE;
	rmt_op = RMT_OP_NONE;
	nrmt_op = RMT_OP_NONE;

	while(*++argv) {
		if(**argv != '-')
			usage(name);

		i = 0;

		while(*++(*argv)) {
			switch(**argv) {
			case 'a':
				i++;
				accessfile = *(argv + i);
				break;

			case 'C':
				fix++;
				/* FALLTHROUGH */

			case 'c':
				nmode = MODE_DB;
				check++;

				i++;
				p = *(argv + i);

				if(p == 0)
					usage(name);

				if(!strcmp(p, "default")) {
					check_level = MIN_CHECK_LEVEL;
				}
				else {
					check_level = atoi(p);
					if(check_level < MIN_CHECK_LEVEL)
						usage(name);

					if(check_level > MAX_CHECK_LEVEL)
						usage(name);
				}


				break;

			case 'd':
				debug++;
				break;

			case 'f':
				nmode = MODE_DB;
				build_fuzzy++;
				break;

			case 'l':
				nmode = MODE_DB;
				nrmt_op = RMT_OP_LOG;

				i++;
				remote = *(argv + i);
				break;

			case 'm':
				nmode = MODE_MAIL;
				mail++;
				break;

			case 'M':
				nmode = MODE_MAIL;
				mail++;
				test_mail++;
				break;

			case 'q':
				quiet++;
				break;

			case 's':
				nmode = MODE_SERV;
				standalone++;

				i++;
				p = *(argv + i);

				if(p == 0)
					usage(name);

				if(!strcmp(p, "default")) {
					port = 0;
				}
				else {
					port = atoi(p);
					if(port <= 0)
						usage(name);
				}

				break;

			case 'T':
				nmode = MODE_DB;
				rmt_op = RMT_OP_CATCHUP;

				i++;
				remote = *(argv + i);

				if(remote == 0)
					usage(name);

				i++;
				p = *(argv + i);
				if(p == 0)
					usage(name);

				if(!strcmp("now", p)) {
					strcpy(when, p);
				}
				else {
					if(cvt_date(p, when) == 0)
						usage(name);
				}

				break;

			case 't':
				nmode = MODE_DB;
				nrmt_op = RMT_OP_TRANSMIT;

				i++;
				remote = *(argv + i);

				break;

			case 'u':
				nmode = MODE_DB;
				update++;
				break;

			case 'v':
				verbose++;
				break;

			default:
				usage(name);
			}

			if(i > 0 && *(argv + i) == 0)
				usage(name);

			if(nrmt_op != RMT_OP_NONE) {
				if(rmt_op != RMT_OP_NONE)
					usage(name);

				rmt_op = nrmt_op;
				nrmt_op = RMT_OP_NONE;
			}

			if(nmode != MODE_NONE) {
				if(mode != MODE_NONE && mode != nmode)
					usage(name);

				mode = nmode;
				nmode = MODE_NONE;
			}
		}

		argv += i;
	}

	/* The default mode is server mode. */
	if(mode == MODE_NONE)
		mode = MODE_SERV;

	/* Initialize data structures. */
	cddbd_init();

	/* Wait for connections. */
	if(standalone)
		cddbd_stand(port);

	/* Find the remote hostname. */
	get_rmt_hostname(0, rhost);

	/* Set up the signal handler. */
	for(i = 0; i < NSIG; i++)
		signal(i, sighand);

	/* Ignore these signals just in case. */
	signal(SIGCHLD, SIG_IGN);
	signal(SIGPIPE, SIG_IGN);
	signal(SIGHUP, SIG_IGN);

	/* Get the configuration. */
	cddbd_check_access();

	/* Initialize the database. */
	cddbd_db_init();

	/* Be a mail filter. */
	if(mail)
		cddbd_mail();

	/* Clean up the database. */
	if(check)
		cddbd_check_db(check_level, fix);

	/* Update the database with new entries. */
	if(update)
		cddbd_update();

	/* Build the fuzzy matching hash table. */
	if(build_fuzzy)
		cddbd_build_fuzzy();

	/* Perform remote ops. */
	if(rmt_op != RMT_OP_NONE)
		cddbd_rmt_op(rmt_op, remote, when);

	if(mode == MODE_SERV) {
		/* Make sure there aren't too many users. */
		cddbd_check_nus();

		/* Start up timers. */
		cddbd_timer_init(timers);

		/* Start serving. */
		cddbd_serve();
	}

	quit(QUIT_OK);
}


void
cddbd_serve(void)
{
	int pf;
	char *p;
	cmd_t *c;
	arg_t args;
	int errcnt;

	/* Dup stdin to stdout and stderr. */
	dup2(0, 1);
	dup2(0, 2);

	/* Start up banner. */
	p = (write_ok ? "200" : "201");
	printf("%s %s CDDBP server v%sPL%d ready at %s\r\n",
		p, host, VERSION, PATCHLEVEL, get_time(1));
	fflush(stdout);

	/* Command processing loop. */
	for(errcnt = 0;;) {
		if(cddbd_gets(args.buf, sizeof(args.buf)) == NULL)
			break;

		/* Remove CR/LF so we can print this. */
		strip_crlf(args.buf);

		if(is_valid(args.buf)) {
			cddbd_log(LOG_INPUT, "Input: \"%s\"", args.buf);

			/* Parse the input. */
			if(PROTO_ENAB(P_QUOTE))
				pf = PF_HONOR_QUOTE | PF_REMAP_QSPC;
			else
				pf = 0;

			cddbd_parse_args(&args, pf);

			/* Skip empty input lines. */
			if(args.nargs == 0)
				continue;

			/* Execute the command. */
			for(c = cmd; c->cmd; c++) {
				if(!cddbd_strcasecmp(args.arg[0], c->cmd)) {
					(*(c->func))(&args);
					break;
				}
			}

			/* No command match found. */
			if(c->cmd == 0) {
				errcnt++;
				printf("500 Unrecognized command.\r\n");
			}
			else
				errcnt = 0;
		}
		else {
			errcnt++;
			printf("500 Illegal input.\r\n");
		}

		if(errcnt >= CDDBMAXERR) {
			printf("530 Too many errors, closing connection.\r\n");
			_quit(QUIT_ERR, 0);
		}

		fflush(stdout);
	}

	quit(QUIT_OK);
}


void
sighand(int sig)
{
	int i;

	/* Lock out (almost) all signals so we can exit in peace. */
	for(i = 0; i < NSIG; i++)
		if(i != SIGSEGV || i != SIGBUS)
			signal(i, SIG_IGN);

	/* Avoid recursion if we're crashing due to a bug in quit(). */
	if(sig == SIGSEGV || sig == SIGBUS)
		signal(i, SIG_DFL);

	if(sig == SIGPIPE)
		quit(QUIT_OK);

	cddbd_log(LOG_ERR, "Received signal %d.", sig);
	quit(QUIT_ERR);
}


void
quit(int status)
{
	_quit(status, 1);
}


void
_quit(int status, int msg)
{
	char file[CDDBBUFSIZ];

	/* Log connections from gawkers. */
	if(!hello && !skip_log && mode == MODE_SERV)
		cddbd_log(LOG_HELLO, "Connect: %s, no handshake.", rhost);

	/* Unlink our private files. */
	if(lockfile[0] != '\0')
		unlink(lockfile);

	if(uhistfile[0] != '\0')
		unlink(uhistfile);

	if(lock_tfuzz != 0 && lock_tfuzz->lk_refcnt > 0) {
		cddbd_snprintf(file, sizeof(file), "%s/%s", hashdir, tfuzzfile);
		unlink(file);
	}

	/* We may need to clean up the history/tmp history files. */
	if(lock_hist != 0) {
		if(lock_hist->lk_refcnt > 0 && thistfile[0] != '\0')
			unlink(thistfile);
		else if(lock_hist->lk_refcnt == 0)
			(void)cddbd_close_history();
	}

	if(status != QUIT_OK)
		cddbd_log(LOG_INFO, "Quitting, status %d.", status);

	/* Free any pending locks. */
	if(lock_head != 0)
		list_free(lock_head);

	if(mode == MODE_SERV && msg) {
		if(status == QUIT_OK) {
			printf("230 %s Closing connection.  Goodbye.\r\n",
			    host);
		}
		else {
			printf("530 %s Server error, closing connection.\r\n",
			    host);
		}
	}

	fflush(stdout);
	exit(status);
}


int
cddbd_strcasecmp(char *s1, char *s2)
{
	char c1;
	char c2;

	if(s1 == NULL || s2 == NULL)
		return 0;

	for(;;) {
		if(isupper(*s1))
			c1 = tolower(*s1);
		else
			c1 = *s1;

		if(isupper(*s2))
			c2 = tolower(*s2);
		else
			c2 = *s2;

		if(c1 != c2 || c1 == '\0' || c2 == '\0')
			return(c1 - c2);

		s1++;
		s2++;
	}
}


int
cddbd_strncasecmp(char *s1, char *s2, int n)
{
	char c1;
	char c2;

	if(s1 == NULL || s2 == NULL)
		return 0;

	while(n--) {
		if(isupper(*s1))
			c1 = tolower(*s1);
		else
			c1 = *s1;

		if(isupper(*s2))
			c2 = tolower(*s2);
		else
			c2 = *s2;

		if(c1 != c2 || c1 == '\0' || c2 == '\0')
			return(c1 - c2);

		s1++;
		s2++;
	}

	return 0;
}


char *
cddbd_strcasestr(char *hay, char *needle)
{
	int len;

	if(hay == NULL || needle == NULL)
		return 0;

	len = strlen(needle);

	for(; *hay != '\0'; hay++)
		if(!cddbd_strncasecmp(hay, needle, len))
			return hay;

	return 0;
}


void
cddbd_check_nus(void)
{
	int nus;
	FILE *fp;

	if(lockdir[0] == '\0' || max_users == 0)
		return;

	(int)cddbd_lock(lock_nus, 1);

	nus = cddbd_nus(0);

	if(nus >= max_users) {
		cddbd_log(LOG_INFO,
		    "Service rejected: nus (%d) >= max_users (%d).",
		    nus, max_users);

		printf("433 No connections allowed: ");
		printf("%d users allowed, %d currently active.\r\n",
		    max_users, nus);

		_quit(QUIT_ERR, 0);
	}

	cddbd_snprintf(lockfile, sizeof(lockfile), "%s/%s.%05d",
	    lockdir, lockprefix, curpid);

	if((fp = fopen(lockfile, "w")) == NULL) {
		cddbd_log(LOG_ERR, "Can't create lock file: %s (%d).",
		    lockfile, errno);
		quit(QUIT_ERR);
	}

	fprintf(fp, "unknown unknown - %s", rhost);
	fclose(fp);

	(void)cddbd_fix_file(lockfile, file_mode, uid, gid);

	cddbd_unlock(lock_nus);
}


int
cddbd_nus(int list)
{
	int len;
	int nus;
	int pid;
	DIR *dirp;
	FILE *fp;
	char hst[CDDBBUFSIZ];
	char rhst[CDDBBUFSIZ];
	char user[CDDBBUFSIZ];
	char clnt[CDDBBUFSIZ];
	char lock[CDDBBUFSIZ];
	struct stat sbuf;
	struct dirent *dp;

	/* Check for lockdir, and create if it doesn't exist. */
	if(stat(lockdir, &sbuf)) {
		if(mkdir(lockdir, (mode_t)dir_mode)) {
			cddbd_log(LOG_ERR, "Failed to create lock dir %s.",
			    lockdir);
			return -1;
		}

		(void)cddbd_fix_file(lockdir, dir_mode, uid, gid);
	}
	else if(!S_ISDIR(sbuf.st_mode)) {
		cddbd_log(LOG_ERR, "%s is not a directory.", lockdir);
		return -1;
	}

	/* Count active users. */
	if((dirp = opendir(lockdir)) == NULL) {
		cddbd_log(LOG_ERR, "Can't open lock dir: %s (%d).",
		    lockdir, errno);
		return -1;
	}

	nus = 0;

	while((dp = readdir(dirp)) != NULL) {
		len = strlen(lockprefix);

		if(!strncmp(dp->d_name, lockprefix, len)) {
			if(sscanf(&dp->d_name[len], ".%d", &pid) != 1)
				continue;

			cddbd_snprintf(lock, sizeof(lock), "%s/%s",
			    lockdir, dp->d_name);

			if(kill((short)pid, 0) != 0 && errno == ESRCH) {
				/* The lock file is stale, remove it. */
				unlink(lock);
			}
			else {
				if(list) {
					fp = fopen(lock, "r");
					if(fp == NULL || fscanf(fp, "%s%s%s%s",
					    user, hst, clnt, rhst) != 4) {
						printf("%-5d  -         ", pid);
						printf("unknown\r\n");
					}
					else {
						clnt[8] = '\0';
						printf("%-5d  %-8s  %s@%s (%s)",
						    pid, clnt, user, hst, rhst);
						printf("\r\n");
					}
				}

				/* Count this user. */
				nus++;
			}
		}
	}

	closedir(dirp);

	return nus;
}


void
cddbd_check_access(void)
{
	int i;
	int n;
	int line;
	int connect_ok;
	FILE *fp;
	log_t *lp;
	arg_t args;
	ctimer_t *tp;
	access_t *at;
	char buf[BUFSIZ];

	/* Set up defaults. */
	for(i = 0; acctab[i].at_fname != 0; i++) {
		at = &acctab[i];

		/* No default for this field. */
		if(at->at_flags & AF_NODEF)
			continue;

		switch(at->at_type) {
		case AC_PATH:
		case AC_STRING:
			strcpy((char *)at->at_addr, (char *)at->at_def);
			break;

		case AC_NUMERIC:
		case AC_USER:
		case AC_GROUP:
		case AC_MODE:
			*(int *)at->at_addr = at->at_def;
			break;

		default:
			cddbd_log(LOG_ERR,
			    "Internal error: bad access type: %d.",
			    acctab[i].at_type);

			quit(QUIT_ERR);
		}
	}

	if(mode == MODE_SERV) {
		get_ok = PERM_DEF_GET;
		write_ok = PERM_DEF_WRITE;
		update_ok = PERM_DEF_UPDATE;
		connect_ok = PERM_DEF_CONN;
	}
	else {
		get_ok = 1;
		write_ok = 1;
		update_ok = 1;
		connect_ok = 1;
	}

	dup_ok = 0;
	dup_policy = DUP_DEFAULT;
	log_flags = (short)LOG_ALL;
	
	uid = geteuid();
	gid = getegid();
	db_uid = uid;
	db_gid = gid;

	if((fp = fopen(accessfile, "r")) == NULL) {
		cddbd_log(LOG_ERR, "Can't open access file: %s (%d).",
		    accessfile, errno);
		quit(QUIT_ERR);
	}

	line = 0;

	while(fgets(args.buf, sizeof(args.buf), fp) != NULL) {
		line++;
		cddbd_parse_args(&args, 0);

		/* Skip comments and blank lines. */
		if(args.nargs == 0 || args.arg[0][0] == '#')
			continue;

		/* Check for generic fields. */
		if(cddbd_parse_access(&args, line))
			continue;

		/* Read permissions. */
		if(!cddbd_strcasecmp("permissions:", args.arg[0])) {
			if(args.nargs != 6)
				cddbd_access_err(line);

			/* Don't scan these if we're just doing DB ops. */
			if(mode != MODE_SERV)
				continue;

			switch(match_host(args.arg[1])) {
			case 0:
				break;

			case 1:
				if(!cddbd_strcasecmp("connect", args.arg[2]))
					connect_ok++;
				else if(!cddbd_strcasecmp("noconnect",
				    args.arg[2]))
					connect_ok = 0;
				else
					cddbd_access_err(line);

				if(!cddbd_strcasecmp("post", args.arg[3]))
					write_ok++;
				else if(!cddbd_strcasecmp("nopost",
				    args.arg[3]))
					write_ok = 0;
				else
					cddbd_access_err(line);

				if(!cddbd_strcasecmp("update", args.arg[4]))
					update_ok++;
				else if(!cddbd_strcasecmp("noupdate",
				    args.arg[4]))
					update_ok = 0;
				else
					cddbd_access_err(line);

				if(!cddbd_strcasecmp("get", args.arg[5]))
					get_ok++;
				else if(!cddbd_strcasecmp("noget", args.arg[5]))
					get_ok = 0;
				else
					cddbd_access_err(line);

				break;

			default:
				cddbd_access_err(line);
			}

			continue;
		}

		/* Scan for timers. */
		for(tp = timers; tp->func != 0; ++tp) {
			cddbd_snprintf(buf, sizeof(buf), "%s_time:", tp->name);

			if(!cddbd_strcasecmp(buf, args.arg[0])) {
				if(args.nargs != 2)
					cddbd_access_err(line);

				if(sscanf(args.arg[1], "%d", &n) != 1)
					cddbd_access_err(line);

				/* Ensure the timeout is legal. */
				if(n < 0) {
					cddbd_log(LOG_ERR,
					    "Illegal %s value, line %d.",
					    tp->name, accessfile);
					quit(QUIT_ERR);
				}

				tp->seconds = n;

				break;
			}
		}

		/* Found a match, so go back and read the next line. */
		if(tp->func != 0)
			continue;

		/* Scan for logging flags. */
		if(!cddbd_strcasecmp("logging:", args.arg[0])) {
			if(args.nargs < 2)
				cddbd_access_err(line);

			log_flags = 0;

			for(i = 1; i < args.nargs; i++) {
				for(lp = log; lp->name; lp++) {
					if(!cddbd_strcasecmp(args.arg[i],
					    lp->name)) {
						if(lp->flag)
							log_flags |= lp->flag;
						else
							log_flags = 0;
						break;
					}
				}

				/* Illegal arg. */
				if(lp->name == 0)
					cddbd_access_err(line);
			}

			continue;
		}

		/* Scan for update flags. */
		if(!cddbd_strcasecmp("dup_policy:", args.arg[0])) {
			if(args.nargs != 2)
				cddbd_access_err(line);

			if(!strcmp(args.arg[1], "never")) {
				dup_policy = DUP_NEVER;
				continue;
			}

			if(!strcmp(args.arg[1], "compare")) {
				dup_policy = DUP_COMPARE;
				continue;
			}

			if(!strcmp(args.arg[1], "always")) {
				dup_policy = DUP_ALWAYS;
				continue;
			}

			cddbd_access_err(line);
		}

		/* Garbage line in access file. */
		cddbd_access_err(line);
	}

	/* Verify what we've found */

	if(logfile[0] == '\0' && log_flags != 0) {
		cddbd_log(LOG_ERR, "No log directory - disabling logging.");
		log_flags = 0;
	}

	if(!connect_ok) {
		cddbd_log(LOG_INFO, "Connection from %s rejected.", rhost);
		printf("432 No connections allowed: permission denied.\r\n");
		_quit(QUIT_OK, 0);
	}

	/* Must have a database dir. */
	if(cddbdir[0] == '\0') {
		cddbd_log(LOG_ERR, "No CD DB directory specified in %s.",
		    accessfile);
		quit(QUIT_ERR);
	}

	if(postdir[0] == '\0' && write_ok) {
		if(verbose) {
			cddbd_log(LOG_ERR,
			    "No post directory - disallowing posting.");
		}

		write_ok = 0;
	}

	if(dupdir[0] == '\0' && write_ok) {
		if(verbose)
			cddbd_log(LOG_ERR, "No dup directory - ignoring dups.");

		dup_ok = 0;
	}
	else if(write_ok)
		dup_ok = 1;

	if(lockdir[0] == '\0') {
		if(verbose) {
			cddbd_log(LOG_ERR,
			    "No lock directory - disallowing locking.");
		}

		max_users = 0;
	}

	if(log_hiwat <= log_lowat && log_hiwat > 0) {
		cddbd_log(LOG_ERR, "Log file low water mark is not lower than"
		    " the high water mark - disabling auto-paring.");

		log_hiwat = 0;
		log_lowat = 0;
	}

	if(log_hiwat == 0 || log_lowat == 0) {
		log_hiwat = 0;
		log_lowat = 0;
	}

	fclose(fp);
}


int
cddbd_parse_access(arg_t *args, int line)
{
	int i;
	int n;
	char *p;
	access_t *at;
	struct group *gr;
	struct passwd *pw;

	for(i = 0; acctab[i].at_fname != 0; i++) {
		at = &acctab[i];
		n = strlen(at->at_fname);

		if(!strncmp(at->at_fname, args->arg[0], n) &&
		    args->arg[0][n] == ':')
			break;
	}

	if(acctab[i].at_fname == 0)
		return 0;

	switch(at->at_type) {
	case AC_PATH:
		if(args->nargs != 2)
			cddbd_access_err(line);

		if((int)strlen(args->arg[1]) > CDDBPATHSZ) {
			cddbd_log(LOG_ERR,
			    "Pathname too long (max %d) on line %d in %s.",
			    CDDBPATHSZ, line, accessfile);

			quit(QUIT_ERR);
		}

		p = (char *)at->at_addr;
		strncpy(p, args->arg[1], CDDBPATHSZ);

		/* Ensure the value is legal. */
		if(p[0] != '/') {
			cddbd_log(LOG_ERR,
			    "Absolute path required for %s on line %d in %s.",
			    at->at_fname, line, accessfile);

			quit(QUIT_ERR);
		}

		break;

	case AC_STRING:
		if(args->nargs != 2)
			cddbd_access_err(line);

		if((int)strlen(args->arg[1]) > CDDBBUFSIZ) {
			cddbd_log(LOG_ERR,
			    "Value too long (max %d) on line %d in %s.",
			    CDDBBUFSIZ, line, accessfile);

			quit(QUIT_ERR);
		}

		p = (char *)at->at_addr;
		strncpy(p, args->arg[1], CDDBBUFSIZ);
		p[CDDBBUFSIZ - 1] = '\0';

		break;

	case AC_NUMERIC:
		if(args->nargs != 2)
			cddbd_access_err(line);

		if(sscanf(args->arg[1], "%d", &n) != 1)
			cddbd_access_err(line);

		/* Ensure the value is positive. */
		if((at->at_flags & AF_ZERO) && n < 0) {
			cddbd_log(LOG_ERR,
			    "Illegal value for %s (%d), line %d in %s.",
			    at->at_fname, n, line, accessfile);
			quit(QUIT_ERR);
		}

		/* Ensure the value is greater than zero. */
		if((at->at_flags & AF_NONZ) && n < 1) {
			cddbd_log(LOG_ERR,
			    "Illegal value for %s (%d), line %d in %s.",
			    at->at_fname, n, line, accessfile);
			quit(QUIT_ERR);
		}

		*(int *)at->at_addr = n;

		break;

	case AC_USER:
		if(args->nargs != 2)
			cddbd_access_err(line);

		if(!strcmp(args->arg[1], "default"))
			break;

		if(isdigit(args->arg[1][0])) {
			*(int *)at->at_addr = atoi(args->arg[1]);
			break;
		}

		/* Ensure the value is legal. */
		if((pw = getpwnam(args->arg[1])) == NULL) {
			cddbd_log(LOG_ERR,
			    "Unknown user \"%s\", line %d in %s.",
			    args->arg[1], line, accessfile);
			quit(QUIT_ERR);
		}

		*(int *)at->at_addr = (int)pw->pw_uid;
		endpwent();

		break;

	case AC_GROUP:
		if(args->nargs != 2)
			cddbd_access_err(line);

		if(!strcmp(args->arg[1], "default"))
			break;

		if(isdigit(args->arg[1][0])) {
			*(int *)at->at_addr = atoi(args->arg[1]);
			break;
		}

		/* Ensure the value is legal. */
		if((gr = getgrnam(args->arg[1])) == NULL) {
			cddbd_log(LOG_ERR,
			    "Unknown group \"%s\", line %d in %s.",
			    args->arg[1], line, accessfile);
			quit(QUIT_ERR);
		}

		*(int *)at->at_addr = (int)gr->gr_gid;
		endgrent();

		break;

	case AC_MODE:
		if(args->nargs != 2)
			cddbd_access_err(line);

		if(sscanf(args->arg[1], "%o", &n) != 1)
			cddbd_access_err(line);

		/* Ensure the value is legal. */
		if((n & 0777) != n) {
			cddbd_log(LOG_ERR,
			    "Illegal mode for %s (%o), line %d in %s.",
			    at->at_fname, n, line, accessfile);
			quit(QUIT_ERR);
		}

		*(int *)at->at_addr = n;

		break;

	default:
		cddbd_log(LOG_ERR, "Internal error: bad access type: %d.",
		    acctab[i].at_type);

		quit(QUIT_ERR);
	}

	return 1;
}


void
cddbd_access_err(int line)
{
	cddbd_log(LOG_ERR, "Syntax error, line %d in %s.", line, accessfile);
	quit(QUIT_ERR);
}


void
cddbd_db_init(void)
{
	int cnt;
	DIR *dirp;
	char *p;
	char buf[CDDBBUFSIZ];
	struct stat sbuf;
	struct dirent *dp;
	static int initted = 0;

	if(initted)
		return;

	initted++;

	if((dirp = opendir(cddbdir)) == NULL) {
		cddbd_log(LOG_ERR, "Can't open database directory: %s (%d).",
		    cddbdir, errno);
		quit(QUIT_ERR);
	}

	cnt = 0;
	while((dp = readdir(dirp)) != NULL && cnt < CDDBMAXDBDIR) {
		/* Categories must not be of illegal length. */
		if((int)strlen(dp->d_name) > CDDBCATEGSZ)
			continue;

		cddbd_snprintf(buf, sizeof(buf), "%s/%s", cddbdir, dp->d_name);

		/* If we can't stat, keep going anyway. */
		if(stat(buf, &sbuf))
			continue;

		/* We only care about directories. */
		if(!S_ISDIR(sbuf.st_mode))
			continue;

		/* No hidden, current or parent directories. */
		if(dp->d_name[0] == '.')
			continue;

		p = (char *)malloc(strlen(dp->d_name) + 1);
		if(p == NULL) {
			cddbd_log(LOG_ERR, "Malloc failed.");
			quit(QUIT_ERR);
		}

		strcpy(p, dp->d_name);
		categlist[cnt] = p;

		cnt++;
	}

	if(cnt == 0)
		cddbd_log(LOG_ERR, "Database is empty.");

	categlist[cnt] = 0;
	categcnt = cnt;

	closedir(dirp);
}


void
cddbd_init(void)
{
	struct passwd *pw;

	/* Initialize locks. */
	lock_log = cddbd_lock_alloc("log");
	lock_nus = cddbd_lock_alloc("nus");
	lock_hash = cddbd_lock_alloc("hash");
	lock_hist = cddbd_lock_alloc("hist");
	lock_tfuzz = cddbd_lock_alloc("tfuzz");
	lock_update = cddbd_lock_alloc("update");

	/* Initialize some global variables. */
	trklen = strlen(trkstr);
	hdrlen = strlen(hdrstr);
	sublen = strlen(substr);
	prclen = strlen(prcstr);

	curpid = getpid();

	pw = getpwuid(geteuid());
	if(pw == 0) {
		strcpy(admin_email, "postmaster");
		strcpy(user, "cddbd");
	}
	else {
		strcpy(admin_email, pw->pw_name);
		strcpy(user, pw->pw_name);
	}
	endpwent();

	if(gethostname(host, sizeof(host)) < 0)
		strcpy(host, "unknown");
}


void
cddbd_input_to(void)
{
        printf("530 %s Inactivity timeout after %d seconds, ",
		host, timers[INPUT_TIMER].seconds);
	printf("closing connection.\r\n");

	cddbd_log(LOG_INFO, "Input timeout.");

        _quit(QUIT_ERR, 0);
}


void
cddbd_access_to(void)
{
        printf("530 %s Database access timeout after %d seconds, ",
		host, timers[ACCESS_TIMER].seconds);
	printf("closing connection.\r\n");

	cddbd_log(LOG_INFO, "Access timeout.");

        _quit(QUIT_ERR, 0);
}


void
cddbd_connect_to(void)
{
        printf("530 %s Server expiration timeout after %d seconds, ",
		host, timers[CONNECT_TIMER].seconds);
	printf("closing connection.\r\n");

	cddbd_log(LOG_INFO, "Connect timeout.");

        _quit(QUIT_ERR, 0);
}


void
cddbd_timer_init(ctimer_t *timers)
{
	ctimer_t *tp;

        /* Reset all timers. */
	for(tp = timers; tp->func != 0; ++tp)
		if(tp->seconds > 0 || !(write_ok && (tp->flags & CT_WRITE_DIS)))
			tp->left = tp->seconds;
		else
			tp->left = -1;

	/* Start clock. */
	lastsecs = time(0);
}


int
cddbd_timer_sleep(ctimer_t *timers)
{
	int n;
	long secs;
	ctimer_t *tp;
	fd_set readfds;
	struct timeval timeout;
	struct timeval *timeoutp;

	/* If we accessed the database, reset timers that care. */
	if(db_access) {
		for(tp = timers; tp->func != 0; ++tp) {
			if((tp->flags & CT_ACCESS_RST) && (tp->left > 0))
				tp->left = tp->seconds;
		}

		db_access = 0;
	}

	/* Length of next timeout is minimum of all timers. */
	timeout.tv_sec = -1;
	timeout.tv_usec = 0;

	for(tp = timers; tp->func != 0; ++tp) {
		if(tp->left >= 0 &&
		    ((tp->left < timeout.tv_sec) || (timeout.tv_sec < 0)))
			timeout.tv_sec = tp->left;
	}

	/* If there are no active timeouts, block until keyboard input. */
	if(timeout.tv_sec < 0)
		timeoutp = 0;
	else
		timeoutp = &timeout;

	/* Do the select. */
	FD_ZERO(&readfds);
	FD_SET(0, &readfds);

	errno = 0;

#ifdef __hpux
	n = select(1, (int *)&readfds, (int *)NULL, (int *)NULL, timeoutp);
#else
	n = select(1, &readfds, (fd_set *)NULL, (fd_set *)NULL, timeoutp);
#endif

	/* "Interrupted system call" isn't a real error. */
	if(n < 0 && errno != EINTR)
		quit(QUIT_OK);

	/* Calculate the number of seconds since the last loop. */
	secs = time(0) - lastsecs;
	if(secs < 0)
		secs = 0;

	/* Subtract time from timers that have time remaining. */
	for(tp = timers; tp->func != 0; ++tp) {
		if(tp->left > 0)
			tp->left -= secs;
		if(tp->left < 0)
			tp->left = 0;
	}

	/* Update lastsecs */
	lastsecs += secs;

	/* If we have input, reset timers that care. */
	if(n > 0) {
		for(tp = timers; tp->func != 0; ++tp) {
			if((tp->flags & CT_INPUT_RST) && tp->left >= 0)
				tp->left = tp->seconds;
		}
	}

	/* Process timers that have expired. */
	for(tp = timers; tp->func != 0; ++tp) {
		if(tp->left == 0) {
			(*tp->func)();
			tp->left = tp->seconds;
		}
	}

	/* Did we find input? */
	return (n > 0);
}


void
do_get(arg_t *args)
{
	printf("500 Command unimplemented.\r\n");
}


void
do_help(arg_t *args)
{
	int i;
	int sub;
	char **hlp;

	args->nextarg++;
	sub = 0;

	if(args->nargs == 1) {
		hlp = 0;
	}
	else {
		for(i = 0; cmd[i].cmd; i++) {
			if(!cddbd_strcasecmp(args->arg[args->nextarg],
			    cmd[i].cmd))
				break;
		}

		if(cmd[i].cmd == 0) {
			printf("500 Unknown HELP subcommand.\r\n");
			return;
		}

		if((args->nargs > 3 && (cmd[i].flags & CF_SUBCMD)) ||
		    (args->nargs > 2 && !(cmd[i].flags & CF_SUBCMD))) {
			printf("500 Too many args to HELP command.\r\n");
			return;
		}

		hlp = cmd[i].help;

		if(cmd[i].flags & CF_SUBCMD && (args->nargs != 3))
			sub = 1;

		if(cmd[i].flags & CF_SUBCMD && (args->nargs == 3)) {
			args->nextarg++;

			for(i = 0; cddb_cmd[i].cmd; i++) {
				if(!cddbd_strcasecmp(args->arg[args->nextarg],
				    cddb_cmd[i].cmd))
					break;
			}

			if(cddb_cmd[i].cmd == 0) {
				printf("500 Unknown CDDB HELP subcommand.\r\n");
				return;
			}

			hlp = cddb_cmd[i].help;
		}
	}

	printf("210 OK, help information follows (until terminating `.')\r\n");

	if(hlp == 0) {
		printf("The following commands are supported:\r\n\r\n");

		for(i = 0; help[i]; i++)
			printf("%s\r\n", help[i][0]);

		for(i = 0; help_info[i]; i++)
			printf("%s\r\n", help_info[i]);
	}
	else {
		for(i = 0; hlp[i]; i++)
			printf("%s\r\n", hlp[i]);

		if(sub)
			for(i = 0; chelp[i]; i++)
				printf("%s\r\n", chelp[i][0]);
	}

	printf(".\r\n");
}


void
do_cddb_query(arg_t *args)
{
	int i;
	FILE *fp;
	db_t *db;
	int ntrks;
	int offtab[CDDBMAXTRK];
	unsigned int discid;
	char buf[CDDBBUFSIZ];

	if(args->nargs < 6 ||
	    sscanf(args->arg[args->nextarg + 1], "%x", &discid) != 1 ||
	    sscanf(args->arg[args->nextarg + 2], "%d", &ntrks) != 1 ||
	    ntrks < 0) {
		printf("500 Command syntax error.\r\n");
		return;
	}

	if(args->nargs != (ntrks + 5)) {
		printf("500 Command syntax error.\r\n");
		return;
	}

	for(i = 0; i < ntrks; i++) {
		if(sscanf(args->arg[args->nextarg + i + 3], "%d",
		    &offtab[i]) != 1 || offtab[i] < 0) {
			printf("500 Command syntax error.\r\n");
			return;
		}
	}

	fp = NULL;
	for(i = 0; categlist[i] != 0; i++) {
		cddbd_snprintf(buf, sizeof(buf), "%s/%s/%08x",
		    cddbdir, categlist[i], discid);

		if((fp = fopen(buf, "r")) != NULL)
			break;
	}

	/* Find a close match if possible. */
	if(fp == NULL) {
		do_cddb_query_fuzzy(args);
		return;
	}

	if((db = db_read(fp, buf, 0)) == 0) {
		cddbd_log(LOG_ERR, "Failed to read DB entry: %s.", buf);
		do_cddb_query_fuzzy(args);
		fclose(fp);

		return;
	}

	fclose(fp);

	if(ntrks != db->db_trks ||
	    !is_fuzzy_match(offtab, db->db_offset, ntrks)) {
		cddbd_log(LOG_INFO, "Discid collision on DB entry: %08x.",
		    discid);

		db_free(db);
		do_cddb_query_fuzzy(args);

		return;
	}

	db_strcpy(db, DP_DTITLE, 0, buf, sizeof(buf));
	db_free(db);
	strip_crlf(buf);

	printf("200 %s %08x %s\r\n", categlist[i], discid, buf);
	cddbd_log(LOG_ACCESS | LOG_QUERY, "Query: %08x successful",
	    discid);
}


void
do_cddb_read(arg_t *args)
{
	FILE *fp;
	char *category;
	char buf[CDDBBUFSIZ];
	char file[CDDBBUFSIZ];
	unsigned int discid;

	category = args->arg[++args->nextarg];

	if(args->nargs != 4 ||
	    sscanf(args->arg[++args->nextarg], "%08x", &discid) != 1) {
		printf("500 Command syntax error.\r\n");
		return;
	}

	cddbd_snprintf(file, sizeof(file), "%s/%s/%08x", cddbdir, category,
	    discid);

	if((fp = fopen(file, "r")) == NULL) {
		printf("401 %s %08x No such CD entry in database.\r\n",
		    category, discid);
		cddbd_log(LOG_INFO, "Read: %s %08x failed", category, discid);

		return;
	}

	printf("210 %s %08x CD database entry follows ", category, discid);
	printf("(until terminating `.')\r\n");

	while(fgets(buf, sizeof(buf), fp) != NULL) {
		strip_crlf(buf);
		printf("%s\r\n", buf);
	}

	printf(".\r\n");
	fclose(fp);

	cddbd_log(LOG_ACCESS | LOG_READ, "Read: %s %08x", category, discid);
}


void
do_cddb_write(arg_t *args)
{
	db_t *db;
	char *category;
	char pdir[CDDBBUFSIZ];
	char errstr[CDDBBUFSIZ];
	unsigned int discid;
	struct stat sbuf;

	category = args->arg[++args->nextarg];

	if(args->nargs != 4 ||
	    sscanf(args->arg[++args->nextarg], "%08x", &discid) != 1) {
		printf("500 Command syntax error.\r\n");
		return;
	}

	if(!write_ok) {
		printf("401 Permission denied.\r\n");
		return;
	}

	/* Check for postdir, and create if it doesn't exist. */
	if(stat(postdir, &sbuf)) {
		if(mkdir(postdir, (mode_t)db_dir_mode)) {
			cddbd_log(LOG_ERR, "Failed to create post dir %s.",
			    postdir);
			printf("402 File access failed.\r\n");
			return;
		}

		(void)cddbd_fix_file(postdir, db_dir_mode, db_uid, db_gid);
	}
	else if(!S_ISDIR(sbuf.st_mode)) {
		cddbd_log(LOG_ERR, "%s is not a directory.", postdir);
		printf("402 File access failed.\r\n");
		return;
	}

	if(categ_index(category) < 0) {
		printf("501 Invalid category: %s.\r\n", category);
		return;
	}

	printf("320 OK, input CDDB data (terminate with `.')\r\n");
	fflush(stdout);

	if((db = db_read(stdin, errstr, DF_STDIN)) == 0) {
		if(db_errno == DE_INVALID)
			printf("501 Entry rejected: %s.\r\n", errstr);
		else {
			printf("402 Internal server error: %s.\r\n",
			    db_errmsg[db_errno]);
			cddbd_log(LOG_ERR, "Failed to write DB entry: %s.",
			    db_errmsg[db_errno]);
		}

		return;
	}

	cddbd_snprintf(pdir, sizeof(pdir), "%s/%s", postdir, category);
	(void)db_post(db, pdir, discid, errstr);

	switch(db_errno) {
	case DE_NO_ERROR:
		cddbd_log(LOG_WRITE, "Write: (via CDDBP - %s) %s %08x",
		    rhost, category, discid);
		printf("200 CDDB entry accepted.\r\n");
		break;

	case DE_INVALID:
		cddbd_log(LOG_ERR, "Failed to write DB entry: %s.", errstr);
		printf("501 Entry rejected: %s.\r\n", errstr);
		break;

	default:
		cddbd_log(LOG_ERR, "Failed to write DB entry: %s.", errstr);
		printf("402 Server error: %s.\r\n", errstr);
		break;
	}

	db_free(db);
}



/* ARGSUSED */
void
do_quit(arg_t *args)
{
	quit(QUIT_OK);
}


/* ARGSUSED */
void
do_cddb_srch(arg_t *args)
{
	printf("500 Command unimplemented.\r\n");
}


/* ARGSUSED */
void
do_ver(arg_t *args)
{
	printf("200 ");
	printf(verstr, VERSION, PATCHLEVEL);
	printf("\r\n");
}


void
do_proto(arg_t *args)
{
	int nlevel;

	if(args->nargs > 2) {
		printf("500 Command syntax error.\r\n");
		return;
	}

	if(args->nargs == 1) {
		printf("200 CDDB protocol level: current %d, supported %d\r\n",
		    level, MAX_PROTO_LEVEL);
		return;
	}

	if(sscanf(args->arg[1], "%d", &nlevel) != 1) {
		printf("500 Command syntax error.\r\n");
		return;
	}

	if(nlevel < MIN_PROTO_LEVEL || nlevel > MAX_PROTO_LEVEL) {
		printf("501 Illegal CDDB protocol level.\r\n");
		return;
	}

	if(nlevel == level) {
		printf("502 CDDB protocol level already %d.\r\n", level);
		return;
	}

	level = nlevel;
	printf("201 OK, CDDB protocol level now: %d\r\n", level);
}


/* ARGSUSED */
void
do_whom(arg_t *args)
{
	if(cddbd_nus(0) < 0) {
		printf("401 No user information available.\r\n");
		return;
	}

	printf("210 OK, user list follows (until terminating `.')\r\n");
	printf("Pid:   Client:   User:\r\n");

	(void)cddbd_nus(1);

	printf(".\r\n");
}


void
do_update(arg_t *args)
{
	int f;

	if(args->nargs > 1) {
		printf("500 Command syntax error.\r\n");
		return;
	}

	if(!update_ok) {
		printf("401 Permission denied.\r\n");
		return;
	}

	f = cddbd_fork();

	/* The child does the update. */
	if(f == 0) {
		close(0);
		close(1);
		close(2);

		cddbd_update();
		cddbd_build_fuzzy();

		return;
	}

	if(f < 0) {
		printf("402 Unable to update the database.\r\n");
		cddbd_log(LOG_ERR, "Can't fork child for update (%d).", errno);

		return;
	}

	printf("200 Updating the database.\r\n");
}


void
do_stat(arg_t *args)
{
	char *p;
	int i;
	int cnt;
	int *counts;
	int first;
	site_t *sp;

	if(args->nargs > 1) {
		printf("500 Command syntax error.\r\n");
		return;
	}

	skip_log++;

	printf("210 OK, status information follows ");
	printf("(until terminating `.')\r\n");

	printf("Server status:\r\n");
	printf("    current proto: %d\r\n", level);
	printf("    max proto: %d\r\n", MAX_PROTO_LEVEL);

	p = (get_ok ? "yes" : "no");
	printf("    gets: %s\r\n", p);

	p = (update_ok ? "yes" : "no");
	printf("    updates: %s\r\n", p);

	p = (write_ok ? "yes" : "no");
	printf("    posting: %s\r\n", p);

	p = (PROTO_ENAB(P_QUOTE) ? "yes" : "no");
	printf("    quotes: %s\r\n", p);

	if(write_ok)
		printf("    max lines: %d\r\n", max_lines);

	cnt = cddbd_nus(0);
	if(cnt >= 0) {
		printf("    current users: %d\r\n", cnt);
		printf("    max users: %d\r\n", max_users);
	}

	counts = cddbd_count();
	printf("Database entries: %d\r\n", counts[0]);

	printf("Database entries by category:\r\n");
	for(i = 0; i < categcnt; i++)
		printf("    %s: %d\r\n", categlist[i], counts[i + 1]);

	first = 1;

	while((sp = getsiteent(SITE_XMIT)) != NULL) {
		if(first) {
			first = 0;
			printf("Pending file transmissions:\r\n");
		}

		cnt = cddbd_count_history(sp->st_name);
		if(cnt < 0)
			printf("    %s: -\r\n", sp->st_name);
		else
			printf("    %s: %d\r\n", sp->st_name, cnt);
	}

	endsiteent();

	printf(".\r\n");
}


void
do_sites(arg_t *args)
{
	int cnt;
	site_t *sp;
	char lat[CDDBBUFSIZ];
	char lng[CDDBBUFSIZ];

	if(args->nargs > 1) {
		printf("500 Command syntax error.\r\n");
		return;
	}

	skip_log++;

	setsiteent();

	for(cnt = 0; (sp = getsiteent(SITE_INFO)) != NULL; cnt++) {
		if(cnt == 0) {
			printf("210 OK, site information follows ");
			printf("(until terminating `.')\r\n");
		}

		copy_coord(lat, &sp->st_lat);
		copy_coord(lng, &sp->st_long);

		printf("%s %d %s %s %s\r\n", sp->st_name, sp->st_port, lat,
		    lng, sp->st_desc);
	}

	endsiteent();

	if(cnt == 0) {
		printf("401 No site information available.\r\n");
		return;
	}
	else
		printf(".\r\n");
}


void
do_cddb_hello(arg_t *args)
{
	FILE *fp;
	char *username;
	char *clientname;
	char *hostname;
	char *version;

	if(args->nargs != 6) {
		printf("500 Command syntax error.\r\n");
		return;
	}

	if(hello) {
		printf("402 Already shook hands.\r\n");
		return;
	}

	hello++;

	username = args->arg[++args->nextarg];
	hostname = args->arg[++args->nextarg];
	clientname = args->arg[++args->nextarg];
	version = args->arg[++args->nextarg];

	printf("200 Hello and welcome %s@%s running %s %s.\r\n",
	    username, hostname, clientname, version);

	cddbd_log(LOG_HELLO, hellostr, username, hostname, rhost, clientname,
	    version);

	if(lockdir[0] != '\0' && max_users != 0 &&
	    (fp = fopen(lockfile, "w")) != NULL) {
		fprintf(fp, "%s %s %s %s ",
		    username, hostname, clientname, rhost);
		fclose(fp);
	}
}


void
do_motd(arg_t *args)
{
	int len;
	FILE *fp;
	struct stat sbuf;
	char buf[CDDBBUFSIZ];

	if(args->nargs != 1) {
		printf("500 Command syntax error.\r\n");
		return;
	}

	if(motdfile[0] == '\0' || stat(motdfile, &sbuf) != 0) {
		printf("401 No message of the day available.\r\n");
		return;
	}

	if((fp = fopen(motdfile, "r")) == NULL) {
		cddbd_log(LOG_ERR, "Can't open motd file: %s (%d).",
		    motdfile, errno);
		printf("401 No message of the day available.\r\n");
		return;
	}

	printf("210 Last modified: %s ", make_time2(localtime(&sbuf.st_mtime)));
	printf("MOTD follows (until terminating `.')\r\n");

	while(fgets(buf, sizeof(buf), fp) != NULL) {
		len = strlen(buf);
		if(buf[len - 1] == '\n')
		    buf[len - 1] = '\0';

		if(buf[0] == '.')
			printf(">");

		printf("%s\r\n", buf);
	}

	printf(".\r\n");

	fclose(fp);
}


void
do_log(arg_t *args)
{
	FILE *fp;
	int i;
	int x;
	int days;
	int get;
	int pid;
	int d[7];
	int flag;
	int incl;
	int lines;
	int nclient;
	int log_all;
	int other_cnt;
	int ctab[NCLIENT];
	char buf[CDDBBUFSIZ];
	char user[CDDBBUFSIZ];
	char site[CDDBBUFSIZ];
	char cur[CDDBBUFSIZ];
	char end[CDDBBUFSIZ];
	char start[CDDBBUFSIZ];
	char first[CDDBBUFSIZ];
	char last[CDDBBUFSIZ];
	char tmp_client[CDDBBUFSIZ];
	char *p;
	struct stat sbuf;
	lhead_t *lh;
	time_t t;


	days = 0;
	get = 0;

	if(args->nargs > 1) {
		if(!strcmp(args->arg[1], "day")) {
			if(args->nargs == 3) {
				days = atoi(args->arg[2]);
				if(days <= 0) {
					printf("500 Bad day count: %d.\r\n",
					    days);
					return;
				}
			}
			else
				days = 1;

			t = time(0);
			cvt_time(t - (days * SECS_PER_DAY), start);
			cvt_time(t, end);
		}
		else
			if(!strcmp(args->arg[1], "get"))
				get = 1;
	}

	if(args->nargs > 3 || (args->nargs > 2 && get)) {
		printf("500 Command syntax error.\r\n");
		return;
	}

	/* Lock the log. */
	(void)cddbd_lock(lock_log, 1);

	if(log_flags == 0 || (fp = fopen(logfile, "r")) == NULL) {
		cddbd_unlock(lock_log);
		printf("402 No log information available.\r\n");
		return;
	}

	if(stat(logfile, &sbuf))
		sbuf.st_size = 0;

	/* Unlock the log. It's safe now, since we have a file descriptor. */
	cddbd_unlock(lock_log);

	skip_log++;

	if(get) {
		if(!get_ok) {
			printf("401 Permission denied.\r\n");
			return;
		}

		cddbd_log(LOG_INFO, "Log downloaded from %s.", rhost);

		printf("211 OK, log follows (until terminating `.')\r\n");
		while(fgets(buf, sizeof(buf), fp) && fputs(buf, stdout) != EOF)
			continue;
		printf(".\r\n");

		fclose(fp);

		return;
	}

	log_all = 0;

	if(days == 0) {
		if(args->nargs == 3) {
			if(!cvt_date(args->arg[2], end)) {
				printf("501 Invalid end date.\r\n");
				return;
			}
		}
		else
			(void)cvt_date("", end);

		if(args->nargs >= 2) {
			if(!cvt_date(args->arg[1], start)) {
				printf("501 Invalid start date.\r\n");
				return;
			}
		}
		else
			log_all = 1;
	}

	/* End is earlier than start. */
	if(!log_all && strcmp(start, end) > 0) {
		printf("501 End date is earlier than start date.\r\n");
		return;
	}

	if(log_all) {
		first[0] = '\0';
		last[0] = '\0';
	}

	for(i = 0; log[i].name; i++)
		log[i].cnt = 0;

	lines = 0;
	nclient = 0; 
	other_cnt = 0; 

	if((lh = list_init(0, comp_user, free_user, 0)) == 0) {
		cddbd_log(LOG_ERR, "Can't allocate list head.\n");
		quit(QUIT_ERR);
	}

	while(fgets(buf, sizeof(buf), fp)) {
		x = sscanf(buf, "%d/%d/%d%d:%d:%d [%d,%x]", &d[0], &d[1],
		    &d[2], &d[3], &d[4], &d[5], &pid, &flag);

		/* Bogus log string. */
		if(x != 8)
			continue;

		/* Safe to use sprintf. Need speed. */
		sprintf(cur, "%04d%04d%04d%04d%04d%04d",
		    make_year(d[2]), d[0], d[1], d[3], d[4], d[5]);

		/* Bogus log date. */
		if(date_to_tm(cur) == 0)
			continue;

		/* Note the first and last message. */
		if(strcmp(first, cur) > 0 || first[0] == '\0')
			strcpy(first, cur);

		if(strcmp(last, cur) < 0 || last[0] == '\0')
			strcpy(last, cur);

		if(log_all || (strcmp(cur, start) >= 0 &&
		    strcmp(cur, end) <= 0))
			incl = 1;
		else
			incl = 0;

		if(incl)
			lines++;

		/* Log the message if it's acceptable. */
		for(i = 0; log[i].name; i++) {
			if((log[i].flag & flag) && incl)
				log[i].cnt++;
		}

		/* Count clients. */
		if((flag & LOG_HELLO) && incl) {
			x = sscanf(buf, hellosstr, user, site, tmp_client);
			if(x != 3)
				strcpy(tmp_client, "Unknown");

			cddbd_snprintf(buf, sizeof(buf), "%s/%s", user, site);

			if(x == 3 && list_find(lh, (void *)buf) == 0) {
				if((p = strdup(buf)) == NULL) {
					cddbd_log(LOG_ERR,
					    "Can't allocate user string.");
					quit(QUIT_ERR);
				}

				if(list_add_cur(lh, (void *)p) == 0) {
					cddbd_log(LOG_ERR,
					    "Can't allocate list entry.");
					quit(QUIT_ERR);
				}
			}

			for(i = 0; i < nclient; i++) {
				if(!strcmp(tmp_client, client[i])) {
					client_cnt[i]++;
					break;
				}
			}

			if(i == nclient) {
				if(nclient < NCLIENT) {
					client_cnt[nclient] = 1;
					strcpy(client[nclient], tmp_client);
					nclient++;
				}
				else
					other_cnt++;
			}
		}
	}

	fclose(fp);

	printf("210 OK, log summary follows (until terminating `.')\r\n");

	if(log_all) {
		if(first[0] != '\0') {
			strcpy(start, make_time2(date_to_tm(first)));
			strcpy(end, make_time2(date_to_tm(last)));
			printf("Log status between %s and %s\r\n", start, end);
		}
	}
	else {
		strcpy(start, make_time2(date_to_tm(start)));
		strcpy(end, make_time2(date_to_tm(end)));
		printf("Log status between: %s and %s\r\n", start, end);
	}

	for(i = 0; log[i].name; i++)
		if(log[i].banr != 0 && !((log[i].dflag & L_NOSHOW) &&
		    log[i].cnt == 0))
			printf("%s: %d\r\n", log[i].banr, log[i].cnt);

	if(nclient > 0) {
		for(i = 0; i < nclient; i++)
			ctab[i] = i;

		qsort((void *)ctab, nclient, sizeof(ctab[0]), comp_client);

		printf("Connections by client type:\r\n");

		for(i = 0; i < nclient; i++)
			printf("    %s: %d\r\n", client[ctab[i]],
			    client_cnt[ctab[i]]);

		if(other_cnt > 0)
			printf("    Other: %d\r\n", other_cnt);
	}

	printf("Users: %d\r\n", list_count(lh));
	list_free(lh);

	printf("Total messages: %d\r\n", lines);
	if(sbuf.st_size != 0)
		printf("Log size: %d bytes\r\n", sbuf.st_size);

	printf(".\r\n");
}


void
free_user(void *u)
{
	free(u);
}


int
comp_user(void *u1, void *u2)
{
	return(cddbd_strcasecmp((char *)u1, (char *)u2));
}


int
comp_client(void *c1, void *c2)
{
	int x;

	x = client_cnt[*(int *)c2] - client_cnt[*(int *)c1];
	if(x != 0)
		return(x);

	return(cddbd_strcasecmp(client[*(int *)c1], client[*(int *)c2]));
}


void
do_cddb(arg_t *args)
{
	cmd_t *c;

	if(args->nargs < 2) {
		printf("500 Command syntax error.\r\n");
		return;
	}

	args->nextarg++;

	/* Execute the cddb command. */
	for(c = cddb_cmd; c->cmd; c++) {
		if(cddbd_strcasecmp(args->arg[args->nextarg], c->cmd))
			continue;

		/* Ensure we have a handshake. */
		if((c->flags & CF_HELLO) && !hello) {
			printf("409 No handshake.\r\n");
			return;
		}

		/* Note database accesses. */
		if(c->flags & CF_ACCESS)
			db_access++;

		(*(c->func))(args);

		return;
	}

	/* No valid command found. */
	printf("500 Unrecognized command.\r\n");
}


char *
get_time(int type)
{
	time_t now;
	now = time(0);

	if(type)
		return make_time(localtime(&now));
	else
		return make_time2(localtime(&now));
}


char *
make_time(struct tm *tm)
{
	static char buf[CDDBBUFSIZ];

	cddbd_snprintf(buf, sizeof(buf), "%s %s %02d %02d:%02d:%02d %d",
	    day[tm->tm_wday],
	    month[tm->tm_mon].name,
	    tm->tm_mday,
	    tm->tm_hour,
	    tm->tm_min,
	    tm->tm_sec,
	    (tm->tm_year + 1900));

	return buf;
}


char *
make_time2(struct tm *tm)
{
	static char buf[CDDBBUFSIZ];

	cddbd_snprintf(buf, sizeof(buf), "%02d/%02d/%02d %02d:%02d:%02d",
	    (tm->tm_mon + 1),
	    tm->tm_mday,
	    tm->tm_year,
	    tm->tm_hour,
	    tm->tm_min,
	    tm->tm_sec);

	return buf;
}


int
make_year(int year)
{
	if(year <= 99 && year >= 70)
		year += 1900;
	else if(year < 70 && year >= 0)
		year += 2000;

	return year;
}


void
cddbd_parse_args(arg_t *args, int flags)
{
	int i;
	char *p;
	int isarg;
	int isquote;

	args->nargs = 0;
	args->nextarg = 0;

	for(p = args->buf, isarg = 0, isquote = 0; *p != '\0'; p++) {
		if(*p == '"' && (flags & PF_HONOR_QUOTE)) {
			isquote = !isquote;

			if(isarg) {
				*p = '\0';
				isarg = 0;
			}

			continue;
		}

		if(isspace(*p) && !isquote) {
			if(isarg) {
				*p = '\0';
				isarg = 0;
			}
		}
		else {
			if(isspace(*p) && (flags & PF_REMAP_QSPC))
				*p = '_';

			if(!isarg) {
				args->arg[args->nargs] = p;
				args->nargs++;
				isarg++;
			}
		}
	}

	/* Ensure that all args are limited in length. */
	for(i = 0; i < args->nargs; i++)
		if((int)strlen(args->arg[i]) > CDDBARGSZ)
			args->arg[i][CDDBARGSZ] = '\0';
}


void
usage(char *name)
{
	char *n;

	n = strrchr(name, '/');
	if(n == NULL)
		n = name;
	else
		n++;

	printf("usage: %s [-Cc <level|\"default\">] [-uf] ", n);
	printf("[-l|t <rhost|\"all\">> |\n");
	printf("       <-T <rhost|\"all\"> <hh[mm[ss[MM[DD[YY]]]]]>] ");
	printf("[-dqv] [-a access_file]\n");
	printf("or:    %s <-m|-M> [-dqv] [-a access_file]\n", n);
	printf("or:    %s -s <port|\"default\"> [-dqv] [-a access_file]\n", n);

	exit(QUIT_ERR);
}


struct tm *
date_to_tm(char *date)
{
	int x;
	int d[6];
	static struct tm tm;

	x = sscanf(date, "%04d%04d%04d%04d%04d%04d", &d[0], &d[1],
	    &d[2], &d[3], &d[4], &d[5]);

	if(x != 6)
		return 0;

	/* Check the year. */
	if(d[0] < 0)
		return 0;

	tm.tm_year = d[0] - 1900;

	/* Check the month. */
	d[1]--;
	if(d[1] > 11 || d[1] < 0)
		return 0;

	tm.tm_mon = d[1];

	/* Check the day, including leap years. */
	x = month[d[1]].ndays;
	if(d[2] > x || d[2] < 1 || (d[1] == LEAPMONTH && d[2] == LEAPDAY &&
	    !LEAPYEAR(d[0])))
		return 0;

	tm.tm_mday = d[2];

	/* Check the hour. */
	if(d[3] > 23 || d[3] < 0)
		return 0;

	tm.tm_hour = d[3];

	/* Check the minute. */
	if(d[4] > 59 || d[4] < 0)
		return 0;

	tm.tm_min = d[4];

	/* Check the seconds, allowing for leap seconds. */
	if(d[5] > 61 || d[5] < 0)
		return 0;

	tm.tm_sec = d[5];

	return &tm;
}


int
cvt_date(char *date, char *buf)
{
	int i;
	int x;
	int d[6];
	int ord[6] = { 3, 4, 5, 1, 2, 0 };
	time_t now;
	char tmp[5];
	struct tm *tm;

	now = time(0);
	tm = localtime(&now);


	/* Compute the date for today. */
	cddbd_snprintf(buf, CDDBBUFSIZ, "%04d%04d%04d%04d%04d%04d",
	    make_year(tm->tm_year + 1900), (++tm->tm_mon), tm->tm_mday,
	    tm->tm_hour, tm->tm_min, tm->tm_sec);

	/* Compute the user's date. */
	x = sscanf(date, "%02d%02d%02d%02d%02d%d", &d[0], &d[1],
	    &d[2], &d[3], &d[4], &d[5]);

	if(x == EOF)
		return 0;

	d[5] = make_year(d[5]);

	/* Generate a new date based on the user's input. */
	for(i = 0; i < x; i++) {
		cddbd_snprintf(tmp, sizeof(tmp), "%04d", d[i]);
		strncpy(&buf[(ord[i] * 4)], tmp, 4);
	}

	return(date_to_tm(buf) != 0);
}


void
cvt_time(time_t t, char *buf)
{
	struct tm *tm;

	tm = localtime(&t);

	/* Compute the date. */
	cddbd_snprintf(buf, CDDBBUFSIZ, "%04d%04d%04d%04d%04d%04d",
	    make_year(tm->tm_year + 1900), (++tm->tm_mon), tm->tm_mday,
	    tm->tm_hour, tm->tm_min, tm->tm_sec);
}


void
cddbd_lock_free_lk(void *lk)
{
	cddbd_unlock(lk);
	free(lk);
}


void
cddbd_lock_free(clck_t *lk)
{
	link_t *lp;

	if((lp = list_find(lock_head, lk)) == 0) {
		cddbd_log(LOG_ERR | LOG_LOCK,
		    "Attempt to free unknown lock %s.", lk->lk_name);
		quit(QUIT_ERR);
	}

	list_delete(lock_head, lp);
	cddbd_lock_free_lk(lk);
}


clck_t *
cddbd_lock_alloc(char *name)
{
	clck_t *lk;

	if(!lock_initted) {
		lock_initted++;

		if((lock_head = list_init(0, 0, cddbd_lock_free_lk, 0)) == 0) {
			cddbd_log(LOG_ERR | LOG_LOCK,
			    "Can't allocate lock head.");
			quit(QUIT_ERR);
		}
	}

	if((int)strlen(name) > CDDBLCKNAMLEN) {
		cddbd_log(LOG_ERR | LOG_LOCK, "Lock name %s too long.",
		    name);
		quit(QUIT_ERR);
	}

	if((lk = (clck_t *)malloc(sizeof(clck_t))) == NULL) {
		cddbd_log(LOG_ERR | LOG_LOCK, "Can't allocate %s lock memory.",
		    name);
		quit(QUIT_ERR);
	}

	strcpy(lk->lk_name, name);
	lk->lk_refcnt = 0;

	if(list_add_back(lock_head, (void *)lk) == 0) {
		cddbd_log(LOG_ERR | LOG_LOCK, "Can't allocate %s lock.",
		    name);
		quit(QUIT_ERR);
	}

	return lk;
}


int
cddbd_locked(clck_t *lk)
{
	return(lk->lk_refcnt > 0);
}


int
cddbd_lock(clck_t *lk, int dowait)
{
	int i;
	int fd;
	int cnt;
	int pid;
	int len;
	DIR *dirp;
	struct stat sbuf;
	struct dirent *dp;
	char myname[CDDBBUFSIZ];
	char hisname[CDDBBUFSIZ];

	/* No lock dir. Just return. */
	if(lockdir[0] == '\0' || lk == 0)
		return 1;

	/* Check for lockdir, and create if it doesn't exist. */
	if(stat(lockdir, &sbuf)) {
		if(mkdir(lockdir, (mode_t)dir_mode)) {
			cddbd_log(LOG_ERR | LOG_LOCK,
			    "Failed to create lock dir %s.", lockdir);
			quit(QUIT_ERR);
		}

		(void)cddbd_fix_file(lockdir, dir_mode, uid, gid);
	}
	else if(!S_ISDIR(sbuf.st_mode)) {
		cddbd_log(LOG_ERR | LOG_LOCK, "%s is not a directory.",
		    lockdir);
		quit(QUIT_ERR);
	}

	/* Already have the lock. */
	if(lk->lk_refcnt > 0) {
		lk->lk_refcnt++;
		return 1;
	}
		
	len = strlen(lk->lk_name);
	cddbd_snprintf(myname, sizeof(myname), "%s/%s.%05d", lockdir,
	    lk->lk_name, curpid);

	/* Try to acquire the lock in a loop, forever. */
	for(i = 0; i < MAX_LOCK_LOOP; i++) {
		if((fd = open(myname, (O_WRONLY | O_CREAT), file_mode)) < 0) {
			cddbd_log(LOG_ERR | LOG_LOCK,
			    "Can't create lock file: %s (%d).", myname, errno);
			quit(QUIT_ERR);
		}

		close(fd);

		/* Scan for inactive links. */
		if((dirp = opendir(lockdir)) == NULL) {
			cddbd_log(LOG_ERR | LOG_LOCK,
			    "Can't open lock dir: %s (%d).", lockdir, errno);
			quit(QUIT_ERR);
		}

		/* Count lock files and remove inactive links. */
		cnt = 0;

		while((dp = readdir(dirp)) != NULL) {
			if(!strncmp(dp->d_name, lk->lk_name, len)) {
				if(sscanf(&dp->d_name[len+1], "%d", &pid) != 1)
					continue;

				/* If the lock file is stale, remove it. */
				if(kill((short)pid, 0) != 0 && errno == ESRCH) {
					cddbd_snprintf(hisname, sizeof(hisname),
					    "%s/%s", lockdir, dp->d_name);
					unlink(hisname);
				}
				else
					cnt++;
			}
		}

		closedir(dirp);

		/* We own the lock. */
		if(cnt == 1) {
			lk->lk_refcnt = 1;
			return 1;
		}

		unlink(myname);

		/* Leave if we're not waiting for the lock. */
		if(!dowait)
			return 0;

		sleep(1);
	}

	/* We failed to get the lock after a lot of tries. */
	if(i == MAX_LOCK_LOOP) {
		cddbd_log(LOG_ERR | LOG_LOCK, "Failed to acquire lock: %s",
		    lk->lk_name);

		quit(QUIT_ERR);
	}

	return 0;
}


void
cddbd_unlock(clck_t *lk)
{
	char myname[CDDBBUFSIZ];

	if(lk->lk_refcnt == 0 || --(lk->lk_refcnt) > 0)
		return;

	cddbd_snprintf(myname, sizeof(myname), "%s/%s.%05d", lockdir,
	    lk->lk_name, curpid);

	unlink(myname);
}


int
cddbd_link(char *old, char *new)
{
	struct stat sbuf1;
	struct stat sbuf2;

	link(old, new);

	if(stat(new, &sbuf1) != 0 || stat(old, &sbuf2) != 0 ||
	    sbuf1.st_ino != sbuf2.st_ino)
		return 1;

	return 0;
}


int
cddbd_fork(void)
{
	int f;
	link_t *lp;
	clck_t *lk;

	if((f = fork()) < 0)
		return f;

	if(f == 0) {
		curpid = getpid();
		child++;

		/* Free any pending locks, since they're not inherited. */
		for(list_rewind(lock_head), list_forw(lock_head);
		    !list_rewound(lock_head); list_forw(lock_head)) {
			lp = list_cur(lock_head);
			lk = (clck_t *)lp->l_data;
			lk->lk_refcnt = 0;
		}

		/* Zero out per-process filenames. */
		lockfile[0] = '\0';
		uhistfile[0] = '\0';
	}

	return f;
}


int
cddbd_fix_file(char *file, int mode, int uid, int gid)
{
	if(chmod(file, mode) != 0) {
		cddbd_log(LOG_ERR, "Can't change permissions on: %s (%d).",
		    file, errno);
		return 1;
	}

	if(chown(file, uid, gid) != 0) {
		cddbd_log(LOG_ERR, "Can't change owner/group on: %s (%d).",
		    file, errno);
		return 1;
	}

	return 0;
}


char *
cddbd_gets(char *buf, int cnt)
{
	int c;

	while((c = cddbd_getchar()) >= 0) {
		*buf = (char)c;

		buf++;
		cnt--;

		if(c == '\n' || cnt == 1) {
			*buf = '\0';
			return buf;
		}
	}

	return NULL;
}


int
cddbd_getchar(void)
{
	static int i = 0;
	static int e = 0;
	static unsigned char buf[BUFSIZ];

	if(i == e) {
		while(!cddbd_timer_sleep(timers))
			continue;

		e = read(0, buf, sizeof(buf));
		if(e <= 0) {
			e = i;
			return -1;
		}

		i = 1;
	}
	else
		i++;

	return(buf[i - 1]);
}


void
trunc_log(void)
{
	int fd;
	int len;
	char *p;
	char *buf;
	struct stat sbuf;

	/* Truncate the log if it's too long. */
	if(stat(logfile, &sbuf) != 0) {
		cddbd_log(LOG_ERR, "Can't stat logfile: %s (%d).", logfile);
		return;
	}

	len = sbuf.st_size;
	if(len < log_hiwat || log_hiwat == 0) {
		return;
	}

	/* Get a buffer for the file. */
	if((buf = (char *)malloc(len)) == 0) {
		cddbd_log(LOG_ERR, "Can't malloc %d bytes for logfile (%d).",
		    (void *)len, errno);
		return;
	}

	if((fd = open(logfile, O_RDONLY)) < 0) {
		cddbd_log(LOG_ERR, "Can't open logfile %s for reading (%d).",
		    logfile, (void *)errno);
		free(buf);
		return;
	}

	/* Read in the whole log file. */
	if(read(fd, buf, len) != len) {
		cddbd_log(LOG_ERR, "Can't read logfile: %s (%d).", logfile,
		    (void *)errno);
		close(fd);
		free(buf);
		return;
	}

	close(fd);
	
	p = &buf[len - log_lowat];
	while(*p != '\n' && p < &buf[len])
		p++;
	if(*p == '\n')
		p++;

	len -= (int)(p - buf);

	if(len <= 0) {
		cddbd_log(LOG_ERR, "Log file %s seems corrupt.", logfile);
		free(buf);
		return;
	}

	if((fd = open(logfile, O_TRUNC | O_WRONLY)) < 0) {
		cddbd_log(LOG_ERR, "Can't open logfile %s for writing (%d).",
		    logfile, (void *)errno);
		free(buf);
		return;
	}

	/* Write out the truncated file. */
	if(write(fd, p, len) != len) {
		cddbd_log(LOG_ERR, "Can't write logfile: %s (%d).",
		    logfile, (void *)errno);
	}

	close(fd);
	free(buf);

	return;
}


#define SCOPY(p, buf) if(cc < (CDDBBUFSIZ-1)) {(buf)[cc] = **(p),(*(p))++,cc++;}
#define SASGN(c, buf) if(cc < (CDDBBUFSIZ-1)) {(buf)[cc] = c, cc++;}
#define SPUT(c, buf) (buf)[cc] = c, cc++


int
get_conv(char **p, char *buf)
{
	int cc;
	int acnt;
	int pre1;
	int pre2;
	char c;
	char m;
	char *cp;
	char *p2;
	char cbuf[CDDBBUFSIZ];
	char *mod = "hlL";
	char *conv = "cdDeEfginoOpsuUxX%";

	cc = 0;
	acnt = 0;

	SCOPY(p, buf);

	if(**p == '%') {
		SCOPY(p, buf);
		SPUT('\0', buf);
		return 0;
	}

	p2 = *p;

	while(*p2 != '\0')
		if((cp = strchr(conv, *p2)) == 0)
			p2++;
		else
			break;

	if(*p2 == '\0') {
		strncpy(buf, *p, (CDDBBUFSIZ - 1));
		buf[CDDBBUFSIZ - 1] = '\0';
		*p = p2;
		return 0;
	}

	c = *cp;

	if((cp = strchr(mod, *(p2 - 1))) != 0)
		m = *(p2 - 1);
	else
		m = 0;

	while(**p != '\0' && **p != c && (!m || **p != m) && !isdigit(**p) &&
	    **p != '.')
		SCOPY(p, buf);

	pre1 = -1;
	pre2 = -1;

	if(isdigit(**p) || **p == '.') {
		if(**p == '0')
			SASGN('0', buf);

		if(sscanf(*p, "%d.%d", &pre1, &pre2) != 2)
			if(sscanf(*p, "%d", &pre1) != 1)
				sscanf(*p, ".%d", &pre2);
	}

	if(pre1 >= CDDBBUFSIZ)
		pre1 = CDDBBUFSIZ - 1;

	if(pre2 >= CDDBBUFSIZ || (pre2 == -1 && c == 's'))
		pre2 = CDDBBUFSIZ - 1;

	if(pre1 >= 0) {
		sprintf(cbuf, "%d", pre1);
		strncpy(&buf[cc], cbuf, (CDDBBUFSIZ - cc - 1));
		buf[CDDBBUFSIZ - 1] = '\0';
		cc = strlen(buf);
	}

	if(pre2 >= 0) {
		sprintf(cbuf, ".%d", pre2);
		strncpy(&buf[cc], cbuf, (CDDBBUFSIZ - cc - 1));
		buf[CDDBBUFSIZ - 1] = '\0';
		cc = strlen(buf);
	}

	*p = p2 + 1;

	if(m != 0)
		SASGN(m, buf);

	SASGN(c, buf);
	SPUT('\0', buf);

	return 1;
}


void
cddbd_snprintf(char *buf, int size, char *fmt, void *a0, void *a1, void *a2,
    void *a3, void *a4, void *a5, void *a6, void *a7, void *a8, void *a9)
{
	int x;
	int ano;
	int cnt;
	void *arg[10];
	char *p;
	char cbuf[CDDBBUFSIZ];
	char tbuf[CDDBBUFSIZ];

	arg[0] = a0;
	arg[1] = a1;
	arg[2] = a2;
	arg[3] = a3;
	arg[4] = a4;
	arg[5] = a5;
	arg[6] = a6;
	arg[7] = a7;
	arg[8] = a8;
	arg[9] = a9;

	p = fmt;
	cnt = 0;
	ano = 0;

	while(cnt < size) {
		switch(*p) {
		case '%':
			x = get_conv(&p, cbuf);

			if((ano + x) >= 10) {
				if(!logging) {
					cddbd_log(LOG_ERR,
					    "Too many args to snprintf, "
					    "format: \"%s\"", fmt);
				}

				buf[cnt] = '\0';
				return;
			}

			sprintf(tbuf, cbuf, arg[ano]);
			strncpy(&buf[cnt], tbuf, (size - cnt - 1));
			buf[size - 1] = '\0';
			cnt = strlen(buf);

			ano += x;

			break;

		case '\0':
			buf[cnt] = '\0';
			return;

		default:
			buf[cnt] = *p;
			cnt++;
			p++;

			break;
		}
	}

	buf[size - 1] = '\0';

	if(!logging)
		cddbd_log(LOG_ERR, "Buf overflow in snprintf: \"%s\"", buf);
}


/* This should be the last function in the file, to avoid compile errors. */
void
cddbd_log(unsigned int flags, char *fmt, void *a1, void *a2, void *a3,
    void *a4, void *a5, void *a6)
{
	FILE *fp;
	char buf[CDDBBUFSIZ];

	if((!(log_flags & (short)flags) && !debug) || quiet)
		return;

	/* Detect recursion. */
	logging++;

	/* Lock the log file. */
	if(!debug && logging == 1)
		(void)cddbd_lock(lock_log, 1);

	/* We can't open the log file. Continue anyway. */
	if(debug || (fp = fopen(logfile, "a")) == NULL) {
		if(mode == MODE_DB)
			fp = stdout;
		else
			fp = stderr;

		if(!debug) {
			fprintf(fp,
			    "%s [%05d,%04X] Warning: can't open log file: "
			    "%s (%d)\n",
			    get_time(0), curpid, LOG_ERR, logfile, errno);
		}
	}
	else if(logging == 1)
		(void)cddbd_fix_file(logfile, file_mode, uid, gid);

	/* Log it. */
	if(!debug)
		fprintf(fp, "%s [%05d,%04X] ", get_time(0), curpid, flags);

	/* Copy fmt to a temp buffer so we can work on it. */
	strncpy(buf, fmt, sizeof(buf));
	buf[sizeof(buf) - 1] = '\0';
	strip_crlf(buf);

	/* Print the log message. */
	fprintf(fp, buf, a1, a2, a3, a4, a5, a6);
	fputc('\n', fp);

	if(fp != stderr && fp != stdout) {
		fclose(fp);
		if(logging == 1)
			trunc_log();
	}

	cddbd_unlock(lock_log);

	logging--;

	return;
}
