/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*-

   eel-background-style.c: Style subclass used by EelBackground.
 
   Copyright (C) 2001 Bent Spoon Software
  
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
  
   You should have received a copy of the GNU Library General Public
   License along with this program; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
  
   Author: Darin Adler <darin@bentspoon.com>
*/

#include <config.h>
#include "eel-background-style.h"

#include "eel-gdk-extensions.h"
#include <gtk/gtktreeview.h>
#include <libgnome/gnome-macros.h>

GNOME_CLASS_BOILERPLATE (EelBackgroundStyle, eel_background_style,
			 GtkStyle, GTK_TYPE_STYLE)

struct EelBackgroundStyleDetails {
	EelBackground *background;
};

static void
eel_background_style_instance_init (EelBackgroundStyle *style)
{
	style->details = g_new0 (EelBackgroundStyleDetails, 1);
}

static void
eel_background_style_finalize (GObject *object)
{
	EelBackgroundStyle *style;

	style = EEL_BACKGROUND_STYLE (object);

	if (style->details->background) {
		g_object_unref (style->details->background);
	}
	
	g_free (style->details);

	GNOME_CALL_PARENT (G_OBJECT_CLASS, finalize, (object));
}

static void
eel_background_style_draw_flat_box (GtkStyle *style,
				    GdkWindow *window,
				    GtkStateType state_type,
				    GtkShadowType shadow_type,
				    GdkRectangle *area,
				    GtkWidget *widget,
				    const char *detail,
				    int x,
				    int y,
				    int width,
				    int height)
{
	gboolean call_parent;
	EelBackground *background;
	GdkGC *gc;
	GdkGCValues gc_values;
	EelBackgroundStyleDetails *details;
	GdkGCValuesMask value_mask;
	GdkColor color;
	GdkPixmap *pixmap;
	gboolean changes_with_size;
	int window_width, window_height;
	
	call_parent = TRUE;

	background = NULL;
	if (state_type == GTK_STATE_NORMAL) {
		background = eel_get_widget_background (widget);
		if (background != NULL && !GTK_IS_TREE_VIEW (widget)) {
			call_parent = FALSE;
		}
	}

	if (call_parent) {
		GNOME_CALL_PARENT (GTK_STYLE_CLASS, draw_flat_box,
				   (style, window, state_type, shadow_type, area, widget,
				    detail, x, y, width, height));
		return;
	}


	details = EEL_BACKGROUND_STYLE (style)->details;

	gdk_drawable_get_size (window, &window_width, &window_height);
	
	pixmap = eel_background_get_pixmap_and_color (background, window,
						      window_width, window_height,
						      &color,
						      &changes_with_size);

	if (!changes_with_size) {
		/* The background was already drawn by X, since we set
		 * the GdkWindow background/back_pixmap.
		 * No need to draw it again. */
		return;
	}
	
	if (pixmap != NULL) {
		gc_values.tile = pixmap;
		gc_values.ts_x_origin = 0;
		gc_values.ts_y_origin = 0;
		gc_values.fill = GDK_TILED;
		value_mask = GDK_GC_FILL | GDK_GC_TILE |
			GDK_GC_TS_X_ORIGIN | GDK_GC_TS_Y_ORIGIN;
	} else {
		gdk_rgb_find_color (gtk_widget_get_colormap (widget), &color);
		gc_values.foreground = color;
		gc_values.fill = GDK_SOLID;
		value_mask = GDK_GC_FILL | GDK_GC_FOREGROUND;
	}
		
    	gc = gdk_gc_new_with_values (window, &gc_values, value_mask);
	
	if (area != NULL) {
		gdk_gc_set_clip_rectangle (gc, area);
	}


	if (width == -1) {
		width = window_width;
	}
	
	if (height == -1) {
		height = window_height;
	}

	gdk_draw_rectangle (window, gc, TRUE,
			    x, y,
			    width, height);

	g_object_unref (gc);
	
	if (pixmap != NULL) {
		g_object_unref (pixmap);
	}
}

EelBackgroundStyle *
eel_background_style_new (GtkStyle *base_style,
			  EelBackground *background)
{
	EelBackgroundStyle *style;
	
	g_return_val_if_fail (GTK_IS_STYLE (base_style), NULL);
	g_return_val_if_fail (EEL_IS_BACKGROUND (background), NULL);
	
	style = g_object_new (EEL_TYPE_BACKGROUND_STYLE, NULL);

	GTK_STYLE_CLASS (g_type_class_peek (GTK_TYPE_STYLE))->copy (GTK_STYLE (style), base_style);

	style->details->background = g_object_ref (background);
	
	return style;
}

static void
eel_background_style_set_background (GtkStyle    *gtkstyle,
				     GdkWindow   *window,
				     GtkStateType state_type)
{
	EelBackgroundStyle *style;
	EelBackground *background;
	GdkColor color;
	GdkPixmap *pixmap;
	gboolean changes_with_size;
	int window_width, window_height;
	
	style = EEL_BACKGROUND_STYLE (gtkstyle);

	background = style->details->background;

	g_return_if_fail (background != NULL);

	gdk_drawable_get_size (window, &window_width, &window_height);
	pixmap = eel_background_get_pixmap_and_color (background, window,
						      window_width, window_height,
						      &color,
						      &changes_with_size);
	gdk_rgb_find_color (gtkstyle->colormap, &color);

	if (pixmap != NULL && !changes_with_size) {
		gdk_window_set_back_pixmap (window, pixmap, FALSE);
	} else {
		gdk_window_set_background (window, &color);
	}

	if (pixmap != NULL) {
		g_object_unref (pixmap);
	}
}

static GtkStyle *
eel_background_style_clone (GtkStyle *style)
{
	EelBackgroundStyle *new_style;

	new_style = g_object_new (G_OBJECT_TYPE (style), NULL);
	if (EEL_BACKGROUND_STYLE (style)->details->background != NULL) {
		new_style->details->background = g_object_ref (EEL_BACKGROUND_STYLE (style)->details->background);
	}
	return GTK_STYLE (new_style);
}


static void
eel_background_style_class_init (EelBackgroundStyleClass *class)
{
	G_OBJECT_CLASS (class)->finalize = eel_background_style_finalize;
	GTK_STYLE_CLASS (class)->clone = eel_background_style_clone;
	GTK_STYLE_CLASS (class)->draw_flat_box = eel_background_style_draw_flat_box;
	GTK_STYLE_CLASS (class)->set_background = eel_background_style_set_background;
}
