/*
 * Decompiled with CFR 0.152.
 */
package org.openhab.core.storage.json.internal;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonIOException;
import com.google.gson.JsonSyntaxException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.eclipse.jdt.annotation.NonNullByDefault;
import org.eclipse.jdt.annotation.Nullable;
import org.openhab.core.common.ThreadPoolManager;
import org.openhab.core.config.core.Configuration;
import org.openhab.core.config.core.ConfigurationDeserializer;
import org.openhab.core.config.core.OrderingMapSerializer;
import org.openhab.core.config.core.OrderingSetSerializer;
import org.openhab.core.storage.Storage;
import org.openhab.core.storage.json.internal.InstantTypeAdapter;
import org.openhab.core.storage.json.internal.StorageEntry;
import org.openhab.core.storage.json.internal.StorageEntryMapDeserializer;
import org.openhab.core.storage.json.internal.migration.TypeMigrationException;
import org.openhab.core.storage.json.internal.migration.TypeMigrator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NonNullByDefault
public class JsonStorage<T>
implements Storage<T> {
    private final Logger logger = LoggerFactory.getLogger(JsonStorage.class);
    private final int maxBackupFiles;
    private final int writeDelay;
    private final int maxDeferredPeriod;
    static final String CLASS = "class";
    static final String VALUE = "value";
    private static final String BACKUP_EXTENSION = "backup";
    private static final String SEPARATOR = "--";
    private final ScheduledExecutorService scheduledExecutorService;
    private @Nullable ScheduledFuture<?> commitScheduledFuture;
    private long deferredSince = 0L;
    private final File file;
    private final @Nullable ClassLoader classLoader;
    private final Map<String, StorageEntry> map = new ConcurrentHashMap<String, StorageEntry>();
    private final Map<String, TypeMigrator> typeMigrators;
    private final transient Gson internalMapper;
    private final transient Gson entityMapper;
    private boolean dirty;

    public JsonStorage(File file, @Nullable ClassLoader classLoader, int maxBackupFiles, int writeDelay, int maxDeferredPeriod, List<TypeMigrator> typeMigrators) {
        this.file = file;
        this.classLoader = classLoader;
        this.maxBackupFiles = maxBackupFiles;
        this.writeDelay = writeDelay;
        this.maxDeferredPeriod = maxDeferredPeriod;
        this.typeMigrators = typeMigrators.stream().collect(Collectors.toMap(TypeMigrator::getOldType, e -> e));
        this.internalMapper = new GsonBuilder().setDateFormat("MMM d, yyyy, h:mm:ss aaa").registerTypeHierarchyAdapter(Map.class, (Object)new OrderingMapSerializer()).registerTypeHierarchyAdapter(Set.class, (Object)new OrderingSetSerializer()).registerTypeHierarchyAdapter(Map.class, (Object)new StorageEntryMapDeserializer()).setPrettyPrinting().create();
        this.entityMapper = new GsonBuilder().setDateFormat("MMM d, yyyy, h:mm:ss aaa").registerTypeHierarchyAdapter(Map.class, (Object)new OrderingMapSerializer()).registerTypeHierarchyAdapter(Set.class, (Object)new OrderingSetSerializer()).registerTypeAdapter(Configuration.class, (Object)new ConfigurationDeserializer()).registerTypeAdapter(Instant.class, (Object)new InstantTypeAdapter()).setPrettyPrinting().create();
        this.scheduledExecutorService = ThreadPoolManager.getScheduledPool((String)"JsonStorage");
        Map<String, StorageEntry> inputMap = null;
        if (file.exists()) {
            inputMap = this.readDatabase(file);
        }
        if (inputMap == null) {
            if (file.exists()) {
                this.logger.info("Json storage file at '{}' seems to be corrupt - checking for a backup.", (Object)file.getAbsolutePath());
            } else {
                this.logger.debug("Json storage file at '{}' does not exist - checking for a backup.", (Object)file.getAbsolutePath());
            }
            int cnt = 1;
            while (cnt <= maxBackupFiles) {
                File backupFile = this.getBackupFile(cnt);
                if (backupFile == null) break;
                inputMap = this.readDatabase(backupFile);
                if (inputMap != null) {
                    this.logger.info("Json storage file at '{}' is used (backup {}).", (Object)backupFile.getAbsolutePath(), (Object)cnt);
                    break;
                }
                ++cnt;
            }
        }
        if (inputMap != null) {
            this.map.putAll(inputMap);
            this.logger.debug("Opened Json storage file at '{}'.", (Object)file.getAbsolutePath());
        }
    }

    public @Nullable T put(String key, @Nullable T value) {
        if (value == null) {
            return this.remove(key);
        }
        StorageEntry val = new StorageEntry(value.getClass().getName(), this.entityMapper.toJsonTree(value));
        StorageEntry previousValue = this.map.put(key, val);
        this.deferredCommit();
        if (previousValue == null) {
            return null;
        }
        return this.deserialize(previousValue, null);
    }

    public @Nullable T remove(String key) {
        StorageEntry removedElement = this.map.remove(key);
        this.deferredCommit();
        if (removedElement == null) {
            return null;
        }
        return this.deserialize(removedElement, null);
    }

    public boolean containsKey(String key) {
        return this.map.containsKey(key);
    }

    public @Nullable T get(String key) {
        StorageEntry value = this.map.get(key);
        if (value == null) {
            return null;
        }
        return this.deserialize(value, key);
    }

    public Collection<String> getKeys() {
        return this.map.keySet();
    }

    public Collection<@Nullable T> getValues() {
        ArrayList<@Nullable T> values = new ArrayList<T>();
        for (String key : this.getKeys()) {
            values.add(this.get(key));
        }
        return values;
    }

    private @Nullable T deserialize(@Nullable StorageEntry entry, @Nullable String key) {
        if (entry == null) {
            return null;
        }
        try {
            String entityClassName = entry.getEntityClassName();
            JsonElement entityValue = (JsonElement)entry.getValue();
            TypeMigrator migrator = this.typeMigrators.get(entityClassName);
            if (migrator != null) {
                entityClassName = migrator.getNewType();
                entityValue = migrator.migrate(entityValue);
                if (key != null) {
                    this.map.put(key, new StorageEntry(entityClassName, entityValue));
                    this.deferredCommit();
                }
            }
            Class<?> loadedValueType = this.classLoader != null ? this.classLoader.loadClass(entityClassName) : Class.forName(entityClassName);
            Object value = this.entityMapper.fromJson(entityValue, loadedValueType);
            this.logger.trace("deserialized value '{}' from Json", value);
            return (T)value;
        }
        catch (JsonIOException | JsonSyntaxException | ClassNotFoundException e) {
            this.logger.error("Couldn't deserialize value '{}'. Root cause is: {}", (Object)entry, (Object)e.getMessage());
            return null;
        }
        catch (TypeMigrationException e) {
            this.logger.error("Type '{}' needs migration but migration failed: '{}'", (Object)entry.getEntityClassName(), (Object)e.getMessage());
            return null;
        }
    }

    private @Nullable Map<String, StorageEntry> readDatabase(File inputFile) {
        if (inputFile.length() == 0L) {
            this.logger.warn("Json storage file at '{}' is empty - ignoring corrupt file.", (Object)inputFile.getAbsolutePath());
            return null;
        }
        try {
            ConcurrentHashMap<String, StorageEntry> inputMap = new ConcurrentHashMap<String, StorageEntry>();
            FileReader reader = new FileReader(inputFile);
            Map loadedMap = (Map)this.internalMapper.fromJson((Reader)reader, this.map.getClass());
            if (loadedMap != null && !loadedMap.isEmpty()) {
                inputMap.putAll(loadedMap);
            }
            return inputMap;
        }
        catch (JsonIOException | JsonSyntaxException | FileNotFoundException e) {
            this.logger.error("Error reading JsonDB from {}. Cause {}.", (Object)inputFile.getPath(), (Object)e.getMessage());
            return null;
        }
    }

    private @Nullable File getBackupFile(int age) {
        List<Long> fileTimes = this.calculateFileTimes();
        if (fileTimes.size() < age) {
            return null;
        }
        return new File(this.file.getParent() + File.separator + BACKUP_EXTENSION, String.valueOf(fileTimes.get(fileTimes.size() - age)) + SEPARATOR + this.file.getName());
    }

    private List<Long> calculateFileTimes() {
        File folder = new File(this.file.getParent() + File.separator + BACKUP_EXTENSION);
        if (!folder.isDirectory()) {
            return List.of();
        }
        ArrayList<Long> fileTimes = new ArrayList<Long>();
        File[] files = folder.listFiles();
        if (files != null) {
            File[] fileArray = files;
            int n = files.length;
            int n2 = 0;
            while (n2 < n) {
                String[] parts;
                File value = fileArray[n2];
                if (value.isFile() && (parts = value.getName().split(SEPARATOR)).length == 2 && parts[1].equals(this.file.getName())) {
                    long time = Long.parseLong(parts[0]);
                    fileTimes.add(time);
                }
                ++n2;
            }
        }
        Collections.sort(fileTimes);
        return fileTimes;
    }

    private void writeDatabaseFile(File dataFile, String data) throws IOException {
        try {
            Throwable throwable = null;
            Object var4_6 = null;
            try (FileOutputStream outputStream = new FileOutputStream(dataFile, false);){
                outputStream.write(data.getBytes());
                outputStream.flush();
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (IOException e) {
            throw new IOException(String.format("Error writing JsonDB to %s. Cause %s.", dataFile.getPath(), e.getMessage()), e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void flush() {
        ScheduledFuture<?> commitScheduledFuture = this.commitScheduledFuture;
        if (commitScheduledFuture != null) {
            commitScheduledFuture.cancel(false);
            this.commitScheduledFuture = null;
        }
        if (this.dirty) {
            String json = this.internalMapper.toJson(this.map);
            Map<String, StorageEntry> map = this.map;
            synchronized (map) {
                try {
                    this.writeDatabaseFile(this.file, json);
                    this.writeDatabaseFile(new File(this.file.getParent() + File.separator + BACKUP_EXTENSION, System.currentTimeMillis() + SEPARATOR + this.file.getName()), json);
                    this.cleanupBackups();
                    this.dirty = false;
                }
                catch (IOException e) {
                    this.logger.error("{}", (Object)e.getMessage());
                }
                this.deferredSince = 0L;
            }
        }
    }

    private void cleanupBackups() {
        List<Long> fileTimes = this.calculateFileTimes();
        if (fileTimes.size() > this.maxBackupFiles) {
            int counter = 0;
            while (counter < fileTimes.size() - this.maxBackupFiles) {
                File deleter = new File(this.file.getParent() + File.separator + BACKUP_EXTENSION, String.valueOf(fileTimes.get(counter)) + SEPARATOR + this.file.getName());
                deleter.delete();
                ++counter;
            }
        }
    }

    public synchronized void deferredCommit() {
        this.dirty = true;
        ScheduledFuture<?> commitScheduledFuture = this.commitScheduledFuture;
        if (commitScheduledFuture != null) {
            commitScheduledFuture.cancel(false);
            this.commitScheduledFuture = null;
        }
        if (this.deferredSince != 0L && this.deferredSince < System.currentTimeMillis() - (long)this.maxDeferredPeriod) {
            this.flush();
            return;
        }
        if (this.deferredSince == 0L) {
            this.deferredSince = System.currentTimeMillis();
        }
        this.commitScheduledFuture = this.scheduledExecutorService.schedule(this::flush, (long)this.writeDelay, TimeUnit.MILLISECONDS);
    }
}

