/*
   Project: PcapKit

   Copyright (C) 2013 Sebastian Reitenbach

   Author: Sebastian Reitenbach

   Created: 2013-06-23 09:47:49 +0200 by sebastia

   This application is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This application is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/

#ifndef _PCAPKIT_H_
#define _PCAPKIT_H_

#import <Foundation/Foundation.h>
#include <pcap.h>
#include <sys/types.h>
#include <sys/socket.h>

@protocol PcapKitDelegate
-(void) handlePacket: (NSData *) _packet;
@end

@interface PcapKit : NSObject
{
  // general variables to configure the sniffing
  NSString *device;                   // device to use in live capture mode
  NSString *inputFile;                // read sniffed packets from this file
  NSString *outputFile;               // file where to save sniffed packets
  NSInteger readTimeout;              // in ms, 0 sniffs until error, -1 forever
  char errBuf[PCAP_ERRBUF_SIZE];      // the buffer where pcap returns errors
  NSInteger snaplen;                  // sniff packets up to this size
  BOOL promisc;			      // put device to promisc mode or not
  pcap_t *handle;                     // the handle of the sniffing session
  NSInteger bufsize;		      // buffer size used on capture handle
  // filter configuration
  struct bpf_program filterProgram;   // where the compiled filter is stored
  NSString *bpfFilter;                // the filter string to apply to the
				      // sniffing session
  BOOL optimizeFilter;                // whether to optimize the filter or not
  bpf_u_int32 filterNetmask;          // the netmask of the network the filter
				      // applies to
  bpf_u_int32 IP;                     // IP of the sniffing device 
  BOOL pcapDebug;		      // whether we should print debug output
  pcap_direction_t direction;	      // limit the direction that packets must
				      // be flowing in order to get captured

  id delegate;			      // the delegate, that processes sniffed
				      // packets
}

// class methods
// convert a data link type to a name
+(NSString *) getDataLinkNameForType:(NSInteger) _type;
// convert a data name to a datalink type
+(NSInteger) getDataLinkTypeForName:(NSString *) _name;
// convert a data link type to a description
+(NSString *) getDataLinkDescriptionForType:(NSInteger)_type;
// converts a PCAP_ERROR_ or PCAP_WARNING_ to an error string
+(NSString *) getErrorStringForStatus:(NSInteger *)_status;
// converts an error to a string
+(NSString *) getErrorStringForType:(NSInteger)_type;
// returns the pcap library version number
+(NSString *) getPcapLibraryVersion;

// needed setters and getters
- (void) setDevice: (NSString *) _dev;
- (NSString *) device;
- (void) setInputFile: (NSString *) _if;
- (NSString *) inputFile;
- (void) setOutputFile: (NSString *) _of;
- (NSString *) outputFile;
- (void) setReadTimeout:(NSInteger) _rt;
- (NSInteger) readTimeout;
- (void) setBpfFilter:(NSString *) _bpf;
- (NSString *)bpfFilter;
- (void) setOptimizeFilter:(BOOL) _optimize;
- (BOOL) optimizeFilter;
- (void) setPromisc: (BOOL) _promisc;
- (BOOL) promisc;
- (void) setDelegate: (id) _delegate;
- (id) delegate;
- (void) setDirection: (pcap_direction_t) _direction;
- (pcap_direction_t) direction;
- (void) setBufsize:(NSInteger)_bufsize;
- (NSInteger) bufsize;
- (pcap_t*) handle;


// open a physical device
-(void) openLiveStream;
// open a file to read packets from
-(void) openOfflineStream;
// create a handle to be used for sniffing, try device first, then inputFile
-(BOOL) createSniffingHandle;
// activate a created handle before sniffing starts
-(NSInteger) activateSniffingHandle;
// find a suitable device
-(BOOL)lookupDevice; 
// lookup the netmask on the sniffing devie
-(NSInteger)lookupNetmask;
// read from a saved pcap stream
-(void) openOfflineStream;
// apply the filter for the sniffing session
-(BOOL) applyFilter;
// start the loop
-(NSInteger) startPcapLoop;
// stop the loop
-(void) stopPcapLoop;
// get data list of data link types for the handle
-(NSInteger) getDatalinksIntoArray:(int **)_dlts;
// get a potential error from the pcap handle
-(NSString *)getErrorString;
// print an error with given prefix
-(void) printErrorWithPrefix:(NSString *)_prefix;


// figure out the data link of current sniffer
-(NSInteger) dataLink;
// set data link if multiple are possible on the interface
- (void) setDataLink: (NSInteger)_dlt;
// get the snapshot lengths from the handle
-(NSInteger) getSnaplen;
// set the snapshot length to be used by the handle
-(BOOL) setSnaplen:(NSInteger)_snaplen;
// check if current savefile uses different byteorder than current system
-(BOOL)isSavefileSwapped;
// check if it is possible to put the interface into RF Monitoring mode
-(BOOL)checkSetRFMon;
// set the interface into RF Monitoring mode
-(BOOL)enableRFMon:(BOOL)_flag;
// set the interface into promiscous mode
-(BOOL)enablePromiscuousMode:(BOOL)_flag;
// get the major version of pcap used to write the savefile 
-(NSInteger) getMajorVersion;
// get the minor version of pcap used to write the savefile 
-(NSInteger) getMinorVersion;


@end

#endif // _PCAPKIT_H_

