/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Christian Blättler
 */

import { HttpStatusCode, TalerMerchantApi } from "@gnu-taler/taler-util";
import {
  ButtonBetterBulma,
  LocalNotificationBannerBulma,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { h } from "preact";
import { useState } from "preact/hooks";
import {
  FormErrors,
  FormProvider,
} from "../../../../components/form/FormProvider.js";
import { Input } from "../../../../components/form/Input.js";
import { InputDate } from "../../../../components/form/InputDate.js";
import { InputDuration } from "../../../../components/form/InputDuration.js";
import { useSessionContext } from "../../../../context/session.js";
import { undefinedIfEmpty } from "../../../../utils/table.js";

const TALER_SCREEN_ID = 70;

type Entity = TalerMerchantApi.TokenFamilyUpdateRequest;

interface Props {
  onUpdated: () => void;
  onBack?: () => void;
  tokenFamily: TalerMerchantApi.TokenFamilyDetails;
}

export function UpdatePage({ onUpdated, onBack, tokenFamily }: Props) {
  const [value, valueHandler] = useState<Partial<Entity>>(tokenFamily);
  const { i18n } = useTranslationContext();
  const errors = undefinedIfEmpty<FormErrors<Entity>>({
    name: !value.name ? i18n.str`Required` : undefined,
    description: !value.description ? i18n.str`Required` : undefined,
    valid_after: !value.valid_after ? i18n.str`Required` : undefined,
    valid_before: !value.valid_before ? i18n.str`Required` : undefined,
    duration: !value.duration ? i18n.str`Required` : undefined,
  });
  const { state: session, lib } = useSessionContext();
  const [notification, safeFunctionHandler] = useLocalNotificationBetter();

  const hasErrors = errors !== undefined;

  const update = safeFunctionHandler(
    lib.instance.updateTokenFamily.bind(lib.instance),
    !session.token || !!errors
      ? undefined
      : [session.token, tokenFamily.slug, value as Entity],
  );
  update.onSuccess = onUpdated;
  update.onFail = (fail) => {
    switch (fail.case) {
      case HttpStatusCode.Unauthorized:
        return i18n.str`Unauthorized`;
      case HttpStatusCode.NotFound:
        return i18n.str`Not found`;
    }
  };

  return (
    <div>
      <LocalNotificationBannerBulma notification={notification} />
      <section class="section">
        <section class="hero is-hero-bar">
          <div class="hero-body">
            <div class="level">
              <div class="level-left">
                <div class="level-item">
                  <span class="is-size-4">
                    <i18n.Translate>
                      Token Family: <b>{tokenFamily.name}</b>
                    </i18n.Translate>
                  </span>
                </div>
              </div>
            </div>
          </div>
        </section>
        <hr />

        <section class="section is-main-section">
          <div class="columns">
            <div class="column is-four-fifths">
              <FormProvider<Entity>
                name="token_family"
                errors={errors}
                object={value}
                valueHandler={valueHandler}
              >
                <Input<Entity>
                  name="name"
                  inputType="text"
                  label={i18n.str`Name`}
                  tooltip={i18n.str`User-readable token family name`}
                />
                <Input<Entity>
                  name="description"
                  inputType="multiline"
                  label={i18n.str`Description`}
                  tooltip={i18n.str`Token family description for customers`}
                />
                <InputDate<Entity>
                  name="valid_after"
                  label={i18n.str`Valid After`}
                  tooltip={i18n.str`Token family can issue tokens after this date`}
                  withTimestampSupport
                />
                <InputDate<Entity>
                  name="valid_before"
                  label={i18n.str`Valid Before`}
                  tooltip={i18n.str`Token family can issue tokens until this date`}
                  withTimestampSupport
                />
                <InputDuration<Entity>
                  name="duration"
                  label={i18n.str`Duration`}
                  tooltip={i18n.str`Validity duration of a issued token`}
                  withForever
                  useProtocolDuration
                />
              </FormProvider>

              <div class="buttons is-right mt-5">
                {onBack && (
                  <button type="button"class="button" onClick={onBack}>
                    <i18n.Translate>Cancel</i18n.Translate>
                  </button>
                )}
                <ButtonBetterBulma
                  data-tooltip={
                    hasErrors
                      ? i18n.str`Please complete the marked fields`
                      : i18n.str`Confirm operation`
                  }
                  type="submit"
                  onClick={update}
                >
                  <i18n.Translate>Confirm</i18n.Translate>
                </ButtonBetterBulma>
              </div>
            </div>
          </div>
        </section>
      </section>
    </div>
  );
}
