/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { useSessionContext } from "../../context/session.js";
import { UIElement } from "../../hooks/preference.js";
import { Entity } from "../../paths/admin/create/CreatePage.js";
import { Input } from "../form/Input.js";
import { InputDuration } from "../form/InputDuration.js";
import { InputGroup } from "../form/InputGroup.js";
import { InputImage } from "../form/InputImage.js";
import { InputLocation } from "../form/InputLocation.js";
import { InputToggle } from "../form/InputToggle.js";
import { InputWithAddon } from "../form/InputWithAddon.js";
import { FragmentPersonaFlag } from "../menu/SideBar.js";
import { RoundingInterval } from "@gnu-taler/taler-util";
import { InputSelector } from "../form/InputSelector.js";

const TALER_SCREEN_ID = 16;

export function DefaultInstanceFormFields({
  readonlyId,
  showId,
}: {
  readonlyId?: boolean;
  showId: boolean;
}): VNode {
  const { i18n } = useTranslationContext();
  const { state } = useSessionContext();
  return (
    <Fragment>
      {showId && (
        <InputWithAddon<Entity>
          name="id"
          addonBefore={new URL("instances/", state.backendUrl.href).href}
          readonly={readonlyId}
          label={i18n.str`Username`}
          tooltip={i18n.str`Name of the instance in URLs. The 'admin' instance is special in that it is used to administer other instances.`}
        />
      )}

      <Input<Entity>
        name="name"
        label={i18n.str`Business name`}
        tooltip={i18n.str`Legal name of the business represented by this instance.`}
      />

      <Input<Entity>
        name="email"
        label={i18n.str`Email`}
        tooltip={i18n.str`Contact email`}
      />

      <Input<Entity>
        name="phone_number"
        label={i18n.str`Phone number`}
        tooltip={i18n.str`Contact phone`}
      />
      <FragmentPersonaFlag point={UIElement.option_advanceInstanceSettings}>
        <Input<Entity>
          name="website"
          label={i18n.str`Website URL`}
          tooltip={i18n.str`URL.`}
        />

        <InputImage<Entity>
          name="logo"
          label={i18n.str`Logo`}
          tooltip={i18n.str`Logo image.`}
        />

        <InputGroup
          name="address"
          label={i18n.str`Address`}
          tooltip={i18n.str`Physical location of the merchant.`}
        >
          <InputLocation name="address" />
        </InputGroup>

        <InputGroup
          name="jurisdiction"
          label={i18n.str`Jurisdiction`}
          tooltip={i18n.str`Jurisdiction for legal disputes with the merchant.`}
        >
          <InputLocation name="jurisdiction" />
        </InputGroup>

        <InputToggle<Entity>
          name="use_stefan"
          label={i18n.str`Pay transaction fee`}
          tooltip={i18n.str`Cover the transaction cost or pass it on to the user.`}
        />

        <InputDuration<Entity>
          name="default_pay_delay"
          label={i18n.str`Default payment delay`}
          tooltip={i18n.str`Time customers have to pay an order before the offer expires by default.`}
        />

        <InputDuration<Entity>
          name="default_refund_delay"
          label={i18n.str`Default refund delay`}
          tooltip={i18n.str`Time merchants have to refund an order.`}
        />

        <InputDuration<Entity>
          name="default_wire_transfer_delay"
          label={i18n.str`Default wire transfer delay`}
          tooltip={i18n.str`Maximum time an exchange is allowed to delay wiring funds to the merchant, enabling it to aggregate smaller payments into larger wire transfers and reducing wire fees.`}
        />

        <InputSelector<Entity>
          name="default_wire_transfer_rounding_interval"
          label={i18n.str`Default wire transfer rounding interval`}
          tooltip={i18n.str`Interval to which wire deadlines should be rounded up to.`}
          help={i18n.str`As an example if you set the interval to day the wire transfer deadline will be rounded to midnight.`}
          values={[
            RoundingInterval.DAY,
            RoundingInterval.WEEK,
            RoundingInterval.SECOND,
            RoundingInterval.HOUR,
            RoundingInterval.MINUTE,
            RoundingInterval.MONTH,
            RoundingInterval.QUARTER,
            RoundingInterval.YEAR,
            RoundingInterval.NONE,
          ]}
          toStr={(v: RoundingInterval) => {
            switch (v) {
              case RoundingInterval.NONE:
                return i18n.str`No rounding`;
              case RoundingInterval.SECOND:
                return i18n.str`To second`;
              case RoundingInterval.MINUTE:
                return i18n.str`To minute`;
              case RoundingInterval.HOUR:
                return i18n.str`To hour`;
              case RoundingInterval.DAY:
                return i18n.str`To day`;
              case RoundingInterval.WEEK:
                return i18n.str`To week`;
              case RoundingInterval.MONTH:
                return i18n.str`To month`;
              case RoundingInterval.QUARTER:
                return i18n.str`To quarter`;
              case RoundingInterval.YEAR:
                return i18n.str`To year`;
            }
          }}
        />
      </FragmentPersonaFlag>
    </Fragment>
  );
}
