/*
   This file is part of TALER
   Copyright (C) 2025 Taler Systems SA

   TALER is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 3, or (at your option) any later version.

   TALER is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along with
   TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
/**
 * @file backenddb/pg_update_contract_session.c
 * @brief Implementation of the update_contract_session function for Postgres
 * @author Christian Grothoff
 */
#include "platform.h"
#include <taler/taler_error_codes.h>
#include <taler/taler_dbevents.h>
#include <taler/taler_pq_lib.h>
#include "pg_update_contract_session.h"
#include "pg_helper.h"


enum GNUNET_DB_QueryStatus
TMH_PG_update_contract_session (
  void *cls,
  const char *instance_id,
  const struct TALER_PrivateContractHashP *h_contract_terms,
  const char *session_id,
  char **fulfillment_url,
  bool *refunded)
{
  struct PostgresClosure *pg = cls;
  struct GNUNET_PQ_QueryParam params[] = {
    GNUNET_PQ_query_param_string (instance_id),
    GNUNET_PQ_query_param_auto_from_type (h_contract_terms),
    GNUNET_PQ_query_param_string (session_id),
    GNUNET_PQ_query_param_end
  };
  struct GNUNET_PQ_ResultSpec rs[] = {
    GNUNET_PQ_result_spec_allow_null (
      GNUNET_PQ_result_spec_string ("fulfillment_url",
                                    fulfillment_url),
      NULL),
    GNUNET_PQ_result_spec_bool ("refunded",
                                refunded),
    GNUNET_PQ_result_spec_end
  };

  /* Session ID must always be given by the caller. */
  GNUNET_assert (NULL != session_id);

  /* no preflight check here, run in transaction by caller! */
  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Updating session for h_contract_terms '%s' of `%s' to `%s'\n",
              GNUNET_h2s (&h_contract_terms->hash),
              instance_id,
              session_id);
  PREPARE (pg,
           "update_contract_session",
           "UPDATE merchant_contract_terms mct SET"
           " session_id=$3"
           " WHERE h_contract_terms=$2"
           "   AND merchant_serial="
           "     (SELECT merchant_serial"
           "        FROM merchant_instances"
           "       WHERE merchant_id=$1)"
           " RETURNING"
           "  contract_terms->>'fulfillment_url' AS fulfillment_url"
           " ,EXISTS (SELECT 1"
           "            FROM merchant_refunds mr"
           "           WHERE order_serial=mct.order_serial"
           " ) AS refunded");
  *fulfillment_url = NULL;
  return GNUNET_PQ_eval_prepared_singleton_select (pg->conn,
                                                   "update_contract_session",
                                                   params,
                                                   rs);
}
