package Iterator::Flex::Freeze;

# ABSTRACT:  Freeze an iterator after every next

use v5.28;
use strict;
use warnings;
use experimental 'signatures';

our $VERSION = '0.32';

use Iterator::Flex::Factory 'to_iterator';
use Iterator::Flex::Utils
  qw( RETURN EXHAUSTION :IterAttrs :ExhaustedMethods can_meth load_role throw_failure );
use parent 'Iterator::Flex::Base';
use Scalar::Util;
use Ref::Util;

use namespace::clean;


































sub new ( $class, $code, $iterator, $pars = {} ) {

    throw_failure( parameter => q{'serialize' parameter is not a coderef} )
      unless Ref::Util::is_coderef( $code );

    throw_failure( parameter => "iterator (@{[ $iterator->_name ]}) must provide a freeze method" )
      unless can_meth( $iterator, FREEZE );

    throw_failure(
        parameter => "iterator (@{[ $iterator->_name ]}) must provide set_exhausted/is_exhausted methods" )
      unless can_meth( $iterator, SET_EXHAUSTED )
      && can_meth( $iterator, IS_EXHAUSTED );

    $class->SUPER::new( { serialize => $code, src => $iterator }, $pars );
}


sub construct ( $class, $state ) {

    throw_failure( parameter => q{'state' parameter must be a HASH reference} )
      unless Ref::Util::is_hashref( $state );

    my ( $serialize, $src ) = @{$state}{qw( serialize src )};

    throw_failure( parameter => q{'serialize' must be a CODE reference} )
      unless Ref::Util::is_coderef( $serialize );

    # wrap the source iterator so that it returns undef on exhaustion.
    $src
      = to_iterator( $src, { ( +EXHAUSTION ) => RETURN } );

    my $self;
    my %params = (
        ( +_NAME ) => 'freeze',

        ( +_SELF ) => \$self,

        ( +_DEPENDS ) => $src,
        ( +NEXT )     => sub {
            my $value = $src->();
            local $_ = $src->freeze;
            &$serialize();
            $value = $self->signal_exhaustion if $src->is_exhausted;
            return $value;
        },
    );

    Scalar::Util::weaken $src;
    $params{ +_ROLES } = [];
    for my $meth ( PREV, CURRENT, REWIND, RESET ) {
        next unless $src->may( $meth );
        my $sub = $src->can( $meth );
        Scalar::Util::weaken $sub;
        $params{$meth} = sub {
            $src->$sub();
        };

        # figure out which role was used to describe the capability
        my $Umeth = ucfirst $meth;
        my $role;
        for my $suffix ( 'Closure', 'Method' ) {
            $role
              = eval { load_role( $suffix ? $Umeth . q{::} . $suffix : $Umeth, $class->_role_namespaces ); };
            next if $@ ne q{};
            last if $src->does( $role );
            undef $role;
        }

        throw_failure( class => "unable to find role for '$meth' capability for @{[ $src->_name ]}" )
          unless defined $role;


        # need '+' as role names are fully qualified
        push $params{ +_ROLES }->@*, q{+} . $role;
    }


    return \%params;
}

__PACKAGE__->_add_roles( qw[
      State::Registry
      Next::ClosedSelf
] );

1;

#
# This file is part of Iterator-Flex
#
# This software is Copyright (c) 2018 by Smithsonian Astrophysical Observatory.
#
# This is free software, licensed under:
#
#   The GNU General Public License, Version 3, June 2007
#

__END__

=pod

=for :stopwords Diab Jerius Smithsonian Astrophysical Observatory

=head1 NAME

Iterator::Flex::Freeze - Freeze an iterator after every next

=head1 VERSION

version 0.32

=head1 METHODS

=head2 construct

  $iter = Iterator::Flex::Freeze->new( $coderef, $iterator, ?\%pars );

Construct a pass-through iterator which freezes the input iterator
after every call to C<next>.  C<$coderef> will be passed the frozen state
(generated by calling C<$iterator->freeze> via C<$_>), with which it
can do as it pleases.

<$coderef> I<is> executed when C<$iterator> is exhausted.

The optional C<%pars> hash may contain standard L<signal
parameters|Iterator::Flex::Manual::Overview/Signal Parameters>.

The returned iterator supports the following methods:

=over

=item next

=item prev

If C<$iterator> provides a C<prev> method.

=item rewind

=item freeze

=back

=head1 SUPPORT

=head2 Bugs

Please report any bugs or feature requests to bug-iterator-flex@rt.cpan.org  or through the web interface at: L<https://rt.cpan.org/Public/Dist/Display.html?Name=Iterator-Flex>

=head2 Source

Source is available at

  https://gitlab.com/djerius/iterator-flex

and may be cloned from

  https://gitlab.com/djerius/iterator-flex.git

=head1 SEE ALSO

Please see those modules/websites for more information related to this module.

=over 4

=item *

L<Iterator::Flex|Iterator::Flex>

=back

=head1 AUTHOR

Diab Jerius <djerius@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2018 by Smithsonian Astrophysical Observatory.

This is free software, licensed under:

  The GNU General Public License, Version 3, June 2007

=cut
