#! /usr/bin/env false

use v6.d;

unit grammar Grammar::TodoTxt;

rule TOP {
	[
	|| <records=.record>+
	|| \n # Skip over empty lines
	]
}

token completion-marker {
	<( 'x' )>
	\s+
}

token context {
	'@'
	<( <.word> )>
}

token date {
	<(
		\d ** 4
		'-'
		\d ** 2
		'-'
		\d ** 2
	)>
	\s+
}

token description {
	[
	|| <contexts=.context>
	|| <projects=.project>
	|| <labels=.label>
	|| <words=.word>
	]+ % \h
}

token key {
	<[ \w - ]>+
}

token label {
	<key>
	':'
	<value=.word>
}

token priority {
	'('
	<( <[A..Z]> )>
	')'
	\s+
}

regex record {
	<completion-marker>?
	<priority>?
	[
		<completion=.date>?
		<creation=.date>
	]?
	<description>
	\n?
}

token project {
	'+'
	<( <.word> )>
}

token word {
	\S+
}

=begin pod

=NAME    Grammar::TodoTxt
=AUTHOR  Patrick Spek <p.spek@tyil.work>
=VERSION 0.1.0

=head1 Synopsis

Grammar::TodoTxt.parse($input)

=head1 Description

A Raku grammar to parse todo.txt contents.

=head1 Examples

=head2 todo.txt entry with all features

The following input line contains all features described by the todo.txt
standard.

=begin input
x (A) 2016-05-20 2016-04-30 measure space for @context +chapelShelving @chapel due:2016-05-30
=end input

To parse this, C<use> the library, and call the C<.parse> method from the
imported grammar.

=begin code
use Grammar::TodoTxt;

my $todo = Grammar::TodoTxt.parse($input);
=end code

If you C<say> the resulting C<Match>, you will get the following output.

=begin output
｢x (A) 2016-05-20 2016-04-30 measure space for @context +chapelShelving @chapel due:2016-05-30｣
 records => ｢x (A) 2016-05-20 2016-04-30 measure space for @context +chapelShelving @chapel due:2016-05-30｣
  completion-marker => ｢x｣
  priority => ｢A｣
  completion => ｢2016-05-20｣
  creation => ｢2016-04-30｣
  description => ｢measure space for @context +chapelShelving @chapel due:2016-05-30｣
   words => ｢measure｣
   words => ｢space｣
   words => ｢for｣
   contexts => ｢context｣
   projects => ｢chapelShelving｣
   contexts => ｢chapel｣
   labels => ｢due:2016-05-30｣
    key => ｢due｣
    value => ｢2016-05-30｣
=end output

As you can see, all the optional parts are split up in their own match. The
rest of the task is described in the C<description>. Each word is its own
match, as each word can signify metadata of the task. For instance, you can see
C<context>, C<project> and C<label> elements in the C<description>. All words with
no meaning are just C<word> matches.

=head1 See also

=item http://todotxt.org
=item https://github.com/todotxt/todo.txt

=end pod

# vim: ft=perl6 noet
