# POD documentation - main docs before the code

=head1 NAME

GenOOx::Data::File::SAMstar::Record - Represents a record of a SAM format file generated by STAR

=head1 SYNOPSIS

    # Object representing a record of a sam file 

    # To initialize 
    my $sam_record = GenOOx::Data::File::SAMstar::Record->new(
        fields => [qname,flag, rname, pos, mapq, cigar,
                   rnext, pnext, tlen, seq, qual, tags]
    );


=head1 DESCRIPTION

    This object represents a record of a sam file generated by STAR and offers methods for accessing the different
    attributes. It implements several additional methods that transform original attributes in more manageable
    attributes. eg. from the FLAG attribute the actual strand is extracted etc.

=head1 EXAMPLES

    # Check if the record corresponds to a match
    my $mapped = $sam_record->is_mapped;
    
    # Check if the record corresponds to a non match
    my $unmapped = $sam_record->is_unmapped;
    
    # Parse the FLAG attribute and return 1 or -1 for the strand
    my $strand = $sam_record->strand;

=cut

# Let the code begin...

package GenOOx::Data::File::SAMstar::Record;
$GenOOx::Data::File::SAMstar::Record::VERSION = '0.0.2';

#######################################################################
#######################   Load External modules   #####################
#######################################################################
use Moose;
use namespace::autoclean;


#######################################################################
############################   Inheritance   ##########################
#######################################################################
extends 'GenOO::Data::File::SAM::Record';


#######################################################################
########################   Interface Methods   ########################
#######################################################################
sub number_of_mappings {
	my ($self) = @_;
	
	return $self->tag('NH:i');
}

sub is_uniquelly_mapped {
	my ($self) = @_;
	
	return 0 if $self->is_unmapped;
	return 1 if $self->mapq == 255;
	return 0;
}

sub is_primary_alignment {
	my ($self) = @_;
	
	return 0 if $self->is_unmapped;
	return 0 if $self->flag & 256;
	return 1;
}

sub is_secondary_alignment {
	my ($self) = @_;
	
	return 0 if $self->is_unmapped;
	return 1 if $self->flag & 256;
	return 0;
}


#######################################################################
############################   Finalize   #############################
#######################################################################
__PACKAGE__->meta->make_immutable;

1;
